/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.util;

import android.os.Handler;
import android.os.Looper;
import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.TaskCompletionSource;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Logger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.annotation.CheckReturnValue;

public class AsyncQueue {
    private final SynchronizedShutdownAwareExecutor executor;
    private final ArrayList<DelayedTask> delayedTasks;
    private final ArrayList<TimerId> timerIdsToSkip = new ArrayList();

    public static <TResult> Task<TResult> callTask(Executor executor, Callable<Task<TResult>> task) {
        TaskCompletionSource tcs = new TaskCompletionSource();
        executor.execute(() -> {
            try {
                ((Task)task.call()).continueWith(executor, task1 -> {
                    if (task1.isSuccessful()) {
                        tcs.setResult(task1.getResult());
                    } else {
                        tcs.setException(task1.getException());
                    }
                    return null;
                });
            }
            catch (Exception e) {
                tcs.setException(e);
            }
            catch (Throwable t) {
                IllegalStateException e = new IllegalStateException("Unhandled throwable in callTask.", t);
                tcs.setException((Exception)e);
            }
        });
        return tcs.getTask();
    }

    public AsyncQueue() {
        this.delayedTasks = new ArrayList();
        this.executor = new SynchronizedShutdownAwareExecutor();
    }

    public Executor getExecutor() {
        return this.executor;
    }

    public void verifyIsCurrentThread() {
        Thread current = Thread.currentThread();
        if (this.executor.thread != current) {
            throw Assert.fail("We are running on the wrong thread. Expected to be on the AsyncQueue thread %s/%d but was %s/%d", this.executor.thread.getName(), this.executor.thread.getId(), current.getName(), current.getId());
        }
    }

    @CheckReturnValue
    public <T> Task<T> enqueue(Callable<T> task) {
        return this.executor.executeAndReportResult(task);
    }

    @CheckReturnValue
    public Task<Void> enqueue(Runnable task) {
        return this.enqueue(() -> {
            task.run();
            return null;
        });
    }

    public Task<Void> enqueueAndInitiateShutdown(Runnable task) {
        return this.executor.executeAndInitiateShutdown(task);
    }

    public void enqueueAndForgetEvenAfterShutdown(Runnable task) {
        this.executor.executeEvenAfterShutdown(task);
    }

    public boolean isShuttingDown() {
        return this.executor.isShuttingDown();
    }

    public void enqueueAndForget(Runnable task) {
        this.enqueue(task);
    }

    public DelayedTask enqueueAfterDelay(TimerId timerId, long delayMs, Runnable task) {
        if (this.timerIdsToSkip.contains((Object)timerId)) {
            delayMs = 0L;
        }
        DelayedTask delayedTask = this.createAndScheduleDelayedTask(timerId, delayMs, task);
        this.delayedTasks.add(delayedTask);
        return delayedTask;
    }

    @VisibleForTesting
    public void skipDelaysForTimerId(TimerId timerId) {
        this.timerIdsToSkip.add(timerId);
    }

    public void panic(Throwable t) {
        this.executor.shutdownNow();
        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(() -> {
            if (t instanceof OutOfMemoryError) {
                OutOfMemoryError error = new OutOfMemoryError("Firestore (24.0.0) ran out of memory. Check your queries to make sure they are not loading an excessive amount of data.");
                error.initCause(t);
                throw error;
            }
            throw new RuntimeException("Internal error in Cloud Firestore (24.0.0).", t);
        });
    }

    @VisibleForTesting
    public void runSync(Runnable task) throws InterruptedException {
        Semaphore done = new Semaphore(0);
        Throwable[] t = new Throwable[1];
        this.enqueueAndForget(() -> {
            try {
                task.run();
            }
            catch (Throwable throwable) {
                t[0] = throwable;
            }
            done.release();
        });
        done.acquire(1);
        if (t[0] != null) {
            throw new RuntimeException("Synchronous task failed", t[0]);
        }
    }

    @VisibleForTesting
    public boolean containsDelayedTask(TimerId timerId) {
        for (DelayedTask delayedTask : this.delayedTasks) {
            if (delayedTask.timerId != timerId) continue;
            return true;
        }
        return false;
    }

    @VisibleForTesting
    public void runDelayedTasksUntil(TimerId lastTimerId) throws InterruptedException {
        this.runSync(() -> {
            Assert.hardAssert(lastTimerId == TimerId.ALL || this.containsDelayedTask(lastTimerId), "Attempted to run tasks until missing TimerId: %s", new Object[]{lastTimerId});
            Collections.sort(this.delayedTasks, (a, b) -> Long.compare(((DelayedTask)a).targetTimeMs, ((DelayedTask)b).targetTimeMs));
            for (DelayedTask delayedTask : new ArrayList<DelayedTask>(this.delayedTasks)) {
                delayedTask.skipDelay();
                if (lastTimerId == TimerId.ALL || delayedTask.timerId != lastTimerId) continue;
                break;
            }
        });
    }

    public void shutdown() {
        this.executor.setCorePoolSize(0);
    }

    private DelayedTask createAndScheduleDelayedTask(TimerId timerId, long delayMs, Runnable task) {
        long targetTimeMs = System.currentTimeMillis() + delayMs;
        DelayedTask delayedTask = new DelayedTask(timerId, targetTimeMs, task);
        delayedTask.start(delayMs);
        return delayedTask;
    }

    private void removeDelayedTask(DelayedTask task) {
        boolean found = this.delayedTasks.remove(task);
        Assert.hardAssert(found, "Delayed task not found.", new Object[0]);
    }

    private class SynchronizedShutdownAwareExecutor
    implements Executor {
        private final ScheduledThreadPoolExecutor internalExecutor;
        private boolean isShuttingDown;
        private final Thread thread;

        SynchronizedShutdownAwareExecutor() {
            DelayedStartFactory threadFactory = new DelayedStartFactory();
            this.thread = Executors.defaultThreadFactory().newThread(threadFactory);
            this.thread.setName("FirestoreWorker");
            this.thread.setDaemon(true);
            this.thread.setUncaughtExceptionHandler((crashingThread, throwable) -> AsyncQueue.this.panic(throwable));
            this.internalExecutor = new ScheduledThreadPoolExecutor(1, threadFactory){

                @Override
                protected void afterExecute(Runnable r, Throwable t) {
                    super.afterExecute(r, t);
                    if (t == null && r instanceof Future) {
                        Future future = (Future)((Object)r);
                        try {
                            if (future.isDone()) {
                                future.get();
                            }
                        }
                        catch (CancellationException cancellationException) {
                        }
                        catch (ExecutionException ee) {
                            t = ee.getCause();
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                    }
                    if (t != null) {
                        AsyncQueue.this.panic(t);
                    }
                }
            };
            this.internalExecutor.setKeepAliveTime(3L, TimeUnit.SECONDS);
            this.isShuttingDown = false;
        }

        private synchronized boolean isShuttingDown() {
            return this.isShuttingDown;
        }

        @Override
        public synchronized void execute(Runnable command) {
            if (!this.isShuttingDown) {
                this.internalExecutor.execute(command);
            }
        }

        public void executeEvenAfterShutdown(Runnable command) {
            try {
                this.internalExecutor.execute(command);
            }
            catch (RejectedExecutionException e) {
                Logger.warn(AsyncQueue.class.getSimpleName(), "Refused to enqueue task after panic", new Object[0]);
            }
        }

        private <T> Task<T> executeAndReportResult(Callable<T> task) {
            TaskCompletionSource completionSource = new TaskCompletionSource();
            try {
                this.execute(() -> {
                    try {
                        completionSource.setResult(task.call());
                    }
                    catch (Exception e) {
                        completionSource.setException(e);
                        throw new RuntimeException(e);
                    }
                });
            }
            catch (RejectedExecutionException e) {
                Logger.warn(AsyncQueue.class.getSimpleName(), "Refused to enqueue task after panic", new Object[0]);
            }
            return completionSource.getTask();
        }

        private synchronized Task<Void> executeAndInitiateShutdown(Runnable task) {
            if (this.isShuttingDown()) {
                TaskCompletionSource source = new TaskCompletionSource();
                source.setResult(null);
                return source.getTask();
            }
            Task<Void> t = this.executeAndReportResult(() -> {
                task.run();
                return null;
            });
            this.isShuttingDown = true;
            return t;
        }

        private synchronized ScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit) {
            if (!this.isShuttingDown) {
                return this.internalExecutor.schedule(command, delay, unit);
            }
            return null;
        }

        private void shutdownNow() {
            this.internalExecutor.shutdownNow();
        }

        private void setCorePoolSize(int size) {
            this.internalExecutor.setCorePoolSize(size);
        }

        private class DelayedStartFactory
        implements Runnable,
        ThreadFactory {
            private final CountDownLatch latch = new CountDownLatch(1);
            private Runnable delegate;

            private DelayedStartFactory() {
            }

            @Override
            public void run() {
                try {
                    this.latch.await();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                this.delegate.run();
            }

            @Override
            public Thread newThread(@NonNull Runnable runnable) {
                Assert.hardAssert(this.delegate == null, "Only one thread may be created in an AsyncQueue.", new Object[0]);
                this.delegate = runnable;
                this.latch.countDown();
                return SynchronizedShutdownAwareExecutor.this.thread;
            }
        }
    }

    public class DelayedTask {
        private final TimerId timerId;
        private final long targetTimeMs;
        private final Runnable task;
        private ScheduledFuture scheduledFuture;

        private DelayedTask(TimerId timerId, long targetTimeMs, Runnable task) {
            this.timerId = timerId;
            this.targetTimeMs = targetTimeMs;
            this.task = task;
        }

        private void start(long delayMs) {
            this.scheduledFuture = AsyncQueue.this.executor.schedule(this::handleDelayElapsed, delayMs, TimeUnit.MILLISECONDS);
        }

        void skipDelay() {
            this.handleDelayElapsed();
        }

        public void cancel() {
            AsyncQueue.this.verifyIsCurrentThread();
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(false);
                this.markDone();
            }
        }

        private void handleDelayElapsed() {
            AsyncQueue.this.verifyIsCurrentThread();
            if (this.scheduledFuture != null) {
                this.markDone();
                this.task.run();
            }
        }

        private void markDone() {
            Assert.hardAssert(this.scheduledFuture != null, "Caller should have verified scheduledFuture is non-null.", new Object[0]);
            this.scheduledFuture = null;
            AsyncQueue.this.removeDelayedTask(this);
        }
    }

    public static enum TimerId {
        ALL,
        LISTEN_STREAM_IDLE,
        LISTEN_STREAM_CONNECTION_BACKOFF,
        WRITE_STREAM_IDLE,
        WRITE_STREAM_CONNECTION_BACKOFF,
        HEALTH_CHECK_TIMEOUT,
        ONLINE_STATE_TIMEOUT,
        GARBAGE_COLLECTION,
        RETRY_TRANSACTION,
        CONNECTIVITY_ATTEMPT_TIMER,
        INDEX_BACKFILL;

    }
}

