/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.remote;

import androidx.annotation.Nullable;
import com.google.firebase.database.collection.ImmutableSortedSet;
import com.google.firebase.firestore.core.DocumentViewChange;
import com.google.firebase.firestore.core.Target;
import com.google.firebase.firestore.local.QueryPurpose;
import com.google.firebase.firestore.local.TargetData;
import com.google.firebase.firestore.model.DatabaseId;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.MutableDocument;
import com.google.firebase.firestore.model.SnapshotVersion;
import com.google.firebase.firestore.remote.BloomFilter;
import com.google.firebase.firestore.remote.RemoteEvent;
import com.google.firebase.firestore.remote.TargetChange;
import com.google.firebase.firestore.remote.TargetState;
import com.google.firebase.firestore.remote.TestingHooks;
import com.google.firebase.firestore.remote.WatchChange;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Logger;
import com.google.protobuf.ByteString;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class WatchChangeAggregator {
    private final TargetMetadataProvider targetMetadataProvider;
    private final Map<Integer, TargetState> targetStates = new HashMap<Integer, TargetState>();
    private Map<DocumentKey, MutableDocument> pendingDocumentUpdates = new HashMap<DocumentKey, MutableDocument>();
    private Map<DocumentKey, Set<Integer>> pendingDocumentTargetMapping = new HashMap<DocumentKey, Set<Integer>>();
    private Map<Integer, QueryPurpose> pendingTargetResets = new HashMap<Integer, QueryPurpose>();
    private static final String LOG_TAG = "WatchChangeAggregator";

    public WatchChangeAggregator(TargetMetadataProvider targetMetadataProvider) {
        this.targetMetadataProvider = targetMetadataProvider;
    }

    public void handleDocumentChange(WatchChange.DocumentChange documentChange) {
        MutableDocument document = documentChange.getNewDocument();
        DocumentKey documentKey = documentChange.getDocumentKey();
        for (int targetId : documentChange.getUpdatedTargetIds()) {
            if (document != null && document.isFoundDocument()) {
                this.addDocumentToTarget(targetId, document);
                continue;
            }
            this.removeDocumentFromTarget(targetId, documentKey, document);
        }
        for (int targetId : documentChange.getRemovedTargetIds()) {
            this.removeDocumentFromTarget(targetId, documentKey, documentChange.getNewDocument());
        }
    }

    public void handleTargetChange(WatchChange.WatchTargetChange targetChange) {
        block7: for (int targetId : this.getTargetIds(targetChange)) {
            TargetState targetState = this.ensureTargetState(targetId);
            switch (targetChange.getChangeType()) {
                case NoChange: {
                    if (!this.isActiveTarget(targetId)) continue block7;
                    targetState.updateResumeToken(targetChange.getResumeToken());
                    continue block7;
                }
                case Added: {
                    targetState.recordTargetResponse();
                    if (!targetState.isPending()) {
                        targetState.clearChanges();
                    }
                    targetState.updateResumeToken(targetChange.getResumeToken());
                    continue block7;
                }
                case Removed: {
                    targetState.recordTargetResponse();
                    if (!targetState.isPending()) {
                        this.removeTarget(targetId);
                    }
                    Assert.hardAssert(targetChange.getCause() == null, "WatchChangeAggregator does not handle errored targets", new Object[0]);
                    continue block7;
                }
                case Current: {
                    if (!this.isActiveTarget(targetId)) continue block7;
                    targetState.markCurrent();
                    targetState.updateResumeToken(targetChange.getResumeToken());
                    continue block7;
                }
                case Reset: {
                    if (!this.isActiveTarget(targetId)) continue block7;
                    this.resetTarget(targetId);
                    targetState.updateResumeToken(targetChange.getResumeToken());
                    continue block7;
                }
            }
            throw Assert.fail("Unknown target watch change state: %s", new Object[]{targetChange.getChangeType()});
        }
    }

    private Collection<Integer> getTargetIds(WatchChange.WatchTargetChange targetChange) {
        List<Integer> targetIds = targetChange.getTargetIds();
        if (!targetIds.isEmpty()) {
            return targetIds;
        }
        ArrayList<Integer> activeIds = new ArrayList<Integer>();
        for (Integer id : this.targetStates.keySet()) {
            if (!this.isActiveTarget(id)) continue;
            activeIds.add(id);
        }
        return activeIds;
    }

    public void handleExistenceFilter(WatchChange.ExistenceFilterWatchChange watchChange) {
        int targetId = watchChange.getTargetId();
        int expectedCount = watchChange.getExistenceFilter().getCount();
        TargetData targetData = this.queryDataForActiveTarget(targetId);
        if (targetData != null) {
            Target target = targetData.getTarget();
            if (target.isDocumentQuery()) {
                if (expectedCount == 0) {
                    DocumentKey key = DocumentKey.fromPath(target.getPath());
                    MutableDocument result = MutableDocument.newNoDocument(key, SnapshotVersion.NONE);
                    this.removeDocumentFromTarget(targetId, key, result);
                } else {
                    Assert.hardAssert(expectedCount == 1, "Single document existence filter with count: %d", expectedCount);
                }
            } else {
                int currentSize = this.getCurrentDocumentCountForTarget(targetId);
                if (currentSize != expectedCount) {
                    BloomFilterApplicationStatus status;
                    BloomFilter bloomFilter = this.parseBloomFilter(watchChange);
                    BloomFilterApplicationStatus bloomFilterApplicationStatus = status = bloomFilter != null ? this.applyBloomFilter(bloomFilter, watchChange, currentSize) : BloomFilterApplicationStatus.SKIPPED;
                    if (status != BloomFilterApplicationStatus.SUCCESS) {
                        this.resetTarget(targetId);
                        QueryPurpose purpose = status == BloomFilterApplicationStatus.FALSE_POSITIVE ? QueryPurpose.EXISTENCE_FILTER_MISMATCH_BLOOM : QueryPurpose.EXISTENCE_FILTER_MISMATCH;
                        this.pendingTargetResets.put(targetId, purpose);
                    }
                    TestingHooks.getInstance().notifyOnExistenceFilterMismatch(TestingHooks.ExistenceFilterMismatchInfo.from(currentSize, watchChange.getExistenceFilter(), this.targetMetadataProvider.getDatabaseId(), bloomFilter, status));
                }
            }
        }
    }

    @Nullable
    private BloomFilter parseBloomFilter(WatchChange.ExistenceFilterWatchChange watchChange) {
        BloomFilter bloomFilter;
        com.google.firestore.v1.BloomFilter unchangedNames = watchChange.getExistenceFilter().getUnchangedNames();
        if (unchangedNames == null || !unchangedNames.hasBits()) {
            return null;
        }
        ByteString bitmap = unchangedNames.getBits().getBitmap();
        try {
            bloomFilter = BloomFilter.create(bitmap, unchangedNames.getBits().getPadding(), unchangedNames.getHashCount());
        }
        catch (BloomFilter.BloomFilterCreateException e) {
            Logger.warn(LOG_TAG, "Applying bloom filter failed: (" + e.getMessage() + "); ignoring the bloom filter and falling back to full re-query.", new Object[0]);
            return null;
        }
        if (bloomFilter.getBitCount() == 0) {
            return null;
        }
        return bloomFilter;
    }

    private BloomFilterApplicationStatus applyBloomFilter(BloomFilter bloomFilter, WatchChange.ExistenceFilterWatchChange watchChange, int currentCount) {
        int removedDocumentCount;
        int expectedCount = watchChange.getExistenceFilter().getCount();
        return expectedCount == currentCount - (removedDocumentCount = this.filterRemovedDocuments(bloomFilter, watchChange.getTargetId())) ? BloomFilterApplicationStatus.SUCCESS : BloomFilterApplicationStatus.FALSE_POSITIVE;
    }

    private int filterRemovedDocuments(BloomFilter bloomFilter, int targetId) {
        ImmutableSortedSet<DocumentKey> existingKeys = this.targetMetadataProvider.getRemoteKeysForTarget(targetId);
        int removalCount = 0;
        for (DocumentKey key : existingKeys) {
            DatabaseId databaseId = this.targetMetadataProvider.getDatabaseId();
            String documentPath = "projects/" + databaseId.getProjectId() + "/databases/" + databaseId.getDatabaseId() + "/documents/" + key.getPath().canonicalString();
            if (bloomFilter.mightContain(documentPath)) continue;
            this.removeDocumentFromTarget(targetId, key, null);
            ++removalCount;
        }
        return removalCount;
    }

    public RemoteEvent createRemoteEvent(SnapshotVersion snapshotVersion) {
        HashMap<Integer, TargetChange> targetChanges = new HashMap<Integer, TargetChange>();
        for (Map.Entry<Integer, TargetState> entry : this.targetStates.entrySet()) {
            DocumentKey key;
            int n = entry.getKey();
            TargetState targetState = entry.getValue();
            TargetData targetData = this.queryDataForActiveTarget(n);
            if (targetData == null) continue;
            if (targetState.isCurrent() && targetData.getTarget().isDocumentQuery() && this.pendingDocumentUpdates.get(key = DocumentKey.fromPath(targetData.getTarget().getPath())) == null && !this.targetContainsDocument(n, key)) {
                MutableDocument result = MutableDocument.newNoDocument(key, snapshotVersion);
                this.removeDocumentFromTarget(n, key, result);
            }
            if (!targetState.hasChanges()) continue;
            targetChanges.put(n, targetState.toTargetChange());
            targetState.clearChanges();
        }
        HashSet<DocumentKey> resolvedLimboDocuments = new HashSet<DocumentKey>();
        for (Map.Entry<DocumentKey, Set<Integer>> entry : this.pendingDocumentTargetMapping.entrySet()) {
            DocumentKey key = entry.getKey();
            Set<Integer> targets = entry.getValue();
            boolean isOnlyLimboTarget = true;
            for (int targetId : targets) {
                TargetData targetData = this.queryDataForActiveTarget(targetId);
                if (targetData == null || targetData.getPurpose().equals((Object)QueryPurpose.LIMBO_RESOLUTION)) continue;
                isOnlyLimboTarget = false;
                break;
            }
            if (!isOnlyLimboTarget) continue;
            resolvedLimboDocuments.add(key);
        }
        for (MutableDocument mutableDocument : this.pendingDocumentUpdates.values()) {
            mutableDocument.setReadTime(snapshotVersion);
        }
        RemoteEvent remoteEvent = new RemoteEvent(snapshotVersion, Collections.unmodifiableMap(targetChanges), Collections.unmodifiableMap(this.pendingTargetResets), Collections.unmodifiableMap(this.pendingDocumentUpdates), Collections.unmodifiableSet(resolvedLimboDocuments));
        this.pendingDocumentUpdates = new HashMap<DocumentKey, MutableDocument>();
        this.pendingDocumentTargetMapping = new HashMap<DocumentKey, Set<Integer>>();
        this.pendingTargetResets = new HashMap<Integer, QueryPurpose>();
        return remoteEvent;
    }

    private void addDocumentToTarget(int targetId, MutableDocument document) {
        if (!this.isActiveTarget(targetId)) {
            return;
        }
        DocumentViewChange.Type changeType = this.targetContainsDocument(targetId, document.getKey()) ? DocumentViewChange.Type.MODIFIED : DocumentViewChange.Type.ADDED;
        TargetState targetState = this.ensureTargetState(targetId);
        targetState.addDocumentChange(document.getKey(), changeType);
        this.pendingDocumentUpdates.put(document.getKey(), document);
        this.ensureDocumentTargetMapping(document.getKey()).add(targetId);
    }

    private void removeDocumentFromTarget(int targetId, DocumentKey key, @Nullable MutableDocument updatedDocument) {
        if (!this.isActiveTarget(targetId)) {
            return;
        }
        TargetState targetState = this.ensureTargetState(targetId);
        if (this.targetContainsDocument(targetId, key)) {
            targetState.addDocumentChange(key, DocumentViewChange.Type.REMOVED);
        } else {
            targetState.removeDocumentChange(key);
        }
        this.ensureDocumentTargetMapping(key).add(targetId);
        if (updatedDocument != null) {
            this.pendingDocumentUpdates.put(key, updatedDocument);
        }
    }

    void removeTarget(int targetId) {
        this.targetStates.remove(targetId);
    }

    private int getCurrentDocumentCountForTarget(int targetId) {
        TargetState targetState = this.ensureTargetState(targetId);
        TargetChange targetChange = targetState.toTargetChange();
        return this.targetMetadataProvider.getRemoteKeysForTarget(targetId).size() + targetChange.getAddedDocuments().size() - targetChange.getRemovedDocuments().size();
    }

    void recordPendingTargetRequest(int targetId) {
        TargetState targetState = this.ensureTargetState(targetId);
        targetState.recordPendingTargetRequest();
    }

    private TargetState ensureTargetState(int targetId) {
        TargetState targetState = this.targetStates.get(targetId);
        if (targetState == null) {
            targetState = new TargetState();
            this.targetStates.put(targetId, targetState);
        }
        return targetState;
    }

    private Set<Integer> ensureDocumentTargetMapping(DocumentKey key) {
        Set<Integer> targetMapping = this.pendingDocumentTargetMapping.get(key);
        if (targetMapping == null) {
            targetMapping = new HashSet<Integer>();
            this.pendingDocumentTargetMapping.put(key, targetMapping);
        }
        return targetMapping;
    }

    private boolean isActiveTarget(int targetId) {
        return this.queryDataForActiveTarget(targetId) != null;
    }

    @Nullable
    private TargetData queryDataForActiveTarget(int targetId) {
        TargetState targetState = this.targetStates.get(targetId);
        return targetState != null && targetState.isPending() ? null : this.targetMetadataProvider.getTargetDataForTarget(targetId);
    }

    private void resetTarget(int targetId) {
        Assert.hardAssert(this.targetStates.get(targetId) != null && !this.targetStates.get(targetId).isPending(), "Should only reset active targets", new Object[0]);
        this.targetStates.put(targetId, new TargetState());
        ImmutableSortedSet<DocumentKey> existingKeys = this.targetMetadataProvider.getRemoteKeysForTarget(targetId);
        for (DocumentKey key : existingKeys) {
            this.removeDocumentFromTarget(targetId, key, null);
        }
    }

    private boolean targetContainsDocument(int targetId, DocumentKey key) {
        ImmutableSortedSet<DocumentKey> existingKeys = this.targetMetadataProvider.getRemoteKeysForTarget(targetId);
        return existingKeys.contains((Object)key);
    }

    static enum BloomFilterApplicationStatus {
        SUCCESS,
        SKIPPED,
        FALSE_POSITIVE;

    }

    public static interface TargetMetadataProvider {
        public ImmutableSortedSet<DocumentKey> getRemoteKeysForTarget(int var1);

        @Nullable
        public TargetData getTargetDataForTarget(int var1);

        public DatabaseId getDatabaseId();
    }
}

