/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.perf.transport;

import android.content.Context;
import androidx.annotation.NonNull;
import com.google.android.gms.common.util.VisibleForTesting;
import com.google.firebase.perf.config.ConfigResolver;
import com.google.firebase.perf.internal.ResourceType;
import com.google.firebase.perf.logging.AndroidLogger;
import com.google.firebase.perf.util.Clock;
import com.google.firebase.perf.util.Constants;
import com.google.firebase.perf.util.Timer;
import com.google.firebase.perf.util.Utils;
import com.google.firebase.perf.v1.PerfMetric;
import com.google.firebase.perf.v1.PerfSession;
import com.google.firebase.perf.v1.SessionVerbosity;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.concurrent.TimeUnit;

final class RateLimiter {
    private final float samplingBucketId;
    private boolean isLogcatEnabled = false;
    private RateLimiterImpl mTraceLimiter = null;
    private RateLimiterImpl mNetworkLimiter = null;
    private final ConfigResolver configResolver;

    public RateLimiter(@NonNull Context context, double rate, long capacity) {
        this(rate, capacity, new Clock(), RateLimiter.getSamplingBucketId(), ConfigResolver.getInstance());
        this.isLogcatEnabled = Utils.isDebugLoggingEnabled(context);
    }

    @VisibleForTesting
    static float getSamplingBucketId() {
        return new Random().nextFloat();
    }

    RateLimiter(double rate, long capacity, Clock clock, float samplingBucketId, ConfigResolver configResolver) {
        Utils.checkArgument(0.0f <= samplingBucketId && samplingBucketId < 1.0f, "Sampling bucket ID should be in range [0.0f, 1.0f).");
        this.samplingBucketId = samplingBucketId;
        this.configResolver = configResolver;
        this.mTraceLimiter = new RateLimiterImpl(rate, capacity, clock, configResolver, "Trace", this.isLogcatEnabled);
        this.mNetworkLimiter = new RateLimiterImpl(rate, capacity, clock, configResolver, "Network", this.isLogcatEnabled);
    }

    private boolean isDeviceAllowedToSendTraces() {
        float validTraceSamplingBucketIdThreshold = this.configResolver.getTraceSamplingRate();
        return this.samplingBucketId < validTraceSamplingBucketIdThreshold;
    }

    private boolean isDeviceAllowedToSendNetworkEvents() {
        float validNetworkSamplingBucketIdThreshold = this.configResolver.getNetworkRequestSamplingRate();
        return this.samplingBucketId < validNetworkSamplingBucketIdThreshold;
    }

    boolean check(PerfMetric metric) {
        if (metric.hasTraceMetric() && !this.isDeviceAllowedToSendTraces() && !this.hasVerboseSessions(metric.getTraceMetric().getPerfSessionsList())) {
            return false;
        }
        if (metric.hasNetworkRequestMetric() && !this.isDeviceAllowedToSendNetworkEvents() && !this.hasVerboseSessions(metric.getNetworkRequestMetric().getPerfSessionsList())) {
            return false;
        }
        if (!this.isRateLimited(metric)) {
            return true;
        }
        if (metric.hasNetworkRequestMetric()) {
            return this.mNetworkLimiter.check(metric);
        }
        if (metric.hasTraceMetric()) {
            return this.mTraceLimiter.check(metric);
        }
        return false;
    }

    private boolean hasVerboseSessions(List<PerfSession> perfSessions) {
        if (perfSessions.size() > 0 && perfSessions.get(0).getSessionVerbosityCount() > 0) {
            return perfSessions.get(0).getSessionVerbosity(0) == SessionVerbosity.GAUGES_AND_SYSTEM_EVENTS;
        }
        return false;
    }

    boolean isRateLimited(@NonNull PerfMetric metric) {
        if (metric.hasTraceMetric() && (metric.getTraceMetric().getName().equals(Constants.TraceNames.FOREGROUND_TRACE_NAME.toString()) || metric.getTraceMetric().getName().equals(Constants.TraceNames.BACKGROUND_TRACE_NAME.toString())) && metric.getTraceMetric().getCountersCount() > 0) {
            return false;
        }
        return !metric.hasGaugeMetric();
    }

    void changeRate(boolean isForeground) {
        this.mTraceLimiter.changeRate(isForeground);
        this.mNetworkLimiter.changeRate(isForeground);
    }

    @VisibleForTesting
    boolean getIsDeviceAllowedToSendTraces() {
        return this.isDeviceAllowedToSendTraces();
    }

    @VisibleForTesting
    boolean getIsDeviceAllowedToSendNetworkEvents() {
        return this.isDeviceAllowedToSendNetworkEvents();
    }

    static class RateLimiterImpl {
        private static final AndroidLogger logger = AndroidLogger.getInstance();
        private static final long MICROS_IN_A_SECOND = TimeUnit.SECONDS.toMicros(1L);
        private long mCapacity;
        private double mRate;
        private Timer mLastTimeTokenConsumed;
        private long mTokenCount;
        private final Clock mClock;
        private double mForegroundRate;
        private long mForegroundCapacity;
        private double mBackgroundRate;
        private long mBackgroundCapacity;
        private final boolean isLogcatEnabled;

        RateLimiterImpl(double rate, long capacity, Clock clock, ConfigResolver configResolver, @ResourceType String type, boolean isLogcatEnabled) {
            this.mClock = clock;
            this.mCapacity = capacity;
            this.mRate = rate;
            this.mTokenCount = capacity;
            this.mLastTimeTokenConsumed = this.mClock.getTime();
            this.setRateByReadingRemoteConfigValues(configResolver, type, isLogcatEnabled);
            this.isLogcatEnabled = isLogcatEnabled;
        }

        synchronized boolean check(@NonNull PerfMetric metric) {
            Timer now = this.mClock.getTime();
            long newTokens = Math.max(0L, (long)((double)this.mLastTimeTokenConsumed.getDurationMicros(now) * this.mRate / (double)MICROS_IN_A_SECOND));
            this.mTokenCount = Math.min(this.mTokenCount + newTokens, this.mCapacity);
            if (this.mTokenCount > 0L) {
                --this.mTokenCount;
                this.mLastTimeTokenConsumed = now;
                return true;
            }
            if (this.isLogcatEnabled) {
                logger.warn("Exceeded log rate limit, dropping the log.", new Object[0]);
            }
            return false;
        }

        synchronized void changeRate(boolean isForeground) {
            this.mRate = isForeground ? this.mForegroundRate : this.mBackgroundRate;
            this.mCapacity = isForeground ? this.mForegroundCapacity : this.mBackgroundCapacity;
        }

        private void setRateByReadingRemoteConfigValues(ConfigResolver configResolver, @ResourceType String type, boolean isLogcatEnabled) {
            long fLimitTime = RateLimiterImpl.getFlimitSec(configResolver, type);
            long fLimitEvents = RateLimiterImpl.getFlimitEvents(configResolver, type);
            this.mForegroundRate = (double)fLimitEvents / (double)fLimitTime;
            this.mForegroundCapacity = fLimitEvents;
            if (isLogcatEnabled) {
                logger.debug(String.format(Locale.ENGLISH, "Foreground %s logging rate:%f, burst capacity:%d", type, this.mForegroundRate, this.mForegroundCapacity), new Object[0]);
            }
            long bLimitTime = RateLimiterImpl.getBlimitSec(configResolver, type);
            long bLimitEvents = RateLimiterImpl.getBlimitEvents(configResolver, type);
            this.mBackgroundRate = (double)bLimitEvents / (double)bLimitTime;
            this.mBackgroundCapacity = bLimitEvents;
            if (isLogcatEnabled) {
                logger.debug(String.format(Locale.ENGLISH, "Background %s logging rate:%f, capacity:%d", type, this.mBackgroundRate, this.mBackgroundCapacity), new Object[0]);
            }
        }

        private static long getFlimitSec(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getRateLimitSec();
            }
            return configResolver.getRateLimitSec();
        }

        private static long getFlimitEvents(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getTraceEventCountForeground();
            }
            return configResolver.getNetworkEventCountForeground();
        }

        private static long getBlimitSec(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getRateLimitSec();
            }
            return configResolver.getRateLimitSec();
        }

        private static long getBlimitEvents(ConfigResolver configResolver, @ResourceType String type) {
            if (type == "Trace") {
                return configResolver.getTraceEventCountBackground();
            }
            return configResolver.getNetworkEventCountBackground();
        }

        @VisibleForTesting
        double getForegroundRate() {
            return this.mForegroundRate;
        }

        @VisibleForTesting
        long getForegroundCapacity() {
            return this.mForegroundCapacity;
        }

        @VisibleForTesting
        double getBackgroundRate() {
            return this.mBackgroundRate;
        }

        @VisibleForTesting
        long getBackgroundCapacity() {
            return this.mBackgroundCapacity;
        }

        @VisibleForTesting
        double getRate() {
            return this.mRate;
        }

        @VisibleForTesting
        void setRate(double newRate) {
            this.mRate = newRate;
        }
    }
}

