/*
 * Copyright (C) 2018 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.basics.date;

import static java.time.DayOfWeek.SATURDAY;
import static java.time.DayOfWeek.SUNDAY;

import java.io.Serializable;
import java.time.DayOfWeek;
import java.util.NoSuchElementException;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;
import org.joda.beans.impl.direct.MinimalMetaBean;

import com.google.common.collect.ImmutableList;
import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.basics.ReferenceDataId;

/**
 * An instance of reference data that provides a default value for holiday calendars.
 */
@BeanDefinition(style = "minimal", builderScope = "private", constructorScope = "package")
final class HolidaySafeReferenceData
    implements ReferenceData, ImmutableBean, Serializable {

  /** The Saturday/Sunday list. */
  private static final ImmutableList<DayOfWeek> WEEKEND_DAYS = ImmutableList.of(SATURDAY, SUNDAY);

  /**
   * The underlying reference data.
   */
  @PropertyDefinition(validate = "notNull")
  private final ReferenceData underlying;

  //-------------------------------------------------------------------------
  @Override
  @SuppressWarnings("unchecked")
  public <T> T queryValueOrNull(ReferenceDataId<T> id) {
    T found = underlying.queryValueOrNull(id);
    return found == null ? tryDefaultValue(id) : found;
  }

  // split out for hotspot inlining
  @SuppressWarnings("unchecked")
  private <T> T tryDefaultValue(ReferenceDataId<T> id) {
    // if the id is a holiday calendar id and composite, return null so the calendars are handled separately
    if (id instanceof HolidayCalendarId && !HolidayCalendarId.isCompositeCalendar((HolidayCalendarId) id)) {
      return (T) ImmutableHolidayCalendar.of((HolidayCalendarId) id, ImmutableList.of(), WEEKEND_DAYS);
    }
    return null;
  }

  @Override
  public boolean containsValue(ReferenceDataId<?> id) {
    return underlying.queryValueOrNull(id) != null || id instanceof HolidayCalendarId;
  }

  @Override
  public ReferenceData combinedWith(ReferenceData other) {
    ReferenceData combined = underlying.combinedWith(other);
    return new HolidaySafeReferenceData(combined);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code HolidaySafeReferenceData}.
   */
  private static final TypedMetaBean<HolidaySafeReferenceData> META_BEAN =
      MinimalMetaBean.of(
          HolidaySafeReferenceData.class,
          new String[] {
              "underlying"},
          () -> new HolidaySafeReferenceData.Builder(),
          b -> b.getUnderlying());

  /**
   * The meta-bean for {@code HolidaySafeReferenceData}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<HolidaySafeReferenceData> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Creates an instance.
   * @param underlying  the value of the property, not null
   */
  HolidaySafeReferenceData(
      ReferenceData underlying) {
    JodaBeanUtils.notNull(underlying, "underlying");
    this.underlying = underlying;
  }

  @Override
  public TypedMetaBean<HolidaySafeReferenceData> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the underlying reference data.
   * @return the value of the property, not null
   */
  public ReferenceData getUnderlying() {
    return underlying;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      HolidaySafeReferenceData other = (HolidaySafeReferenceData) obj;
      return JodaBeanUtils.equal(underlying, other.underlying);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(underlying);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("HolidaySafeReferenceData{");
    buf.append("underlying").append('=').append(JodaBeanUtils.toString(underlying));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code HolidaySafeReferenceData}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<HolidaySafeReferenceData> {

    private ReferenceData underlying;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1770633379:  // underlying
          return underlying;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1770633379:  // underlying
          this.underlying = (ReferenceData) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public HolidaySafeReferenceData build() {
      return new HolidaySafeReferenceData(
          underlying);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("HolidaySafeReferenceData.Builder{");
      buf.append("underlying").append('=').append(JodaBeanUtils.toString(underlying));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
