/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.calc.runner;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.util.List;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.google.common.collect.ImmutableList;
import com.opengamma.strata.basics.CalculationTarget;

/**
 * A set of related calculation results for a single calculation target.
 * <p>
 * This contains a list of {@link CalculationResult}, produced by a single {@link CalculationTask}.
 * Each individual result relates to a single cell in the output grid.
 */
@BeanDefinition(style = "light")
public final class CalculationResults
    implements ImmutableBean, Serializable {

  /**
   * The target of the calculation, often a trade.
   */
  @PropertyDefinition(validate = "notNull")
  private final CalculationTarget target;
  /**
   * The calculated cells.
   * Each entry contains a calculation result for a single cell.
   */
  @PropertyDefinition(validate = "notNull")
  private final ImmutableList<CalculationResult> cells;

  //-------------------------------------------------------------------------
  /**
   * Obtains a calculation result from individual calculations.
   * 
   * @param target  the calculation target, such as a trade
   * @param results  the results of the calculation
   * @return the calculation result
   */
  public static CalculationResults of(CalculationTarget target, List<CalculationResult> results) {
    return new CalculationResults(target, results);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code CalculationResults}.
   */
  private static final TypedMetaBean<CalculationResults> META_BEAN =
      LightMetaBean.of(
          CalculationResults.class,
          MethodHandles.lookup(),
          new String[] {
              "target",
              "cells"},
          null,
          ImmutableList.of());

  /**
   * The meta-bean for {@code CalculationResults}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<CalculationResults> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private CalculationResults(
      CalculationTarget target,
      List<CalculationResult> cells) {
    JodaBeanUtils.notNull(target, "target");
    JodaBeanUtils.notNull(cells, "cells");
    this.target = target;
    this.cells = ImmutableList.copyOf(cells);
  }

  @Override
  public TypedMetaBean<CalculationResults> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the target of the calculation, often a trade.
   * @return the value of the property, not null
   */
  public CalculationTarget getTarget() {
    return target;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the calculated cells.
   * Each entry contains a calculation result for a single cell.
   * @return the value of the property, not null
   */
  public ImmutableList<CalculationResult> getCells() {
    return cells;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      CalculationResults other = (CalculationResults) obj;
      return JodaBeanUtils.equal(target, other.target) &&
          JodaBeanUtils.equal(cells, other.cells);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(target);
    hash = hash * 31 + JodaBeanUtils.hashCode(cells);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("CalculationResults{");
    buf.append("target").append('=').append(JodaBeanUtils.toString(target)).append(',').append(' ');
    buf.append("cells").append('=').append(JodaBeanUtils.toString(cells));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
