/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.calc.runner;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.currency.FxRateProvider;
import com.opengamma.strata.data.FxMatrixId;
import com.opengamma.strata.data.MarketData;

/**
 * The matrix FX rate lookup.
 */
@BeanDefinition(style = "light", constructorScope = "package")
final class MatrixFxRateLookup
    implements FxRateLookup, ImmutableBean, Serializable {

  /**
   * The singleton instance.
   */
  static final MatrixFxRateLookup DEFAULT = new MatrixFxRateLookup(FxMatrixId.standard());

  /**
   * The FX matrix identifier.
   */
  @PropertyDefinition(validate = "notNull")
  private final FxMatrixId matrixId;

  //-------------------------------------------------------------------------
  @Override
  public FxRateProvider fxRateProvider(MarketData marketData) {
    return marketData.getValue(matrixId);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code MatrixFxRateLookup}.
   */
  private static final TypedMetaBean<MatrixFxRateLookup> META_BEAN =
      LightMetaBean.of(
          MatrixFxRateLookup.class,
          MethodHandles.lookup(),
          new String[] {
              "matrixId"},
          new Object[0]);

  /**
   * The meta-bean for {@code MatrixFxRateLookup}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<MatrixFxRateLookup> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Creates an instance.
   * @param matrixId  the value of the property, not null
   */
  MatrixFxRateLookup(
      FxMatrixId matrixId) {
    JodaBeanUtils.notNull(matrixId, "matrixId");
    this.matrixId = matrixId;
  }

  @Override
  public TypedMetaBean<MatrixFxRateLookup> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the FX matrix identifier.
   * @return the value of the property, not null
   */
  public FxMatrixId getMatrixId() {
    return matrixId;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      MatrixFxRateLookup other = (MatrixFxRateLookup) obj;
      return JodaBeanUtils.equal(matrixId, other.matrixId);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(matrixId);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("MatrixFxRateLookup{");
    buf.append("matrixId").append('=').append(JodaBeanUtils.toString(matrixId));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
