/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.collect.result;

import static java.util.stream.Collectors.joining;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ImmutableList;
import com.opengamma.strata.collect.tuple.Pair;

/**
 * A list of failure items.
 * <p>
 * This provides a wrapper around a list of {@link FailureItem}.
 * <p>
 * This could be used with {@link Pair} to create an object that handles partial success/failure,
 * such as {@code Pair<List<String>, FailureItems>}.
 * However, from v1.1.1 {@link ValueWithFailures} captures that use case better.
 */
@BeanDefinition(builderScope = "private", constructorScope = "package")
public final class FailureItems
    implements ImmutableBean, Serializable {

  /**
   * An empty failure list.
   */
  public static final FailureItems EMPTY = new FailureItems(ImmutableList.of());

  /**
   * The failures.
   */
  @PropertyDefinition(validate = "notNull")
  private final ImmutableList<FailureItem> failures;

  //-------------------------------------------------------------------------
  /**
   * Creates an instance from the list of failures.
   *
   * @param failures  the failures
   * @return the failure items
   */
  public static FailureItems of(FailureItem... failures) {
    return of(ImmutableList.copyOf(failures));
  }

  /**
   * Creates an instance from the list of failures.
   *
   * @param failures  the failures
   * @return the failure items
   */
  public static FailureItems of(List<FailureItem> failures) {
    if (failures.isEmpty()) {
      return EMPTY;
    }
    return new FailureItems(failures);
  }

  /**
   * Creates a builder to create the list of failures.
   *
   * @return a builder
   */
  public static FailureItemsBuilder builder() {
    return new FailureItemsBuilder();
  }

  //-------------------------------------------------------------------------
  /**
   * Combines these failure items with other failure items.
   *
   * @param other the other failure items
   * @return the combined failure items
   */
  public FailureItems combinedWith(FailureItems other) {
    return combinedWith(other.failures);
  }

  /**
   * Combines these failure items with other failure items.
   *
   * @param otherFailures the other failure items
   * @return the combined failure items
   */
  public FailureItems combinedWith(FailureItem... otherFailures) {
    return combinedWith(ImmutableList.copyOf(otherFailures));
  }

  /**
   * Combines these failure items with other failure items.
   *
   * @param otherFailures the other failure items
   * @return the combined failure items
   */
  public FailureItems combinedWith(List<FailureItem> otherFailures) {
    return builder().addAllFailures(failures).addAllFailures(otherFailures).build();
  }

  //-------------------------------------------------------------------------
  /**
   * Checks if the list of failures is empty.
   *
   * @return true if there are no failures
   */
  public boolean isEmpty() {
    return failures.isEmpty();
  }

  //-------------------------------------------------------------------------
  /**
   * Returns a string summary of the failures, as a single line excluding the stack traces.
   * 
   * @return the summary string
   */
  @Override
  public String toString() {
    return failures.stream()
        .map(f -> f.toString())
        .collect(joining(", ", "FailureItems[", "]"));
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code FailureItems}.
   * @return the meta-bean, not null
   */
  public static FailureItems.Meta meta() {
    return FailureItems.Meta.INSTANCE;
  }

  static {
    MetaBean.register(FailureItems.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Creates an instance.
   * @param failures  the value of the property, not null
   */
  FailureItems(
      List<FailureItem> failures) {
    JodaBeanUtils.notNull(failures, "failures");
    this.failures = ImmutableList.copyOf(failures);
  }

  @Override
  public FailureItems.Meta metaBean() {
    return FailureItems.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the failures.
   * @return the value of the property, not null
   */
  public ImmutableList<FailureItem> getFailures() {
    return failures;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      FailureItems other = (FailureItems) obj;
      return JodaBeanUtils.equal(failures, other.failures);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(failures);
    return hash;
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code FailureItems}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code failures} property.
     */
    @SuppressWarnings({"unchecked", "rawtypes" })
    private final MetaProperty<ImmutableList<FailureItem>> failures = DirectMetaProperty.ofImmutable(
        this, "failures", FailureItems.class, (Class) ImmutableList.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "failures");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 675938345:  // failures
          return failures;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends FailureItems> builder() {
      return new FailureItems.Builder();
    }

    @Override
    public Class<? extends FailureItems> beanType() {
      return FailureItems.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code failures} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ImmutableList<FailureItem>> failures() {
      return failures;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 675938345:  // failures
          return ((FailureItems) bean).getFailures();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code FailureItems}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<FailureItems> {

    private List<FailureItem> failures = ImmutableList.of();

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 675938345:  // failures
          return failures;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @SuppressWarnings("unchecked")
    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 675938345:  // failures
          this.failures = (List<FailureItem>) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public FailureItems build() {
      return new FailureItems(
          failures);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("FailureItems.Builder{");
      buf.append("failures").append('=').append(JodaBeanUtils.toString(failures));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
