/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.data.scenario;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.function.IntToDoubleFunction;
import java.util.stream.Stream;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.collect.array.DoubleArray;

/**
 * A scenario array holding one {@code double} value for each scenario.
 * <p>
 * This contains a list of values, one value for each scenario.
 * The calculation runner will not attempt to convert the currency of the values.
 */
@BeanDefinition(builderScope = "private")
public final class DoubleScenarioArray
    implements ScenarioArray<Double>, ImmutableBean, Serializable {

  /**
   * The calculated values, one per scenario.
   */
  @PropertyDefinition(validate = "notNull")
  private final DoubleArray values;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from the specified array of values.
   *
   * @param values  the values, one value for each scenario
   * @return an instance with the specified values
   */
  public static DoubleScenarioArray of(DoubleArray values) {
    return new DoubleScenarioArray(values);
  }

  /**
   * Obtains an instance from the specified list of values.
   *
   * @param values  the values, one value for each scenario
   * @return an instance with the specified values
   */
  public static DoubleScenarioArray of(List<Double> values) {
    return new DoubleScenarioArray(DoubleArray.copyOf(values));
  }

  /**
   * Obtains an instance using a function to create the entries.
   * <p>
   * The function is passed the scenario index and returns the value for that index.
   * 
   * @param size  the number of elements
   * @param valueFunction  the function used to obtain each value
   * @return an instance initialized using the function
   * @throws IllegalArgumentException is size is zero or less
   */
  public static DoubleScenarioArray of(int size, IntToDoubleFunction valueFunction) {
    ArgChecker.notNegativeOrZero(size, "size");
    return new DoubleScenarioArray(DoubleArray.of(size, valueFunction));
  }

  //-------------------------------------------------------------------------
  @Override
  public int getScenarioCount() {
    return values.size();
  }

  @Override
  public Double get(int index) {
    return values.get(index);
  }

  @Override
  public Stream<Double> stream() {
    return values.stream().boxed();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code DoubleScenarioArray}.
   * @return the meta-bean, not null
   */
  public static DoubleScenarioArray.Meta meta() {
    return DoubleScenarioArray.Meta.INSTANCE;
  }

  static {
    MetaBean.register(DoubleScenarioArray.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private DoubleScenarioArray(
      DoubleArray values) {
    JodaBeanUtils.notNull(values, "values");
    this.values = values;
  }

  @Override
  public DoubleScenarioArray.Meta metaBean() {
    return DoubleScenarioArray.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the calculated values, one per scenario.
   * @return the value of the property, not null
   */
  public DoubleArray getValues() {
    return values;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      DoubleScenarioArray other = (DoubleScenarioArray) obj;
      return JodaBeanUtils.equal(values, other.values);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(values);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("DoubleScenarioArray{");
    buf.append("values").append('=').append(JodaBeanUtils.toString(values));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code DoubleScenarioArray}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code values} property.
     */
    private final MetaProperty<DoubleArray> values = DirectMetaProperty.ofImmutable(
        this, "values", DoubleScenarioArray.class, DoubleArray.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "values");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -823812830:  // values
          return values;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends DoubleScenarioArray> builder() {
      return new DoubleScenarioArray.Builder();
    }

    @Override
    public Class<? extends DoubleScenarioArray> beanType() {
      return DoubleScenarioArray.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code values} property.
     * @return the meta-property, not null
     */
    public MetaProperty<DoubleArray> values() {
      return values;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -823812830:  // values
          return ((DoubleScenarioArray) bean).getValues();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code DoubleScenarioArray}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<DoubleScenarioArray> {

    private DoubleArray values;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -823812830:  // values
          return values;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -823812830:  // values
          this.values = (DoubleArray) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public DoubleScenarioArray build() {
      return new DoubleScenarioArray(
          values);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(64);
      buf.append("DoubleScenarioArray.Builder{");
      buf.append("values").append('=').append(JodaBeanUtils.toString(values));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
