/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.bond;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.data.MarketDataName;
import com.opengamma.strata.data.NamedMarketDataId;

/**
 * An identifier used to access bond future volatilities by name.
 * <p>
 * This is used when there is a need to obtain an instance of {@link BondFutureVolatilities}.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class BondFutureVolatilitiesId
    implements NamedMarketDataId<BondFutureVolatilities>, ImmutableBean, Serializable {

  /**
   * The name of the volatilities.
   */
  @PropertyDefinition(validate = "notNull")
  private final BondFutureVolatilitiesName name;

  //-------------------------------------------------------------------------
  /**
   * Obtains an identifier used to find bond future volatilities.
   *
   * @param name  the name
   * @return an identifier for the volatilities
   */
  public static BondFutureVolatilitiesId of(String name) {
    return new BondFutureVolatilitiesId(BondFutureVolatilitiesName.of(name));
  }

  /**
   * Obtains an identifier used to find bond future volatilities.
   *
   * @param name  the name
   * @return an identifier for the volatilities
   */
  public static BondFutureVolatilitiesId of(BondFutureVolatilitiesName name) {
    return new BondFutureVolatilitiesId(name);
  }

  //-------------------------------------------------------------------------
  @Override
  public Class<BondFutureVolatilities> getMarketDataType() {
    return BondFutureVolatilities.class;
  }

  @Override
  public MarketDataName<BondFutureVolatilities> getMarketDataName() {
    return name;
  }

  @Override
  public String toString() {
    return "BondFutureVolatilitiesId:" + name;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code BondFutureVolatilitiesId}.
   */
  private static final TypedMetaBean<BondFutureVolatilitiesId> META_BEAN =
      LightMetaBean.of(
          BondFutureVolatilitiesId.class,
          MethodHandles.lookup(),
          new String[] {
              "name"},
          new Object[0]);

  /**
   * The meta-bean for {@code BondFutureVolatilitiesId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<BondFutureVolatilitiesId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private BondFutureVolatilitiesId(
      BondFutureVolatilitiesName name) {
    JodaBeanUtils.notNull(name, "name");
    this.name = name;
  }

  @Override
  public TypedMetaBean<BondFutureVolatilitiesId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the name of the volatilities.
   * @return the value of the property, not null
   */
  public BondFutureVolatilitiesName getName() {
    return name;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      BondFutureVolatilitiesId other = (BondFutureVolatilitiesId) obj;
      return JodaBeanUtils.equal(name, other.name);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(name);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
