/*
 * Copyright (C) 2012 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.fxopt;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.collect.array.DoubleMatrix;

/**
 * Combines information about a volatility smile expressed in delta form and its sensitivities.
 * <p>
 * This contains a volatility smile expressed in delta form and the bucketed sensitivities
 * of the smile to the data points that were used to construct it.
 */
@BeanDefinition(style = "light")
public final class SmileAndBucketedSensitivities
    implements ImmutableBean, Serializable {

  /**
   * The smile.
   */
  @PropertyDefinition
  private final SmileDeltaParameters smile;
  /**
   * The sensitivities.
   */
  @PropertyDefinition(validate = "notNull")
  private final DoubleMatrix sensitivities;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance.
   * 
   * @param smile  the smile
   * @param sensitivities  the bucketed sensitivities
   * @return the volatility and sensitivities
   */
  public static SmileAndBucketedSensitivities of(SmileDeltaParameters smile, DoubleMatrix sensitivities) {
    return new SmileAndBucketedSensitivities(smile, sensitivities);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SmileAndBucketedSensitivities}.
   */
  private static final TypedMetaBean<SmileAndBucketedSensitivities> META_BEAN =
      LightMetaBean.of(
          SmileAndBucketedSensitivities.class,
          MethodHandles.lookup(),
          new String[] {
              "smile",
              "sensitivities"},
          new Object[0]);

  /**
   * The meta-bean for {@code SmileAndBucketedSensitivities}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<SmileAndBucketedSensitivities> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private SmileAndBucketedSensitivities(
      SmileDeltaParameters smile,
      DoubleMatrix sensitivities) {
    JodaBeanUtils.notNull(sensitivities, "sensitivities");
    this.smile = smile;
    this.sensitivities = sensitivities;
  }

  @Override
  public TypedMetaBean<SmileAndBucketedSensitivities> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the smile.
   * @return the value of the property
   */
  public SmileDeltaParameters getSmile() {
    return smile;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the sensitivities.
   * @return the value of the property, not null
   */
  public DoubleMatrix getSensitivities() {
    return sensitivities;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SmileAndBucketedSensitivities other = (SmileAndBucketedSensitivities) obj;
      return JodaBeanUtils.equal(smile, other.smile) &&
          JodaBeanUtils.equal(sensitivities, other.sensitivities);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(smile);
    hash = hash * 31 + JodaBeanUtils.hashCode(sensitivities);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("SmileAndBucketedSensitivities{");
    buf.append("smile").append('=').append(JodaBeanUtils.toString(smile)).append(',').append(' ');
    buf.append("sensitivities").append('=').append(JodaBeanUtils.toString(sensitivities));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
