/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.bond;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableDefaults;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.product.PortfolioItemInfo;
import com.opengamma.strata.product.ResolvedTrade;
import com.opengamma.strata.product.TradeInfo;
import com.opengamma.strata.product.TradedPrice;

/**
 * A trade in a futures contract based on a basket of fixed coupon bonds, resolved for pricing.
 * <p>
 * This is the resolved form of {@link BondFutureTrade} and is the primary input to the pricers.
 * Applications will typically create a {@code ResolvedBondFutureTrade} from a {@code BondFutureTrade}
 * using {@link BondFutureTrade#resolve(ReferenceData)}.
 * <p>
 * A {@code ResolvedBondFutureTrade} is bound to data that changes over time, such as holiday calendars.
 * If the data changes, such as the addition of a new holiday, the resolved form will not be updated.
 * Care must be taken when placing the resolved form in a cache or persistence layer.
 * 
 * <h4>Price</h4>
 * Strata uses <i>decimal prices</i> for bond futures in the trade model, pricers and market data.
 * This is coherent with the pricing of {@link FixedCouponBond}. The bond futures delivery is a bond
 * for an amount computed from the bond future price, a conversion factor and the accrued interest.
 */
@BeanDefinition(constructorScope = "package")
public final class ResolvedBondFutureTrade
    implements ResolvedTrade, ImmutableBean, Serializable {

  /**
   * The additional information, defaulted to an empty instance.
   * <p>
   * This allows additional information to be attached.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final PortfolioItemInfo info;
  /**
   * The future that was traded.
   * <p>
   * The product captures the contracted financial details of the trade.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final ResolvedBondFuture product;
  /**
   * The quantity that was traded.
   * <p>
   * This is the number of contracts that were traded.
   * This will be positive if buying and negative if selling.
   */
  @PropertyDefinition
  private final double quantity;
  /**
   * The price that was traded, together with the trade date, optional.
   * <p>
   * This is the price agreed when the trade occurred, in decimal form.
   * Strata uses <i>decimal prices</i> for bond futures in the trade model, pricers and market data.
   * This is coherent with the pricing of {@link FixedCouponBond}. The bond futures delivery is a bond
   * for an amount computed from the bond future price, a conversion factor and the accrued interest.
   * <p>
   * This is optional to allow the class to be used to price both trades and positions.
   * When the instance represents a trade, the traded price should be present.
   * When the instance represents a position, the traded price should be empty.
   */
  @PropertyDefinition(get = "optional")
  private final TradedPrice tradedPrice;

  //-------------------------------------------------------------------------
  @ImmutableDefaults
  private static void applyDefaults(Builder builder) {
    builder.info = TradeInfo.empty();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ResolvedBondFutureTrade}.
   * @return the meta-bean, not null
   */
  public static ResolvedBondFutureTrade.Meta meta() {
    return ResolvedBondFutureTrade.Meta.INSTANCE;
  }

  static {
    MetaBean.register(ResolvedBondFutureTrade.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static ResolvedBondFutureTrade.Builder builder() {
    return new ResolvedBondFutureTrade.Builder();
  }

  /**
   * Creates an instance.
   * @param info  the value of the property, not null
   * @param product  the value of the property, not null
   * @param quantity  the value of the property
   * @param tradedPrice  the value of the property
   */
  ResolvedBondFutureTrade(
      PortfolioItemInfo info,
      ResolvedBondFuture product,
      double quantity,
      TradedPrice tradedPrice) {
    JodaBeanUtils.notNull(info, "info");
    JodaBeanUtils.notNull(product, "product");
    this.info = info;
    this.product = product;
    this.quantity = quantity;
    this.tradedPrice = tradedPrice;
  }

  @Override
  public ResolvedBondFutureTrade.Meta metaBean() {
    return ResolvedBondFutureTrade.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the additional information, defaulted to an empty instance.
   * <p>
   * This allows additional information to be attached.
   * @return the value of the property, not null
   */
  @Override
  public PortfolioItemInfo getInfo() {
    return info;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the future that was traded.
   * <p>
   * The product captures the contracted financial details of the trade.
   * @return the value of the property, not null
   */
  @Override
  public ResolvedBondFuture getProduct() {
    return product;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the quantity that was traded.
   * <p>
   * This is the number of contracts that were traded.
   * This will be positive if buying and negative if selling.
   * @return the value of the property
   */
  public double getQuantity() {
    return quantity;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the price that was traded, together with the trade date, optional.
   * <p>
   * This is the price agreed when the trade occurred, in decimal form.
   * Strata uses <i>decimal prices</i> for bond futures in the trade model, pricers and market data.
   * This is coherent with the pricing of {@link FixedCouponBond}. The bond futures delivery is a bond
   * for an amount computed from the bond future price, a conversion factor and the accrued interest.
   * <p>
   * This is optional to allow the class to be used to price both trades and positions.
   * When the instance represents a trade, the traded price should be present.
   * When the instance represents a position, the traded price should be empty.
   * @return the optional value of the property, not null
   */
  public Optional<TradedPrice> getTradedPrice() {
    return Optional.ofNullable(tradedPrice);
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ResolvedBondFutureTrade other = (ResolvedBondFutureTrade) obj;
      return JodaBeanUtils.equal(info, other.info) &&
          JodaBeanUtils.equal(product, other.product) &&
          JodaBeanUtils.equal(quantity, other.quantity) &&
          JodaBeanUtils.equal(tradedPrice, other.tradedPrice);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(info);
    hash = hash * 31 + JodaBeanUtils.hashCode(product);
    hash = hash * 31 + JodaBeanUtils.hashCode(quantity);
    hash = hash * 31 + JodaBeanUtils.hashCode(tradedPrice);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("ResolvedBondFutureTrade{");
    buf.append("info").append('=').append(JodaBeanUtils.toString(info)).append(',').append(' ');
    buf.append("product").append('=').append(JodaBeanUtils.toString(product)).append(',').append(' ');
    buf.append("quantity").append('=').append(JodaBeanUtils.toString(quantity)).append(',').append(' ');
    buf.append("tradedPrice").append('=').append(JodaBeanUtils.toString(tradedPrice));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code ResolvedBondFutureTrade}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code info} property.
     */
    private final MetaProperty<PortfolioItemInfo> info = DirectMetaProperty.ofImmutable(
        this, "info", ResolvedBondFutureTrade.class, PortfolioItemInfo.class);
    /**
     * The meta-property for the {@code product} property.
     */
    private final MetaProperty<ResolvedBondFuture> product = DirectMetaProperty.ofImmutable(
        this, "product", ResolvedBondFutureTrade.class, ResolvedBondFuture.class);
    /**
     * The meta-property for the {@code quantity} property.
     */
    private final MetaProperty<Double> quantity = DirectMetaProperty.ofImmutable(
        this, "quantity", ResolvedBondFutureTrade.class, Double.TYPE);
    /**
     * The meta-property for the {@code tradedPrice} property.
     */
    private final MetaProperty<TradedPrice> tradedPrice = DirectMetaProperty.ofImmutable(
        this, "tradedPrice", ResolvedBondFutureTrade.class, TradedPrice.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "info",
        "product",
        "quantity",
        "tradedPrice");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          return info;
        case -309474065:  // product
          return product;
        case -1285004149:  // quantity
          return quantity;
        case -1873824343:  // tradedPrice
          return tradedPrice;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public ResolvedBondFutureTrade.Builder builder() {
      return new ResolvedBondFutureTrade.Builder();
    }

    @Override
    public Class<? extends ResolvedBondFutureTrade> beanType() {
      return ResolvedBondFutureTrade.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code info} property.
     * @return the meta-property, not null
     */
    public MetaProperty<PortfolioItemInfo> info() {
      return info;
    }

    /**
     * The meta-property for the {@code product} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ResolvedBondFuture> product() {
      return product;
    }

    /**
     * The meta-property for the {@code quantity} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> quantity() {
      return quantity;
    }

    /**
     * The meta-property for the {@code tradedPrice} property.
     * @return the meta-property, not null
     */
    public MetaProperty<TradedPrice> tradedPrice() {
      return tradedPrice;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          return ((ResolvedBondFutureTrade) bean).getInfo();
        case -309474065:  // product
          return ((ResolvedBondFutureTrade) bean).getProduct();
        case -1285004149:  // quantity
          return ((ResolvedBondFutureTrade) bean).getQuantity();
        case -1873824343:  // tradedPrice
          return ((ResolvedBondFutureTrade) bean).tradedPrice;
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code ResolvedBondFutureTrade}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<ResolvedBondFutureTrade> {

    private PortfolioItemInfo info;
    private ResolvedBondFuture product;
    private double quantity;
    private TradedPrice tradedPrice;

    /**
     * Restricted constructor.
     */
    private Builder() {
      applyDefaults(this);
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(ResolvedBondFutureTrade beanToCopy) {
      this.info = beanToCopy.getInfo();
      this.product = beanToCopy.getProduct();
      this.quantity = beanToCopy.getQuantity();
      this.tradedPrice = beanToCopy.tradedPrice;
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          return info;
        case -309474065:  // product
          return product;
        case -1285004149:  // quantity
          return quantity;
        case -1873824343:  // tradedPrice
          return tradedPrice;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 3237038:  // info
          this.info = (PortfolioItemInfo) newValue;
          break;
        case -309474065:  // product
          this.product = (ResolvedBondFuture) newValue;
          break;
        case -1285004149:  // quantity
          this.quantity = (Double) newValue;
          break;
        case -1873824343:  // tradedPrice
          this.tradedPrice = (TradedPrice) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public ResolvedBondFutureTrade build() {
      return new ResolvedBondFutureTrade(
          info,
          product,
          quantity,
          tradedPrice);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the additional information, defaulted to an empty instance.
     * <p>
     * This allows additional information to be attached.
     * @param info  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder info(PortfolioItemInfo info) {
      JodaBeanUtils.notNull(info, "info");
      this.info = info;
      return this;
    }

    /**
     * Sets the future that was traded.
     * <p>
     * The product captures the contracted financial details of the trade.
     * @param product  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder product(ResolvedBondFuture product) {
      JodaBeanUtils.notNull(product, "product");
      this.product = product;
      return this;
    }

    /**
     * Sets the quantity that was traded.
     * <p>
     * This is the number of contracts that were traded.
     * This will be positive if buying and negative if selling.
     * @param quantity  the new value
     * @return this, for chaining, not null
     */
    public Builder quantity(double quantity) {
      this.quantity = quantity;
      return this;
    }

    /**
     * Sets the price that was traded, together with the trade date, optional.
     * <p>
     * This is the price agreed when the trade occurred, in decimal form.
     * Strata uses <i>decimal prices</i> for bond futures in the trade model, pricers and market data.
     * This is coherent with the pricing of {@link FixedCouponBond}. The bond futures delivery is a bond
     * for an amount computed from the bond future price, a conversion factor and the accrued interest.
     * <p>
     * This is optional to allow the class to be used to price both trades and positions.
     * When the instance represents a trade, the traded price should be present.
     * When the instance represents a position, the traded price should be empty.
     * @param tradedPrice  the new value
     * @return this, for chaining, not null
     */
    public Builder tradedPrice(TradedPrice tradedPrice) {
      this.tradedPrice = tradedPrice;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("ResolvedBondFutureTrade.Builder{");
      buf.append("info").append('=').append(JodaBeanUtils.toString(info)).append(',').append(' ');
      buf.append("product").append('=').append(JodaBeanUtils.toString(product)).append(',').append(' ');
      buf.append("quantity").append('=').append(JodaBeanUtils.toString(quantity)).append(',').append(' ');
      buf.append("tradedPrice").append('=').append(JodaBeanUtils.toString(tradedPrice));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
