/*
 * Copyright (C) 2023 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.etd;

import java.util.NoSuchElementException;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.MinimalMetaBean;

import com.opengamma.strata.product.common.ExchangeId;

/**
 * An OG-ETD identifier that has been split into its constituent parts.
 */
@BeanDefinition(style = "minimal")
public final class SplitEtdContractSpecId implements ImmutableBean {

  /**
   * The contract spec ID that was split.
   */
  @PropertyDefinition(validate = "notNull")
  private final EtdContractSpecId specId;
  /**
   * The type of the contract - future or option.
   */
  @PropertyDefinition(validate = "notNull")
  private final EtdType type;
  /**
   * The ID of the exchange where the instruments derived from the product are traded.
   */
  @PropertyDefinition(validate = "notNull")
  private final ExchangeId exchangeId;
  /**
   * The code supplied by the exchange for use in clearing and margining, such as in SPAN.
   */
  @PropertyDefinition(validate = "notNull")
  private final EtdContractCode contractCode;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from a contract spec identifier.
   *
   * @param specId  the OG-ETD spec identifier
   * @return the split identifier
   * @throws IllegalArgumentException if the identifier cannot be split
   */
  public static SplitEtdContractSpecId from(EtdContractSpecId specId) {
    return EtdIdUtils.splitId(specId);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SplitEtdContractSpecId}.
   */
  private static final TypedMetaBean<SplitEtdContractSpecId> META_BEAN =
      MinimalMetaBean.of(
          SplitEtdContractSpecId.class,
          new String[] {
              "specId",
              "type",
              "exchangeId",
              "contractCode"},
          () -> new SplitEtdContractSpecId.Builder(),
          b -> b.getSpecId(),
          b -> b.getType(),
          b -> b.getExchangeId(),
          b -> b.getContractCode());

  /**
   * The meta-bean for {@code SplitEtdContractSpecId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<SplitEtdContractSpecId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static SplitEtdContractSpecId.Builder builder() {
    return new SplitEtdContractSpecId.Builder();
  }

  private SplitEtdContractSpecId(
      EtdContractSpecId specId,
      EtdType type,
      ExchangeId exchangeId,
      EtdContractCode contractCode) {
    JodaBeanUtils.notNull(specId, "specId");
    JodaBeanUtils.notNull(type, "type");
    JodaBeanUtils.notNull(exchangeId, "exchangeId");
    JodaBeanUtils.notNull(contractCode, "contractCode");
    this.specId = specId;
    this.type = type;
    this.exchangeId = exchangeId;
    this.contractCode = contractCode;
  }

  @Override
  public TypedMetaBean<SplitEtdContractSpecId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the contract spec ID that was split.
   * @return the value of the property, not null
   */
  public EtdContractSpecId getSpecId() {
    return specId;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the type of the contract - future or option.
   * @return the value of the property, not null
   */
  public EtdType getType() {
    return type;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the ID of the exchange where the instruments derived from the product are traded.
   * @return the value of the property, not null
   */
  public ExchangeId getExchangeId() {
    return exchangeId;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the code supplied by the exchange for use in clearing and margining, such as in SPAN.
   * @return the value of the property, not null
   */
  public EtdContractCode getContractCode() {
    return contractCode;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SplitEtdContractSpecId other = (SplitEtdContractSpecId) obj;
      return JodaBeanUtils.equal(specId, other.specId) &&
          JodaBeanUtils.equal(type, other.type) &&
          JodaBeanUtils.equal(exchangeId, other.exchangeId) &&
          JodaBeanUtils.equal(contractCode, other.contractCode);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(specId);
    hash = hash * 31 + JodaBeanUtils.hashCode(type);
    hash = hash * 31 + JodaBeanUtils.hashCode(exchangeId);
    hash = hash * 31 + JodaBeanUtils.hashCode(contractCode);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("SplitEtdContractSpecId{");
    buf.append("specId").append('=').append(JodaBeanUtils.toString(specId)).append(',').append(' ');
    buf.append("type").append('=').append(JodaBeanUtils.toString(type)).append(',').append(' ');
    buf.append("exchangeId").append('=').append(JodaBeanUtils.toString(exchangeId)).append(',').append(' ');
    buf.append("contractCode").append('=').append(JodaBeanUtils.toString(contractCode));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SplitEtdContractSpecId}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<SplitEtdContractSpecId> {

    private EtdContractSpecId specId;
    private EtdType type;
    private ExchangeId exchangeId;
    private EtdContractCode contractCode;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(SplitEtdContractSpecId beanToCopy) {
      this.specId = beanToCopy.getSpecId();
      this.type = beanToCopy.getType();
      this.exchangeId = beanToCopy.getExchangeId();
      this.contractCode = beanToCopy.getContractCode();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -896074186:  // specId
          return specId;
        case 3575610:  // type
          return type;
        case 913218206:  // exchangeId
          return exchangeId;
        case -1402840545:  // contractCode
          return contractCode;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -896074186:  // specId
          this.specId = (EtdContractSpecId) newValue;
          break;
        case 3575610:  // type
          this.type = (EtdType) newValue;
          break;
        case 913218206:  // exchangeId
          this.exchangeId = (ExchangeId) newValue;
          break;
        case -1402840545:  // contractCode
          this.contractCode = (EtdContractCode) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public SplitEtdContractSpecId build() {
      return new SplitEtdContractSpecId(
          specId,
          type,
          exchangeId,
          contractCode);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the contract spec ID that was split.
     * @param specId  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder specId(EtdContractSpecId specId) {
      JodaBeanUtils.notNull(specId, "specId");
      this.specId = specId;
      return this;
    }

    /**
     * Sets the type of the contract - future or option.
     * @param type  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder type(EtdType type) {
      JodaBeanUtils.notNull(type, "type");
      this.type = type;
      return this;
    }

    /**
     * Sets the ID of the exchange where the instruments derived from the product are traded.
     * @param exchangeId  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder exchangeId(ExchangeId exchangeId) {
      JodaBeanUtils.notNull(exchangeId, "exchangeId");
      this.exchangeId = exchangeId;
      return this;
    }

    /**
     * Sets the code supplied by the exchange for use in clearing and margining, such as in SPAN.
     * @param contractCode  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder contractCode(EtdContractCode contractCode) {
      JodaBeanUtils.notNull(contractCode, "contractCode");
      this.contractCode = contractCode;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("SplitEtdContractSpecId.Builder{");
      buf.append("specId").append('=').append(JodaBeanUtils.toString(specId)).append(',').append(' ');
      buf.append("type").append('=').append(JodaBeanUtils.toString(type)).append(',').append(' ');
      buf.append("exchangeId").append('=').append(JodaBeanUtils.toString(exchangeId)).append(',').append(' ');
      buf.append("contractCode").append('=').append(JodaBeanUtils.toString(contractCode));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
