/*
 * Copyright (C) 2018 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.rate;

import java.io.Serializable;
import java.time.LocalDate;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableValidator;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.opengamma.strata.basics.ReferenceData;
import com.opengamma.strata.basics.date.HolidayCalendar;
import com.opengamma.strata.basics.index.OvernightIndex;
import com.opengamma.strata.collect.ArgChecker;

/**
 * Defines the computation of a rate from a single overnight index that follows
 * overnight compounding using an annualized rate.
 * <p>
 * An interest rate determined directly from an overnight index that follows
 * overnight compounding using an annualized rate.
 * For example, a rate determined by compounding values from 'BRL-CDI'.
 */
@BeanDefinition
public final class OvernightCompoundedAnnualRateComputation
    implements OvernightRateComputation, ImmutableBean, Serializable {

  /**
   * The Overnight index.
   * <p>
   * The rate to be paid is based on this index.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final OvernightIndex index;
  /**
   * The resolved calendar that the index uses.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final HolidayCalendar fixingCalendar;
  /**
   * The fixing date associated with the start date of the accrual period.
   * <p>
   * This is also the first fixing date.
   * The overnight rate is observed from this date onwards.
   * <p>
   * In general, the fixing dates and accrual dates are the same for an overnight index.
   * However, in the case of a Tomorrow/Next index, the fixing period is one business day
   * before the accrual period.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final LocalDate startDate;
  /**
   * The fixing date associated with the end date of the accrual period.
   * <p>
   * The overnight rate is observed until this date.
   * <p>
   * In general, the fixing dates and accrual dates are the same for an overnight index.
   * However, in the case of a Tomorrow/Next index, the fixing period is one business day
   * before the accrual period.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final LocalDate endDate;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from an index and period dates.
   * 
   * @param index  the index
   * @param startDate  the first date of the accrual period
   * @param endDate  the last date of the accrual period
   * @param refData  the reference data to use when resolving holiday calendars
   * @return the rate computation
   */
  public static OvernightCompoundedAnnualRateComputation of(
      OvernightIndex index,
      LocalDate startDate,
      LocalDate endDate,
      ReferenceData refData) {

    return OvernightCompoundedAnnualRateComputation.builder()
        .index(index)
        .fixingCalendar(index.getFixingCalendar().resolve(refData))
        .startDate(index.calculateFixingFromEffective(startDate, refData))
        .endDate(index.calculateFixingFromEffective(endDate, refData))
        .build();
  }

  @ImmutableValidator
  private void validate() {
    ArgChecker.inOrderNotEqual(startDate, endDate, "startDate", "endDate");
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code OvernightCompoundedAnnualRateComputation}.
   * @return the meta-bean, not null
   */
  public static OvernightCompoundedAnnualRateComputation.Meta meta() {
    return OvernightCompoundedAnnualRateComputation.Meta.INSTANCE;
  }

  static {
    MetaBean.register(OvernightCompoundedAnnualRateComputation.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static OvernightCompoundedAnnualRateComputation.Builder builder() {
    return new OvernightCompoundedAnnualRateComputation.Builder();
  }

  private OvernightCompoundedAnnualRateComputation(
      OvernightIndex index,
      HolidayCalendar fixingCalendar,
      LocalDate startDate,
      LocalDate endDate) {
    JodaBeanUtils.notNull(index, "index");
    JodaBeanUtils.notNull(fixingCalendar, "fixingCalendar");
    JodaBeanUtils.notNull(startDate, "startDate");
    JodaBeanUtils.notNull(endDate, "endDate");
    this.index = index;
    this.fixingCalendar = fixingCalendar;
    this.startDate = startDate;
    this.endDate = endDate;
    validate();
  }

  @Override
  public OvernightCompoundedAnnualRateComputation.Meta metaBean() {
    return OvernightCompoundedAnnualRateComputation.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the Overnight index.
   * <p>
   * The rate to be paid is based on this index.
   * @return the value of the property, not null
   */
  @Override
  public OvernightIndex getIndex() {
    return index;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the resolved calendar that the index uses.
   * @return the value of the property, not null
   */
  @Override
  public HolidayCalendar getFixingCalendar() {
    return fixingCalendar;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the fixing date associated with the start date of the accrual period.
   * <p>
   * This is also the first fixing date.
   * The overnight rate is observed from this date onwards.
   * <p>
   * In general, the fixing dates and accrual dates are the same for an overnight index.
   * However, in the case of a Tomorrow/Next index, the fixing period is one business day
   * before the accrual period.
   * @return the value of the property, not null
   */
  @Override
  public LocalDate getStartDate() {
    return startDate;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the fixing date associated with the end date of the accrual period.
   * <p>
   * The overnight rate is observed until this date.
   * <p>
   * In general, the fixing dates and accrual dates are the same for an overnight index.
   * However, in the case of a Tomorrow/Next index, the fixing period is one business day
   * before the accrual period.
   * @return the value of the property, not null
   */
  @Override
  public LocalDate getEndDate() {
    return endDate;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      OvernightCompoundedAnnualRateComputation other = (OvernightCompoundedAnnualRateComputation) obj;
      return JodaBeanUtils.equal(index, other.index) &&
          JodaBeanUtils.equal(fixingCalendar, other.fixingCalendar) &&
          JodaBeanUtils.equal(startDate, other.startDate) &&
          JodaBeanUtils.equal(endDate, other.endDate);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(index);
    hash = hash * 31 + JodaBeanUtils.hashCode(fixingCalendar);
    hash = hash * 31 + JodaBeanUtils.hashCode(startDate);
    hash = hash * 31 + JodaBeanUtils.hashCode(endDate);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("OvernightCompoundedAnnualRateComputation{");
    buf.append("index").append('=').append(JodaBeanUtils.toString(index)).append(',').append(' ');
    buf.append("fixingCalendar").append('=').append(JodaBeanUtils.toString(fixingCalendar)).append(',').append(' ');
    buf.append("startDate").append('=').append(JodaBeanUtils.toString(startDate)).append(',').append(' ');
    buf.append("endDate").append('=').append(JodaBeanUtils.toString(endDate));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code OvernightCompoundedAnnualRateComputation}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code index} property.
     */
    private final MetaProperty<OvernightIndex> index = DirectMetaProperty.ofImmutable(
        this, "index", OvernightCompoundedAnnualRateComputation.class, OvernightIndex.class);
    /**
     * The meta-property for the {@code fixingCalendar} property.
     */
    private final MetaProperty<HolidayCalendar> fixingCalendar = DirectMetaProperty.ofImmutable(
        this, "fixingCalendar", OvernightCompoundedAnnualRateComputation.class, HolidayCalendar.class);
    /**
     * The meta-property for the {@code startDate} property.
     */
    private final MetaProperty<LocalDate> startDate = DirectMetaProperty.ofImmutable(
        this, "startDate", OvernightCompoundedAnnualRateComputation.class, LocalDate.class);
    /**
     * The meta-property for the {@code endDate} property.
     */
    private final MetaProperty<LocalDate> endDate = DirectMetaProperty.ofImmutable(
        this, "endDate", OvernightCompoundedAnnualRateComputation.class, LocalDate.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "index",
        "fixingCalendar",
        "startDate",
        "endDate");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 100346066:  // index
          return index;
        case 394230283:  // fixingCalendar
          return fixingCalendar;
        case -2129778896:  // startDate
          return startDate;
        case -1607727319:  // endDate
          return endDate;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public OvernightCompoundedAnnualRateComputation.Builder builder() {
      return new OvernightCompoundedAnnualRateComputation.Builder();
    }

    @Override
    public Class<? extends OvernightCompoundedAnnualRateComputation> beanType() {
      return OvernightCompoundedAnnualRateComputation.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code index} property.
     * @return the meta-property, not null
     */
    public MetaProperty<OvernightIndex> index() {
      return index;
    }

    /**
     * The meta-property for the {@code fixingCalendar} property.
     * @return the meta-property, not null
     */
    public MetaProperty<HolidayCalendar> fixingCalendar() {
      return fixingCalendar;
    }

    /**
     * The meta-property for the {@code startDate} property.
     * @return the meta-property, not null
     */
    public MetaProperty<LocalDate> startDate() {
      return startDate;
    }

    /**
     * The meta-property for the {@code endDate} property.
     * @return the meta-property, not null
     */
    public MetaProperty<LocalDate> endDate() {
      return endDate;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 100346066:  // index
          return ((OvernightCompoundedAnnualRateComputation) bean).getIndex();
        case 394230283:  // fixingCalendar
          return ((OvernightCompoundedAnnualRateComputation) bean).getFixingCalendar();
        case -2129778896:  // startDate
          return ((OvernightCompoundedAnnualRateComputation) bean).getStartDate();
        case -1607727319:  // endDate
          return ((OvernightCompoundedAnnualRateComputation) bean).getEndDate();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code OvernightCompoundedAnnualRateComputation}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<OvernightCompoundedAnnualRateComputation> {

    private OvernightIndex index;
    private HolidayCalendar fixingCalendar;
    private LocalDate startDate;
    private LocalDate endDate;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(OvernightCompoundedAnnualRateComputation beanToCopy) {
      this.index = beanToCopy.getIndex();
      this.fixingCalendar = beanToCopy.getFixingCalendar();
      this.startDate = beanToCopy.getStartDate();
      this.endDate = beanToCopy.getEndDate();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 100346066:  // index
          return index;
        case 394230283:  // fixingCalendar
          return fixingCalendar;
        case -2129778896:  // startDate
          return startDate;
        case -1607727319:  // endDate
          return endDate;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 100346066:  // index
          this.index = (OvernightIndex) newValue;
          break;
        case 394230283:  // fixingCalendar
          this.fixingCalendar = (HolidayCalendar) newValue;
          break;
        case -2129778896:  // startDate
          this.startDate = (LocalDate) newValue;
          break;
        case -1607727319:  // endDate
          this.endDate = (LocalDate) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public OvernightCompoundedAnnualRateComputation build() {
      return new OvernightCompoundedAnnualRateComputation(
          index,
          fixingCalendar,
          startDate,
          endDate);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets the Overnight index.
     * <p>
     * The rate to be paid is based on this index.
     * @param index  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder index(OvernightIndex index) {
      JodaBeanUtils.notNull(index, "index");
      this.index = index;
      return this;
    }

    /**
     * Sets the resolved calendar that the index uses.
     * @param fixingCalendar  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder fixingCalendar(HolidayCalendar fixingCalendar) {
      JodaBeanUtils.notNull(fixingCalendar, "fixingCalendar");
      this.fixingCalendar = fixingCalendar;
      return this;
    }

    /**
     * Sets the fixing date associated with the start date of the accrual period.
     * <p>
     * This is also the first fixing date.
     * The overnight rate is observed from this date onwards.
     * <p>
     * In general, the fixing dates and accrual dates are the same for an overnight index.
     * However, in the case of a Tomorrow/Next index, the fixing period is one business day
     * before the accrual period.
     * @param startDate  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder startDate(LocalDate startDate) {
      JodaBeanUtils.notNull(startDate, "startDate");
      this.startDate = startDate;
      return this;
    }

    /**
     * Sets the fixing date associated with the end date of the accrual period.
     * <p>
     * The overnight rate is observed until this date.
     * <p>
     * In general, the fixing dates and accrual dates are the same for an overnight index.
     * However, in the case of a Tomorrow/Next index, the fixing period is one business day
     * before the accrual period.
     * @param endDate  the new value, not null
     * @return this, for chaining, not null
     */
    public Builder endDate(LocalDate endDate) {
      JodaBeanUtils.notNull(endDate, "endDate");
      this.endDate = endDate;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("OvernightCompoundedAnnualRateComputation.Builder{");
      buf.append("index").append('=').append(JodaBeanUtils.toString(index)).append(',').append(' ');
      buf.append("fixingCalendar").append('=').append(JodaBeanUtils.toString(fixingCalendar)).append(',').append(' ');
      buf.append("startDate").append('=').append(JodaBeanUtils.toString(startDate)).append(',').append(' ');
      buf.append("endDate").append('=').append(JodaBeanUtils.toString(endDate));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
