/*
 * Copyright (C) 2021 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.product.swaption;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import org.joda.beans.Bean;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableValidator;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectFieldsBeanBuilder;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import com.opengamma.strata.collect.ArgChecker;

/**
 * The dates when a swaption can be exercised, resolved for pricing.
 * <p>
 * A swaption can have three different kinds of exercise - European, American and Bermudan.
 * A European swaption has one exercise date, an American can exercise on any date, and a Bermudan
 * can exercise on a fixed set of dates.
 * <p>
 * This is the resolved form of {@link SwaptionExercise}.
 */
@BeanDefinition(factoryName = "of")
public final class SwaptionExerciseDates
    implements ImmutableBean, Serializable {

  /**
   * An explicit list of exercise dates.
   * <p>
   * A European swaption has one date in the list.
   * A Bermudan swaption has at least two dates in the list.
   * An American swaption has at exactly two dates in the list, the earliest and latest dates.
   */
  @PropertyDefinition(validate = "notEmpty")
  private final ImmutableList<SwaptionExerciseDate> dates;
  /**
   * Whether all dates are valid dates for swaption exercise between the first and last date.
   * <p>
   * This will be true for an American swaption, and false otherwise.
   */
  @PropertyDefinition
  private final boolean allDates;

  //-------------------------------------------------------------------------
  @ImmutableValidator
  private void validate() {
    ArgChecker.isTrue(
        Ordering.natural().isStrictlyOrdered(dates),
        "Dates must be in order and without duplicates");
    if (allDates && dates.size() != 2) {
      throw new IllegalArgumentException("All dates flag can only be used when two exercise dates are defined");
    }
  }

  //-------------------------------------------------------------------------
  // creates a European instance
  static SwaptionExerciseDates ofEuropean(SwaptionExerciseDate exerciseDate) {
    return SwaptionExerciseDates.builder().dates(exerciseDate).build();
  }

  //-------------------------------------------------------------------------
  /**
   * Checks if the exercise is European.
   * 
   * @return true if European exercise on a single date
   */
  public boolean isEuropean() {
    return dates.size() == 1;
  }

  /**
   * Checks if the exercise is American.
   * 
   * @return true if American exercise on any date
   */
  public boolean isAmerican() {
    return allDates;
  }

  /**
   * Checks if the exercise is Bermudan.
   * 
   * @return true if Bermudan exercise on a specific set of dates
   */
  public boolean isBermudan() {
    return !isEuropean() && !isAmerican();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SwaptionExerciseDates}.
   * @return the meta-bean, not null
   */
  public static SwaptionExerciseDates.Meta meta() {
    return SwaptionExerciseDates.Meta.INSTANCE;
  }

  static {
    MetaBean.register(SwaptionExerciseDates.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Obtains an instance.
   * @param dates  the value of the property, not empty
   * @param allDates  the value of the property
   * @return the instance
   */
  public static SwaptionExerciseDates of(
      List<SwaptionExerciseDate> dates,
      boolean allDates) {
    return new SwaptionExerciseDates(
      dates,
      allDates);
  }

  /**
   * Returns a builder used to create an instance of the bean.
   * @return the builder, not null
   */
  public static SwaptionExerciseDates.Builder builder() {
    return new SwaptionExerciseDates.Builder();
  }

  private SwaptionExerciseDates(
      List<SwaptionExerciseDate> dates,
      boolean allDates) {
    JodaBeanUtils.notEmpty(dates, "dates");
    this.dates = ImmutableList.copyOf(dates);
    this.allDates = allDates;
    validate();
  }

  @Override
  public SwaptionExerciseDates.Meta metaBean() {
    return SwaptionExerciseDates.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets an explicit list of exercise dates.
   * <p>
   * A European swaption has one date in the list.
   * A Bermudan swaption has at least two dates in the list.
   * An American swaption has at exactly two dates in the list, the earliest and latest dates.
   * @return the value of the property, not empty
   */
  public ImmutableList<SwaptionExerciseDate> getDates() {
    return dates;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets whether all dates are valid dates for swaption exercise between the first and last date.
   * <p>
   * This will be true for an American swaption, and false otherwise.
   * @return the value of the property
   */
  public boolean isAllDates() {
    return allDates;
  }

  //-----------------------------------------------------------------------
  /**
   * Returns a builder that allows this bean to be mutated.
   * @return the mutable builder, not null
   */
  public Builder toBuilder() {
    return new Builder(this);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SwaptionExerciseDates other = (SwaptionExerciseDates) obj;
      return JodaBeanUtils.equal(dates, other.dates) &&
          (allDates == other.allDates);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(dates);
    hash = hash * 31 + JodaBeanUtils.hashCode(allDates);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("SwaptionExerciseDates{");
    buf.append("dates").append('=').append(JodaBeanUtils.toString(dates)).append(',').append(' ');
    buf.append("allDates").append('=').append(JodaBeanUtils.toString(allDates));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code SwaptionExerciseDates}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code dates} property.
     */
    @SuppressWarnings({"unchecked", "rawtypes" })
    private final MetaProperty<ImmutableList<SwaptionExerciseDate>> dates = DirectMetaProperty.ofImmutable(
        this, "dates", SwaptionExerciseDates.class, (Class) ImmutableList.class);
    /**
     * The meta-property for the {@code allDates} property.
     */
    private final MetaProperty<Boolean> allDates = DirectMetaProperty.ofImmutable(
        this, "allDates", SwaptionExerciseDates.class, Boolean.TYPE);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "dates",
        "allDates");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case 95356549:  // dates
          return dates;
        case 1772779876:  // allDates
          return allDates;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public SwaptionExerciseDates.Builder builder() {
      return new SwaptionExerciseDates.Builder();
    }

    @Override
    public Class<? extends SwaptionExerciseDates> beanType() {
      return SwaptionExerciseDates.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code dates} property.
     * @return the meta-property, not null
     */
    public MetaProperty<ImmutableList<SwaptionExerciseDate>> dates() {
      return dates;
    }

    /**
     * The meta-property for the {@code allDates} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Boolean> allDates() {
      return allDates;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case 95356549:  // dates
          return ((SwaptionExerciseDates) bean).getDates();
        case 1772779876:  // allDates
          return ((SwaptionExerciseDates) bean).isAllDates();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SwaptionExerciseDates}.
   */
  public static final class Builder extends DirectFieldsBeanBuilder<SwaptionExerciseDates> {

    private List<SwaptionExerciseDate> dates = ImmutableList.of();
    private boolean allDates;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    /**
     * Restricted copy constructor.
     * @param beanToCopy  the bean to copy from, not null
     */
    private Builder(SwaptionExerciseDates beanToCopy) {
      this.dates = beanToCopy.getDates();
      this.allDates = beanToCopy.isAllDates();
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case 95356549:  // dates
          return dates;
        case 1772779876:  // allDates
          return allDates;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @SuppressWarnings("unchecked")
    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case 95356549:  // dates
          this.dates = (List<SwaptionExerciseDate>) newValue;
          break;
        case 1772779876:  // allDates
          this.allDates = (Boolean) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public Builder set(MetaProperty<?> property, Object value) {
      super.set(property, value);
      return this;
    }

    @Override
    public SwaptionExerciseDates build() {
      return new SwaptionExerciseDates(
          dates,
          allDates);
    }

    //-----------------------------------------------------------------------
    /**
     * Sets an explicit list of exercise dates.
     * <p>
     * A European swaption has one date in the list.
     * A Bermudan swaption has at least two dates in the list.
     * An American swaption has at exactly two dates in the list, the earliest and latest dates.
     * @param dates  the new value, not empty
     * @return this, for chaining, not null
     */
    public Builder dates(List<SwaptionExerciseDate> dates) {
      JodaBeanUtils.notEmpty(dates, "dates");
      this.dates = dates;
      return this;
    }

    /**
     * Sets the {@code dates} property in the builder
     * from an array of objects.
     * @param dates  the new value, not empty
     * @return this, for chaining, not null
     */
    public Builder dates(SwaptionExerciseDate... dates) {
      return dates(ImmutableList.copyOf(dates));
    }

    /**
     * Sets whether all dates are valid dates for swaption exercise between the first and last date.
     * <p>
     * This will be true for an American swaption, and false otherwise.
     * @param allDates  the new value
     * @return this, for chaining, not null
     */
    public Builder allDates(boolean allDates) {
      this.allDates = allDates;
      return this;
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("SwaptionExerciseDates.Builder{");
      buf.append("dates").append('=').append(JodaBeanUtils.toString(dates)).append(',').append(' ');
      buf.append("allDates").append('=').append(JodaBeanUtils.toString(allDates));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
