/**
 * Copyright (c) 2016, 2024, Oracle and/or its affiliates.  All rights reserved.
 * This software is dual-licensed to you under the Universal Permissive License (UPL) 1.0 as shown at https://oss.oracle.com/licenses/upl or Apache License 2.0 as shown at http://www.apache.org/licenses/LICENSE-2.0. You may choose either license.
 */
package com.oracle.bmc.email;

import com.oracle.bmc.email.requests.*;
import com.oracle.bmc.email.responses.*;

/**
 * Collection of helper methods that can be used to provide an {@link java.lang.Iterable} interface
 * to any list operations of Email where multiple pages of data may be fetched. Two styles of
 * iteration are supported:
 *
 * <ul>
 *   <li>Iterating over the Response objects returned by the list operation. These are referred to
 *       as ResponseIterators, and the methods are suffixed with ResponseIterator. For example:
 *       <i>listUsersResponseIterator</i>
 *   <li>Iterating over the resources/records being listed. These are referred to as
 *       RecordIterators, and the methods are suffixed with RecordIterator. For example:
 *       <i>listUsersRecordIterator</i>
 * </ul>
 *
 * These iterables abstract away the need to write code to manually handle pagination via looping
 * and using the page tokens. They will automatically fetch more data from the service when
 * required.
 *
 * <p>As an example, if we were using the ListUsers operation in IdentityService, then the {@link
 * java.lang.Iterable} returned by calling a ResponseIterator method would iterate over the
 * ListUsersResponse objects returned by each ListUsers call, whereas the {@link java.lang.Iterable}
 * returned by calling a RecordIterator method would iterate over the User records and we don't have
 * to deal with ListUsersResponse objects at all. In either case, pagination will be automatically
 * handled so we can iterate until there are no more responses or no more resources/records
 * available.
 */
@jakarta.annotation.Generated(value = "OracleSDKGenerator", comments = "API Version: 20170907")
public class EmailPaginators {
    private final Email client;

    public EmailPaginators(Email client) {
        this.client = client;
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the listDkims
     * operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListDkimsResponse> listDkimsResponseIterator(final ListDkimsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListDkimsRequest.Builder, ListDkimsRequest, ListDkimsResponse>(
                new java.util.function.Supplier<ListDkimsRequest.Builder>() {
                    @Override
                    public ListDkimsRequest.Builder get() {
                        return ListDkimsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListDkimsResponse, String>() {
                    @Override
                    public String apply(ListDkimsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListDkimsRequest.Builder>,
                        ListDkimsRequest>() {
                    @Override
                    public ListDkimsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListDkimsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<ListDkimsRequest, ListDkimsResponse>() {
                    @Override
                    public ListDkimsResponse apply(ListDkimsRequest request) {
                        return client.listDkims(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.DkimSummary} objects contained in responses from the listDkims
     * operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.DkimSummary} objects contained in responses received from the
     *     service.
     */
    public Iterable<com.oracle.bmc.email.model.DkimSummary> listDkimsRecordIterator(
            final ListDkimsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListDkimsRequest.Builder,
                ListDkimsRequest,
                ListDkimsResponse,
                com.oracle.bmc.email.model.DkimSummary>(
                new java.util.function.Supplier<ListDkimsRequest.Builder>() {
                    @Override
                    public ListDkimsRequest.Builder get() {
                        return ListDkimsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListDkimsResponse, String>() {
                    @Override
                    public String apply(ListDkimsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListDkimsRequest.Builder>,
                        ListDkimsRequest>() {
                    @Override
                    public ListDkimsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListDkimsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<ListDkimsRequest, ListDkimsResponse>() {
                    @Override
                    public ListDkimsResponse apply(ListDkimsRequest request) {
                        return client.listDkims(request);
                    }
                },
                new java.util.function.Function<
                        ListDkimsResponse,
                        java.util.List<com.oracle.bmc.email.model.DkimSummary>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.DkimSummary> apply(
                            ListDkimsResponse response) {
                        return response.getDkimCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listEmailDomains operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListEmailDomainsResponse> listEmailDomainsResponseIterator(
            final ListEmailDomainsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListEmailDomainsRequest.Builder, ListEmailDomainsRequest, ListEmailDomainsResponse>(
                new java.util.function.Supplier<ListEmailDomainsRequest.Builder>() {
                    @Override
                    public ListEmailDomainsRequest.Builder get() {
                        return ListEmailDomainsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListEmailDomainsResponse, String>() {
                    @Override
                    public String apply(ListEmailDomainsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListEmailDomainsRequest.Builder>,
                        ListEmailDomainsRequest>() {
                    @Override
                    public ListEmailDomainsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListEmailDomainsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListEmailDomainsRequest, ListEmailDomainsResponse>() {
                    @Override
                    public ListEmailDomainsResponse apply(ListEmailDomainsRequest request) {
                        return client.listEmailDomains(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.EmailDomainSummary} objects contained in responses from the
     * listEmailDomains operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.EmailDomainSummary} objects contained in responses received
     *     from the service.
     */
    public Iterable<com.oracle.bmc.email.model.EmailDomainSummary> listEmailDomainsRecordIterator(
            final ListEmailDomainsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListEmailDomainsRequest.Builder,
                ListEmailDomainsRequest,
                ListEmailDomainsResponse,
                com.oracle.bmc.email.model.EmailDomainSummary>(
                new java.util.function.Supplier<ListEmailDomainsRequest.Builder>() {
                    @Override
                    public ListEmailDomainsRequest.Builder get() {
                        return ListEmailDomainsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListEmailDomainsResponse, String>() {
                    @Override
                    public String apply(ListEmailDomainsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListEmailDomainsRequest.Builder>,
                        ListEmailDomainsRequest>() {
                    @Override
                    public ListEmailDomainsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListEmailDomainsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListEmailDomainsRequest, ListEmailDomainsResponse>() {
                    @Override
                    public ListEmailDomainsResponse apply(ListEmailDomainsRequest request) {
                        return client.listEmailDomains(request);
                    }
                },
                new java.util.function.Function<
                        ListEmailDomainsResponse,
                        java.util.List<com.oracle.bmc.email.model.EmailDomainSummary>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.EmailDomainSummary> apply(
                            ListEmailDomainsResponse response) {
                        return response.getEmailDomainCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listEmailReturnPaths operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListEmailReturnPathsResponse> listEmailReturnPathsResponseIterator(
            final ListEmailReturnPathsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListEmailReturnPathsRequest.Builder,
                ListEmailReturnPathsRequest,
                ListEmailReturnPathsResponse>(
                new java.util.function.Supplier<ListEmailReturnPathsRequest.Builder>() {
                    @Override
                    public ListEmailReturnPathsRequest.Builder get() {
                        return ListEmailReturnPathsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListEmailReturnPathsResponse, String>() {
                    @Override
                    public String apply(ListEmailReturnPathsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListEmailReturnPathsRequest.Builder>,
                        ListEmailReturnPathsRequest>() {
                    @Override
                    public ListEmailReturnPathsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListEmailReturnPathsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListEmailReturnPathsRequest, ListEmailReturnPathsResponse>() {
                    @Override
                    public ListEmailReturnPathsResponse apply(ListEmailReturnPathsRequest request) {
                        return client.listEmailReturnPaths(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.EmailReturnPathSummary} objects contained in responses from the
     * listEmailReturnPaths operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.EmailReturnPathSummary} objects contained in responses
     *     received from the service.
     */
    public Iterable<com.oracle.bmc.email.model.EmailReturnPathSummary>
            listEmailReturnPathsRecordIterator(final ListEmailReturnPathsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListEmailReturnPathsRequest.Builder,
                ListEmailReturnPathsRequest,
                ListEmailReturnPathsResponse,
                com.oracle.bmc.email.model.EmailReturnPathSummary>(
                new java.util.function.Supplier<ListEmailReturnPathsRequest.Builder>() {
                    @Override
                    public ListEmailReturnPathsRequest.Builder get() {
                        return ListEmailReturnPathsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListEmailReturnPathsResponse, String>() {
                    @Override
                    public String apply(ListEmailReturnPathsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListEmailReturnPathsRequest.Builder>,
                        ListEmailReturnPathsRequest>() {
                    @Override
                    public ListEmailReturnPathsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListEmailReturnPathsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListEmailReturnPathsRequest, ListEmailReturnPathsResponse>() {
                    @Override
                    public ListEmailReturnPathsResponse apply(ListEmailReturnPathsRequest request) {
                        return client.listEmailReturnPaths(request);
                    }
                },
                new java.util.function.Function<
                        ListEmailReturnPathsResponse,
                        java.util.List<com.oracle.bmc.email.model.EmailReturnPathSummary>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.EmailReturnPathSummary> apply(
                            ListEmailReturnPathsResponse response) {
                        return response.getEmailReturnPathCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the listSenders
     * operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListSendersResponse> listSendersResponseIterator(
            final ListSendersRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListSendersRequest.Builder, ListSendersRequest, ListSendersResponse>(
                new java.util.function.Supplier<ListSendersRequest.Builder>() {
                    @Override
                    public ListSendersRequest.Builder get() {
                        return ListSendersRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListSendersResponse, String>() {
                    @Override
                    public String apply(ListSendersResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListSendersRequest.Builder>,
                        ListSendersRequest>() {
                    @Override
                    public ListSendersRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListSendersRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<ListSendersRequest, ListSendersResponse>() {
                    @Override
                    public ListSendersResponse apply(ListSendersRequest request) {
                        return client.listSenders(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.SenderSummary} objects contained in responses from the listSenders
     * operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.SenderSummary} objects contained in responses received from
     *     the service.
     */
    public Iterable<com.oracle.bmc.email.model.SenderSummary> listSendersRecordIterator(
            final ListSendersRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListSendersRequest.Builder,
                ListSendersRequest,
                ListSendersResponse,
                com.oracle.bmc.email.model.SenderSummary>(
                new java.util.function.Supplier<ListSendersRequest.Builder>() {
                    @Override
                    public ListSendersRequest.Builder get() {
                        return ListSendersRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListSendersResponse, String>() {
                    @Override
                    public String apply(ListSendersResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListSendersRequest.Builder>,
                        ListSendersRequest>() {
                    @Override
                    public ListSendersRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListSendersRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<ListSendersRequest, ListSendersResponse>() {
                    @Override
                    public ListSendersResponse apply(ListSendersRequest request) {
                        return client.listSenders(request);
                    }
                },
                new java.util.function.Function<
                        ListSendersResponse,
                        java.util.List<com.oracle.bmc.email.model.SenderSummary>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.SenderSummary> apply(
                            ListSendersResponse response) {
                        return response.getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listSuppressions operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListSuppressionsResponse> listSuppressionsResponseIterator(
            final ListSuppressionsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListSuppressionsRequest.Builder, ListSuppressionsRequest, ListSuppressionsResponse>(
                new java.util.function.Supplier<ListSuppressionsRequest.Builder>() {
                    @Override
                    public ListSuppressionsRequest.Builder get() {
                        return ListSuppressionsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListSuppressionsResponse, String>() {
                    @Override
                    public String apply(ListSuppressionsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListSuppressionsRequest.Builder>,
                        ListSuppressionsRequest>() {
                    @Override
                    public ListSuppressionsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListSuppressionsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListSuppressionsRequest, ListSuppressionsResponse>() {
                    @Override
                    public ListSuppressionsResponse apply(ListSuppressionsRequest request) {
                        return client.listSuppressions(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.SuppressionSummary} objects contained in responses from the
     * listSuppressions operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.SuppressionSummary} objects contained in responses received
     *     from the service.
     */
    public Iterable<com.oracle.bmc.email.model.SuppressionSummary> listSuppressionsRecordIterator(
            final ListSuppressionsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListSuppressionsRequest.Builder,
                ListSuppressionsRequest,
                ListSuppressionsResponse,
                com.oracle.bmc.email.model.SuppressionSummary>(
                new java.util.function.Supplier<ListSuppressionsRequest.Builder>() {
                    @Override
                    public ListSuppressionsRequest.Builder get() {
                        return ListSuppressionsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListSuppressionsResponse, String>() {
                    @Override
                    public String apply(ListSuppressionsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListSuppressionsRequest.Builder>,
                        ListSuppressionsRequest>() {
                    @Override
                    public ListSuppressionsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListSuppressionsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListSuppressionsRequest, ListSuppressionsResponse>() {
                    @Override
                    public ListSuppressionsResponse apply(ListSuppressionsRequest request) {
                        return client.listSuppressions(request);
                    }
                },
                new java.util.function.Function<
                        ListSuppressionsResponse,
                        java.util.List<com.oracle.bmc.email.model.SuppressionSummary>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.SuppressionSummary> apply(
                            ListSuppressionsResponse response) {
                        return response.getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listWorkRequestErrors operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListWorkRequestErrorsResponse> listWorkRequestErrorsResponseIterator(
            final ListWorkRequestErrorsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListWorkRequestErrorsRequest.Builder,
                ListWorkRequestErrorsRequest,
                ListWorkRequestErrorsResponse>(
                new java.util.function.Supplier<ListWorkRequestErrorsRequest.Builder>() {
                    @Override
                    public ListWorkRequestErrorsRequest.Builder get() {
                        return ListWorkRequestErrorsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestErrorsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestErrorsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestErrorsRequest.Builder>,
                        ListWorkRequestErrorsRequest>() {
                    @Override
                    public ListWorkRequestErrorsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestErrorsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestErrorsRequest, ListWorkRequestErrorsResponse>() {
                    @Override
                    public ListWorkRequestErrorsResponse apply(
                            ListWorkRequestErrorsRequest request) {
                        return client.listWorkRequestErrors(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.WorkRequestError} objects contained in responses from the
     * listWorkRequestErrors operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.WorkRequestError} objects contained in responses received from
     *     the service.
     */
    public Iterable<com.oracle.bmc.email.model.WorkRequestError>
            listWorkRequestErrorsRecordIterator(final ListWorkRequestErrorsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListWorkRequestErrorsRequest.Builder,
                ListWorkRequestErrorsRequest,
                ListWorkRequestErrorsResponse,
                com.oracle.bmc.email.model.WorkRequestError>(
                new java.util.function.Supplier<ListWorkRequestErrorsRequest.Builder>() {
                    @Override
                    public ListWorkRequestErrorsRequest.Builder get() {
                        return ListWorkRequestErrorsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestErrorsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestErrorsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestErrorsRequest.Builder>,
                        ListWorkRequestErrorsRequest>() {
                    @Override
                    public ListWorkRequestErrorsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestErrorsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestErrorsRequest, ListWorkRequestErrorsResponse>() {
                    @Override
                    public ListWorkRequestErrorsResponse apply(
                            ListWorkRequestErrorsRequest request) {
                        return client.listWorkRequestErrors(request);
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestErrorsResponse,
                        java.util.List<com.oracle.bmc.email.model.WorkRequestError>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.WorkRequestError> apply(
                            ListWorkRequestErrorsResponse response) {
                        return response.getWorkRequestErrorCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listWorkRequestLogs operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListWorkRequestLogsResponse> listWorkRequestLogsResponseIterator(
            final ListWorkRequestLogsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListWorkRequestLogsRequest.Builder,
                ListWorkRequestLogsRequest,
                ListWorkRequestLogsResponse>(
                new java.util.function.Supplier<ListWorkRequestLogsRequest.Builder>() {
                    @Override
                    public ListWorkRequestLogsRequest.Builder get() {
                        return ListWorkRequestLogsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestLogsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestLogsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestLogsRequest.Builder>,
                        ListWorkRequestLogsRequest>() {
                    @Override
                    public ListWorkRequestLogsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestLogsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestLogsRequest, ListWorkRequestLogsResponse>() {
                    @Override
                    public ListWorkRequestLogsResponse apply(ListWorkRequestLogsRequest request) {
                        return client.listWorkRequestLogs(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.WorkRequestLogEntry} objects contained in responses from the
     * listWorkRequestLogs operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.WorkRequestLogEntry} objects contained in responses received
     *     from the service.
     */
    public Iterable<com.oracle.bmc.email.model.WorkRequestLogEntry>
            listWorkRequestLogsRecordIterator(final ListWorkRequestLogsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListWorkRequestLogsRequest.Builder,
                ListWorkRequestLogsRequest,
                ListWorkRequestLogsResponse,
                com.oracle.bmc.email.model.WorkRequestLogEntry>(
                new java.util.function.Supplier<ListWorkRequestLogsRequest.Builder>() {
                    @Override
                    public ListWorkRequestLogsRequest.Builder get() {
                        return ListWorkRequestLogsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestLogsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestLogsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestLogsRequest.Builder>,
                        ListWorkRequestLogsRequest>() {
                    @Override
                    public ListWorkRequestLogsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestLogsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestLogsRequest, ListWorkRequestLogsResponse>() {
                    @Override
                    public ListWorkRequestLogsResponse apply(ListWorkRequestLogsRequest request) {
                        return client.listWorkRequestLogs(request);
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestLogsResponse,
                        java.util.List<com.oracle.bmc.email.model.WorkRequestLogEntry>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.WorkRequestLogEntry> apply(
                            ListWorkRequestLogsResponse response) {
                        return response.getWorkRequestLogEntryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listWorkRequests operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListWorkRequestsResponse> listWorkRequestsResponseIterator(
            final ListWorkRequestsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListWorkRequestsRequest.Builder, ListWorkRequestsRequest, ListWorkRequestsResponse>(
                new java.util.function.Supplier<ListWorkRequestsRequest.Builder>() {
                    @Override
                    public ListWorkRequestsRequest.Builder get() {
                        return ListWorkRequestsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestsRequest.Builder>,
                        ListWorkRequestsRequest>() {
                    @Override
                    public ListWorkRequestsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestsRequest, ListWorkRequestsResponse>() {
                    @Override
                    public ListWorkRequestsResponse apply(ListWorkRequestsRequest request) {
                        return client.listWorkRequests(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.email.model.WorkRequestSummary} objects contained in responses from the
     * listWorkRequests operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.email.model.WorkRequestSummary} objects contained in responses received
     *     from the service.
     */
    public Iterable<com.oracle.bmc.email.model.WorkRequestSummary> listWorkRequestsRecordIterator(
            final ListWorkRequestsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListWorkRequestsRequest.Builder,
                ListWorkRequestsRequest,
                ListWorkRequestsResponse,
                com.oracle.bmc.email.model.WorkRequestSummary>(
                new java.util.function.Supplier<ListWorkRequestsRequest.Builder>() {
                    @Override
                    public ListWorkRequestsRequest.Builder get() {
                        return ListWorkRequestsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestsRequest.Builder>,
                        ListWorkRequestsRequest>() {
                    @Override
                    public ListWorkRequestsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestsRequest, ListWorkRequestsResponse>() {
                    @Override
                    public ListWorkRequestsResponse apply(ListWorkRequestsRequest request) {
                        return client.listWorkRequests(request);
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestsResponse,
                        java.util.List<com.oracle.bmc.email.model.WorkRequestSummary>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.email.model.WorkRequestSummary> apply(
                            ListWorkRequestsResponse response) {
                        return response.getWorkRequestSummaryCollection().getItems();
                    }
                });
    }
}
