/**
 * Copyright (c) 2016, 2024, Oracle and/or its affiliates.  All rights reserved.
 * This software is dual-licensed to you under the Universal Permissive License (UPL) 1.0 as shown at https://oss.oracle.com/licenses/upl or Apache License 2.0 as shown at http://www.apache.org/licenses/LICENSE-2.0. You may choose either license.
 */
package com.oracle.bmc.vulnerabilityscanning;

import com.oracle.bmc.vulnerabilityscanning.requests.*;
import com.oracle.bmc.vulnerabilityscanning.responses.*;

/**
 * Collection of helper methods that can be used to provide an {@link java.lang.Iterable} interface
 * to any list operations of VulnerabilityScanning where multiple pages of data may be fetched. Two
 * styles of iteration are supported:
 *
 * <ul>
 *   <li>Iterating over the Response objects returned by the list operation. These are referred to
 *       as ResponseIterators, and the methods are suffixed with ResponseIterator. For example:
 *       <i>listUsersResponseIterator</i>
 *   <li>Iterating over the resources/records being listed. These are referred to as
 *       RecordIterators, and the methods are suffixed with RecordIterator. For example:
 *       <i>listUsersRecordIterator</i>
 * </ul>
 *
 * These iterables abstract away the need to write code to manually handle pagination via looping
 * and using the page tokens. They will automatically fetch more data from the service when
 * required.
 *
 * <p>As an example, if we were using the ListUsers operation in IdentityService, then the {@link
 * java.lang.Iterable} returned by calling a ResponseIterator method would iterate over the
 * ListUsersResponse objects returned by each ListUsers call, whereas the {@link java.lang.Iterable}
 * returned by calling a RecordIterator method would iterate over the User records and we don't have
 * to deal with ListUsersResponse objects at all. In either case, pagination will be automatically
 * handled so we can iterate until there are no more responses or no more resources/records
 * available.
 */
@jakarta.annotation.Generated(value = "OracleSDKGenerator", comments = "API Version: 20210215")
public class VulnerabilityScanningPaginators {
    private final VulnerabilityScanning client;

    public VulnerabilityScanningPaginators(VulnerabilityScanning client) {
        this.client = client;
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listContainerScanRecipes operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListContainerScanRecipesResponse> listContainerScanRecipesResponseIterator(
            final ListContainerScanRecipesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListContainerScanRecipesRequest.Builder,
                ListContainerScanRecipesRequest,
                ListContainerScanRecipesResponse>(
                new java.util.function.Supplier<ListContainerScanRecipesRequest.Builder>() {
                    @Override
                    public ListContainerScanRecipesRequest.Builder get() {
                        return ListContainerScanRecipesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListContainerScanRecipesResponse, String>() {
                    @Override
                    public String apply(ListContainerScanRecipesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListContainerScanRecipesRequest.Builder>,
                        ListContainerScanRecipesRequest>() {
                    @Override
                    public ListContainerScanRecipesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListContainerScanRecipesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListContainerScanRecipesRequest, ListContainerScanRecipesResponse>() {
                    @Override
                    public ListContainerScanRecipesResponse apply(
                            ListContainerScanRecipesRequest request) {
                        return client.listContainerScanRecipes(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.ContainerScanRecipeSummary} objects contained in
     * responses from the listContainerScanRecipes operation. This iterable will fetch more data
     * from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.ContainerScanRecipeSummary} objects contained
     *     in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.ContainerScanRecipeSummary>
            listContainerScanRecipesRecordIterator(final ListContainerScanRecipesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListContainerScanRecipesRequest.Builder,
                ListContainerScanRecipesRequest,
                ListContainerScanRecipesResponse,
                com.oracle.bmc.vulnerabilityscanning.model.ContainerScanRecipeSummary>(
                new java.util.function.Supplier<ListContainerScanRecipesRequest.Builder>() {
                    @Override
                    public ListContainerScanRecipesRequest.Builder get() {
                        return ListContainerScanRecipesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListContainerScanRecipesResponse, String>() {
                    @Override
                    public String apply(ListContainerScanRecipesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListContainerScanRecipesRequest.Builder>,
                        ListContainerScanRecipesRequest>() {
                    @Override
                    public ListContainerScanRecipesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListContainerScanRecipesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListContainerScanRecipesRequest, ListContainerScanRecipesResponse>() {
                    @Override
                    public ListContainerScanRecipesResponse apply(
                            ListContainerScanRecipesRequest request) {
                        return client.listContainerScanRecipes(request);
                    }
                },
                new java.util.function.Function<
                        ListContainerScanRecipesResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .ContainerScanRecipeSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .ContainerScanRecipeSummary>
                            apply(ListContainerScanRecipesResponse response) {
                        return response.getContainerScanRecipeSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listContainerScanResults operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListContainerScanResultsResponse> listContainerScanResultsResponseIterator(
            final ListContainerScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListContainerScanResultsRequest.Builder,
                ListContainerScanResultsRequest,
                ListContainerScanResultsResponse>(
                new java.util.function.Supplier<ListContainerScanResultsRequest.Builder>() {
                    @Override
                    public ListContainerScanResultsRequest.Builder get() {
                        return ListContainerScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListContainerScanResultsResponse, String>() {
                    @Override
                    public String apply(ListContainerScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListContainerScanResultsRequest.Builder>,
                        ListContainerScanResultsRequest>() {
                    @Override
                    public ListContainerScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListContainerScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListContainerScanResultsRequest, ListContainerScanResultsResponse>() {
                    @Override
                    public ListContainerScanResultsResponse apply(
                            ListContainerScanResultsRequest request) {
                        return client.listContainerScanResults(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.ContainerScanResultSummary} objects contained in
     * responses from the listContainerScanResults operation. This iterable will fetch more data
     * from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.ContainerScanResultSummary} objects contained
     *     in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.ContainerScanResultSummary>
            listContainerScanResultsRecordIterator(final ListContainerScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListContainerScanResultsRequest.Builder,
                ListContainerScanResultsRequest,
                ListContainerScanResultsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.ContainerScanResultSummary>(
                new java.util.function.Supplier<ListContainerScanResultsRequest.Builder>() {
                    @Override
                    public ListContainerScanResultsRequest.Builder get() {
                        return ListContainerScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListContainerScanResultsResponse, String>() {
                    @Override
                    public String apply(ListContainerScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListContainerScanResultsRequest.Builder>,
                        ListContainerScanResultsRequest>() {
                    @Override
                    public ListContainerScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListContainerScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListContainerScanResultsRequest, ListContainerScanResultsResponse>() {
                    @Override
                    public ListContainerScanResultsResponse apply(
                            ListContainerScanResultsRequest request) {
                        return client.listContainerScanResults(request);
                    }
                },
                new java.util.function.Function<
                        ListContainerScanResultsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .ContainerScanResultSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .ContainerScanResultSummary>
                            apply(ListContainerScanResultsResponse response) {
                        return response.getContainerScanResultSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listContainerScanTargets operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListContainerScanTargetsResponse> listContainerScanTargetsResponseIterator(
            final ListContainerScanTargetsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListContainerScanTargetsRequest.Builder,
                ListContainerScanTargetsRequest,
                ListContainerScanTargetsResponse>(
                new java.util.function.Supplier<ListContainerScanTargetsRequest.Builder>() {
                    @Override
                    public ListContainerScanTargetsRequest.Builder get() {
                        return ListContainerScanTargetsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListContainerScanTargetsResponse, String>() {
                    @Override
                    public String apply(ListContainerScanTargetsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListContainerScanTargetsRequest.Builder>,
                        ListContainerScanTargetsRequest>() {
                    @Override
                    public ListContainerScanTargetsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListContainerScanTargetsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListContainerScanTargetsRequest, ListContainerScanTargetsResponse>() {
                    @Override
                    public ListContainerScanTargetsResponse apply(
                            ListContainerScanTargetsRequest request) {
                        return client.listContainerScanTargets(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.ContainerScanTargetSummary} objects contained in
     * responses from the listContainerScanTargets operation. This iterable will fetch more data
     * from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.ContainerScanTargetSummary} objects contained
     *     in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.ContainerScanTargetSummary>
            listContainerScanTargetsRecordIterator(final ListContainerScanTargetsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListContainerScanTargetsRequest.Builder,
                ListContainerScanTargetsRequest,
                ListContainerScanTargetsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.ContainerScanTargetSummary>(
                new java.util.function.Supplier<ListContainerScanTargetsRequest.Builder>() {
                    @Override
                    public ListContainerScanTargetsRequest.Builder get() {
                        return ListContainerScanTargetsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListContainerScanTargetsResponse, String>() {
                    @Override
                    public String apply(ListContainerScanTargetsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListContainerScanTargetsRequest.Builder>,
                        ListContainerScanTargetsRequest>() {
                    @Override
                    public ListContainerScanTargetsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListContainerScanTargetsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListContainerScanTargetsRequest, ListContainerScanTargetsResponse>() {
                    @Override
                    public ListContainerScanTargetsResponse apply(
                            ListContainerScanTargetsRequest request) {
                        return client.listContainerScanTargets(request);
                    }
                },
                new java.util.function.Function<
                        ListContainerScanTargetsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .ContainerScanTargetSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .ContainerScanTargetSummary>
                            apply(ListContainerScanTargetsResponse response) {
                        return response.getContainerScanTargetSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostAgentScanResults operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostAgentScanResultsResponse> listHostAgentScanResultsResponseIterator(
            final ListHostAgentScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostAgentScanResultsRequest.Builder,
                ListHostAgentScanResultsRequest,
                ListHostAgentScanResultsResponse>(
                new java.util.function.Supplier<ListHostAgentScanResultsRequest.Builder>() {
                    @Override
                    public ListHostAgentScanResultsRequest.Builder get() {
                        return ListHostAgentScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostAgentScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostAgentScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostAgentScanResultsRequest.Builder>,
                        ListHostAgentScanResultsRequest>() {
                    @Override
                    public ListHostAgentScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostAgentScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostAgentScanResultsRequest, ListHostAgentScanResultsResponse>() {
                    @Override
                    public ListHostAgentScanResultsResponse apply(
                            ListHostAgentScanResultsRequest request) {
                        return client.listHostAgentScanResults(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostAgentScanResultSummary} objects contained in
     * responses from the listHostAgentScanResults operation. This iterable will fetch more data
     * from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostAgentScanResultSummary} objects contained
     *     in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostAgentScanResultSummary>
            listHostAgentScanResultsRecordIterator(final ListHostAgentScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostAgentScanResultsRequest.Builder,
                ListHostAgentScanResultsRequest,
                ListHostAgentScanResultsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostAgentScanResultSummary>(
                new java.util.function.Supplier<ListHostAgentScanResultsRequest.Builder>() {
                    @Override
                    public ListHostAgentScanResultsRequest.Builder get() {
                        return ListHostAgentScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostAgentScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostAgentScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostAgentScanResultsRequest.Builder>,
                        ListHostAgentScanResultsRequest>() {
                    @Override
                    public ListHostAgentScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostAgentScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostAgentScanResultsRequest, ListHostAgentScanResultsResponse>() {
                    @Override
                    public ListHostAgentScanResultsResponse apply(
                            ListHostAgentScanResultsRequest request) {
                        return client.listHostAgentScanResults(request);
                    }
                },
                new java.util.function.Function<
                        ListHostAgentScanResultsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostAgentScanResultSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostAgentScanResultSummary>
                            apply(ListHostAgentScanResultsResponse response) {
                        return response.getHostAgentScanResultSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostCisBenchmarkScanResults operation. This iterable will fetch more data from the server
     * as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostCisBenchmarkScanResultsResponse>
            listHostCisBenchmarkScanResultsResponseIterator(
                    final ListHostCisBenchmarkScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostCisBenchmarkScanResultsRequest.Builder,
                ListHostCisBenchmarkScanResultsRequest,
                ListHostCisBenchmarkScanResultsResponse>(
                new java.util.function.Supplier<ListHostCisBenchmarkScanResultsRequest.Builder>() {
                    @Override
                    public ListHostCisBenchmarkScanResultsRequest.Builder get() {
                        return ListHostCisBenchmarkScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostCisBenchmarkScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostCisBenchmarkScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostCisBenchmarkScanResultsRequest.Builder>,
                        ListHostCisBenchmarkScanResultsRequest>() {
                    @Override
                    public ListHostCisBenchmarkScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostCisBenchmarkScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostCisBenchmarkScanResultsRequest,
                        ListHostCisBenchmarkScanResultsResponse>() {
                    @Override
                    public ListHostCisBenchmarkScanResultsResponse apply(
                            ListHostCisBenchmarkScanResultsRequest request) {
                        return client.listHostCisBenchmarkScanResults(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostCisBenchmarkScanResultSummary} objects
     * contained in responses from the listHostCisBenchmarkScanResults operation. This iterable will
     * fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostCisBenchmarkScanResultSummary} objects
     *     contained in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostCisBenchmarkScanResultSummary>
            listHostCisBenchmarkScanResultsRecordIterator(
                    final ListHostCisBenchmarkScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostCisBenchmarkScanResultsRequest.Builder,
                ListHostCisBenchmarkScanResultsRequest,
                ListHostCisBenchmarkScanResultsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostCisBenchmarkScanResultSummary>(
                new java.util.function.Supplier<ListHostCisBenchmarkScanResultsRequest.Builder>() {
                    @Override
                    public ListHostCisBenchmarkScanResultsRequest.Builder get() {
                        return ListHostCisBenchmarkScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostCisBenchmarkScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostCisBenchmarkScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostCisBenchmarkScanResultsRequest.Builder>,
                        ListHostCisBenchmarkScanResultsRequest>() {
                    @Override
                    public ListHostCisBenchmarkScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostCisBenchmarkScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostCisBenchmarkScanResultsRequest,
                        ListHostCisBenchmarkScanResultsResponse>() {
                    @Override
                    public ListHostCisBenchmarkScanResultsResponse apply(
                            ListHostCisBenchmarkScanResultsRequest request) {
                        return client.listHostCisBenchmarkScanResults(request);
                    }
                },
                new java.util.function.Function<
                        ListHostCisBenchmarkScanResultsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostCisBenchmarkScanResultSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostCisBenchmarkScanResultSummary>
                            apply(ListHostCisBenchmarkScanResultsResponse response) {
                        return response.getHostCisBenchmarkScanResultSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostEndpointProtectionScanResults operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostEndpointProtectionScanResultsResponse>
            listHostEndpointProtectionScanResultsResponseIterator(
                    final ListHostEndpointProtectionScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostEndpointProtectionScanResultsRequest.Builder,
                ListHostEndpointProtectionScanResultsRequest,
                ListHostEndpointProtectionScanResultsResponse>(
                new java.util.function.Supplier<
                        ListHostEndpointProtectionScanResultsRequest.Builder>() {
                    @Override
                    public ListHostEndpointProtectionScanResultsRequest.Builder get() {
                        return ListHostEndpointProtectionScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<
                        ListHostEndpointProtectionScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostEndpointProtectionScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostEndpointProtectionScanResultsRequest.Builder>,
                        ListHostEndpointProtectionScanResultsRequest>() {
                    @Override
                    public ListHostEndpointProtectionScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostEndpointProtectionScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostEndpointProtectionScanResultsRequest,
                        ListHostEndpointProtectionScanResultsResponse>() {
                    @Override
                    public ListHostEndpointProtectionScanResultsResponse apply(
                            ListHostEndpointProtectionScanResultsRequest request) {
                        return client.listHostEndpointProtectionScanResults(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostEndpointProtectionScanResultSummary} objects
     * contained in responses from the listHostEndpointProtectionScanResults operation. This
     * iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostEndpointProtectionScanResultSummary}
     *     objects contained in responses received from the service.
     */
    public Iterable<
                    com.oracle.bmc.vulnerabilityscanning.model
                            .HostEndpointProtectionScanResultSummary>
            listHostEndpointProtectionScanResultsRecordIterator(
                    final ListHostEndpointProtectionScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostEndpointProtectionScanResultsRequest.Builder,
                ListHostEndpointProtectionScanResultsRequest,
                ListHostEndpointProtectionScanResultsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostEndpointProtectionScanResultSummary>(
                new java.util.function.Supplier<
                        ListHostEndpointProtectionScanResultsRequest.Builder>() {
                    @Override
                    public ListHostEndpointProtectionScanResultsRequest.Builder get() {
                        return ListHostEndpointProtectionScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<
                        ListHostEndpointProtectionScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostEndpointProtectionScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostEndpointProtectionScanResultsRequest.Builder>,
                        ListHostEndpointProtectionScanResultsRequest>() {
                    @Override
                    public ListHostEndpointProtectionScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostEndpointProtectionScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostEndpointProtectionScanResultsRequest,
                        ListHostEndpointProtectionScanResultsResponse>() {
                    @Override
                    public ListHostEndpointProtectionScanResultsResponse apply(
                            ListHostEndpointProtectionScanResultsRequest request) {
                        return client.listHostEndpointProtectionScanResults(request);
                    }
                },
                new java.util.function.Function<
                        ListHostEndpointProtectionScanResultsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostEndpointProtectionScanResultSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostEndpointProtectionScanResultSummary>
                            apply(ListHostEndpointProtectionScanResultsResponse response) {
                        return response.getHostEndpointProtectionScanResultSummaryCollection()
                                .getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostPortScanResults operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostPortScanResultsResponse> listHostPortScanResultsResponseIterator(
            final ListHostPortScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostPortScanResultsRequest.Builder,
                ListHostPortScanResultsRequest,
                ListHostPortScanResultsResponse>(
                new java.util.function.Supplier<ListHostPortScanResultsRequest.Builder>() {
                    @Override
                    public ListHostPortScanResultsRequest.Builder get() {
                        return ListHostPortScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostPortScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostPortScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostPortScanResultsRequest.Builder>,
                        ListHostPortScanResultsRequest>() {
                    @Override
                    public ListHostPortScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostPortScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostPortScanResultsRequest, ListHostPortScanResultsResponse>() {
                    @Override
                    public ListHostPortScanResultsResponse apply(
                            ListHostPortScanResultsRequest request) {
                        return client.listHostPortScanResults(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostPortScanResultSummary} objects contained in
     * responses from the listHostPortScanResults operation. This iterable will fetch more data from
     * the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostPortScanResultSummary} objects contained
     *     in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostPortScanResultSummary>
            listHostPortScanResultsRecordIterator(final ListHostPortScanResultsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostPortScanResultsRequest.Builder,
                ListHostPortScanResultsRequest,
                ListHostPortScanResultsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostPortScanResultSummary>(
                new java.util.function.Supplier<ListHostPortScanResultsRequest.Builder>() {
                    @Override
                    public ListHostPortScanResultsRequest.Builder get() {
                        return ListHostPortScanResultsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostPortScanResultsResponse, String>() {
                    @Override
                    public String apply(ListHostPortScanResultsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostPortScanResultsRequest.Builder>,
                        ListHostPortScanResultsRequest>() {
                    @Override
                    public ListHostPortScanResultsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostPortScanResultsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostPortScanResultsRequest, ListHostPortScanResultsResponse>() {
                    @Override
                    public ListHostPortScanResultsResponse apply(
                            ListHostPortScanResultsRequest request) {
                        return client.listHostPortScanResults(request);
                    }
                },
                new java.util.function.Function<
                        ListHostPortScanResultsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostPortScanResultSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostPortScanResultSummary>
                            apply(ListHostPortScanResultsResponse response) {
                        return response.getHostPortScanResultSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostScanRecipes operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostScanRecipesResponse> listHostScanRecipesResponseIterator(
            final ListHostScanRecipesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostScanRecipesRequest.Builder,
                ListHostScanRecipesRequest,
                ListHostScanRecipesResponse>(
                new java.util.function.Supplier<ListHostScanRecipesRequest.Builder>() {
                    @Override
                    public ListHostScanRecipesRequest.Builder get() {
                        return ListHostScanRecipesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostScanRecipesResponse, String>() {
                    @Override
                    public String apply(ListHostScanRecipesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostScanRecipesRequest.Builder>,
                        ListHostScanRecipesRequest>() {
                    @Override
                    public ListHostScanRecipesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostScanRecipesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostScanRecipesRequest, ListHostScanRecipesResponse>() {
                    @Override
                    public ListHostScanRecipesResponse apply(ListHostScanRecipesRequest request) {
                        return client.listHostScanRecipes(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipeSummary} objects contained in
     * responses from the listHostScanRecipes operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipeSummary} objects contained in
     *     responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipeSummary>
            listHostScanRecipesRecordIterator(final ListHostScanRecipesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostScanRecipesRequest.Builder,
                ListHostScanRecipesRequest,
                ListHostScanRecipesResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostScanRecipeSummary>(
                new java.util.function.Supplier<ListHostScanRecipesRequest.Builder>() {
                    @Override
                    public ListHostScanRecipesRequest.Builder get() {
                        return ListHostScanRecipesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostScanRecipesResponse, String>() {
                    @Override
                    public String apply(ListHostScanRecipesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostScanRecipesRequest.Builder>,
                        ListHostScanRecipesRequest>() {
                    @Override
                    public ListHostScanRecipesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostScanRecipesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostScanRecipesRequest, ListHostScanRecipesResponse>() {
                    @Override
                    public ListHostScanRecipesResponse apply(ListHostScanRecipesRequest request) {
                        return client.listHostScanRecipes(request);
                    }
                },
                new java.util.function.Function<
                        ListHostScanRecipesResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostScanRecipeSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostScanRecipeSummary>
                            apply(ListHostScanRecipesResponse response) {
                        return response.getHostScanRecipeSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostScanTargetErrors operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostScanTargetErrorsResponse> listHostScanTargetErrorsResponseIterator(
            final ListHostScanTargetErrorsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostScanTargetErrorsRequest.Builder,
                ListHostScanTargetErrorsRequest,
                ListHostScanTargetErrorsResponse>(
                new java.util.function.Supplier<ListHostScanTargetErrorsRequest.Builder>() {
                    @Override
                    public ListHostScanTargetErrorsRequest.Builder get() {
                        return ListHostScanTargetErrorsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostScanTargetErrorsResponse, String>() {
                    @Override
                    public String apply(ListHostScanTargetErrorsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostScanTargetErrorsRequest.Builder>,
                        ListHostScanTargetErrorsRequest>() {
                    @Override
                    public ListHostScanTargetErrorsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostScanTargetErrorsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostScanTargetErrorsRequest, ListHostScanTargetErrorsResponse>() {
                    @Override
                    public ListHostScanTargetErrorsResponse apply(
                            ListHostScanTargetErrorsRequest request) {
                        return client.listHostScanTargetErrors(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetErrorSummary} objects contained in
     * responses from the listHostScanTargetErrors operation. This iterable will fetch more data
     * from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetErrorSummary} objects contained
     *     in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetErrorSummary>
            listHostScanTargetErrorsRecordIterator(final ListHostScanTargetErrorsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostScanTargetErrorsRequest.Builder,
                ListHostScanTargetErrorsRequest,
                ListHostScanTargetErrorsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetErrorSummary>(
                new java.util.function.Supplier<ListHostScanTargetErrorsRequest.Builder>() {
                    @Override
                    public ListHostScanTargetErrorsRequest.Builder get() {
                        return ListHostScanTargetErrorsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostScanTargetErrorsResponse, String>() {
                    @Override
                    public String apply(ListHostScanTargetErrorsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostScanTargetErrorsRequest.Builder>,
                        ListHostScanTargetErrorsRequest>() {
                    @Override
                    public ListHostScanTargetErrorsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostScanTargetErrorsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostScanTargetErrorsRequest, ListHostScanTargetErrorsResponse>() {
                    @Override
                    public ListHostScanTargetErrorsResponse apply(
                            ListHostScanTargetErrorsRequest request) {
                        return client.listHostScanTargetErrors(request);
                    }
                },
                new java.util.function.Function<
                        ListHostScanTargetErrorsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostScanTargetErrorSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostScanTargetErrorSummary>
                            apply(ListHostScanTargetErrorsResponse response) {
                        return response.getHostScanTargetErrorSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostScanTargets operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostScanTargetsResponse> listHostScanTargetsResponseIterator(
            final ListHostScanTargetsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostScanTargetsRequest.Builder,
                ListHostScanTargetsRequest,
                ListHostScanTargetsResponse>(
                new java.util.function.Supplier<ListHostScanTargetsRequest.Builder>() {
                    @Override
                    public ListHostScanTargetsRequest.Builder get() {
                        return ListHostScanTargetsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostScanTargetsResponse, String>() {
                    @Override
                    public String apply(ListHostScanTargetsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostScanTargetsRequest.Builder>,
                        ListHostScanTargetsRequest>() {
                    @Override
                    public ListHostScanTargetsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostScanTargetsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostScanTargetsRequest, ListHostScanTargetsResponse>() {
                    @Override
                    public ListHostScanTargetsResponse apply(ListHostScanTargetsRequest request) {
                        return client.listHostScanTargets(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetSummary} objects contained in
     * responses from the listHostScanTargets operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetSummary} objects contained in
     *     responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetSummary>
            listHostScanTargetsRecordIterator(final ListHostScanTargetsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostScanTargetsRequest.Builder,
                ListHostScanTargetsRequest,
                ListHostScanTargetsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostScanTargetSummary>(
                new java.util.function.Supplier<ListHostScanTargetsRequest.Builder>() {
                    @Override
                    public ListHostScanTargetsRequest.Builder get() {
                        return ListHostScanTargetsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostScanTargetsResponse, String>() {
                    @Override
                    public String apply(ListHostScanTargetsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostScanTargetsRequest.Builder>,
                        ListHostScanTargetsRequest>() {
                    @Override
                    public ListHostScanTargetsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostScanTargetsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostScanTargetsRequest, ListHostScanTargetsResponse>() {
                    @Override
                    public ListHostScanTargetsResponse apply(ListHostScanTargetsRequest request) {
                        return client.listHostScanTargets(request);
                    }
                },
                new java.util.function.Function<
                        ListHostScanTargetsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostScanTargetSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostScanTargetSummary>
                            apply(ListHostScanTargetsResponse response) {
                        return response.getHostScanTargetSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostVulnerabilities operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostVulnerabilitiesResponse> listHostVulnerabilitiesResponseIterator(
            final ListHostVulnerabilitiesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostVulnerabilitiesRequest.Builder,
                ListHostVulnerabilitiesRequest,
                ListHostVulnerabilitiesResponse>(
                new java.util.function.Supplier<ListHostVulnerabilitiesRequest.Builder>() {
                    @Override
                    public ListHostVulnerabilitiesRequest.Builder get() {
                        return ListHostVulnerabilitiesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostVulnerabilitiesResponse, String>() {
                    @Override
                    public String apply(ListHostVulnerabilitiesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostVulnerabilitiesRequest.Builder>,
                        ListHostVulnerabilitiesRequest>() {
                    @Override
                    public ListHostVulnerabilitiesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostVulnerabilitiesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilitiesRequest, ListHostVulnerabilitiesResponse>() {
                    @Override
                    public ListHostVulnerabilitiesResponse apply(
                            ListHostVulnerabilitiesRequest request) {
                        return client.listHostVulnerabilities(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilitySummary} objects contained in
     * responses from the listHostVulnerabilities operation. This iterable will fetch more data from
     * the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilitySummary} objects contained in
     *     responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilitySummary>
            listHostVulnerabilitiesRecordIterator(final ListHostVulnerabilitiesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostVulnerabilitiesRequest.Builder,
                ListHostVulnerabilitiesRequest,
                ListHostVulnerabilitiesResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilitySummary>(
                new java.util.function.Supplier<ListHostVulnerabilitiesRequest.Builder>() {
                    @Override
                    public ListHostVulnerabilitiesRequest.Builder get() {
                        return ListHostVulnerabilitiesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListHostVulnerabilitiesResponse, String>() {
                    @Override
                    public String apply(ListHostVulnerabilitiesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostVulnerabilitiesRequest.Builder>,
                        ListHostVulnerabilitiesRequest>() {
                    @Override
                    public ListHostVulnerabilitiesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostVulnerabilitiesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilitiesRequest, ListHostVulnerabilitiesResponse>() {
                    @Override
                    public ListHostVulnerabilitiesResponse apply(
                            ListHostVulnerabilitiesRequest request) {
                        return client.listHostVulnerabilities(request);
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilitiesResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostVulnerabilitySummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostVulnerabilitySummary>
                            apply(ListHostVulnerabilitiesResponse response) {
                        return response.getHostVulnerabilitySummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listHostVulnerabilityImpactedHosts operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListHostVulnerabilityImpactedHostsResponse>
            listHostVulnerabilityImpactedHostsResponseIterator(
                    final ListHostVulnerabilityImpactedHostsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListHostVulnerabilityImpactedHostsRequest.Builder,
                ListHostVulnerabilityImpactedHostsRequest,
                ListHostVulnerabilityImpactedHostsResponse>(
                new java.util.function.Supplier<
                        ListHostVulnerabilityImpactedHostsRequest.Builder>() {
                    @Override
                    public ListHostVulnerabilityImpactedHostsRequest.Builder get() {
                        return ListHostVulnerabilityImpactedHostsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilityImpactedHostsResponse, String>() {
                    @Override
                    public String apply(ListHostVulnerabilityImpactedHostsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostVulnerabilityImpactedHostsRequest.Builder>,
                        ListHostVulnerabilityImpactedHostsRequest>() {
                    @Override
                    public ListHostVulnerabilityImpactedHostsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostVulnerabilityImpactedHostsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilityImpactedHostsRequest,
                        ListHostVulnerabilityImpactedHostsResponse>() {
                    @Override
                    public ListHostVulnerabilityImpactedHostsResponse apply(
                            ListHostVulnerabilityImpactedHostsRequest request) {
                        return client.listHostVulnerabilityImpactedHosts(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilityImpactedHostSummary} objects
     * contained in responses from the listHostVulnerabilityImpactedHosts operation. This iterable
     * will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilityImpactedHostSummary} objects
     *     contained in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilityImpactedHostSummary>
            listHostVulnerabilityImpactedHostsRecordIterator(
                    final ListHostVulnerabilityImpactedHostsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListHostVulnerabilityImpactedHostsRequest.Builder,
                ListHostVulnerabilityImpactedHostsRequest,
                ListHostVulnerabilityImpactedHostsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.HostVulnerabilityImpactedHostSummary>(
                new java.util.function.Supplier<
                        ListHostVulnerabilityImpactedHostsRequest.Builder>() {
                    @Override
                    public ListHostVulnerabilityImpactedHostsRequest.Builder get() {
                        return ListHostVulnerabilityImpactedHostsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilityImpactedHostsResponse, String>() {
                    @Override
                    public String apply(ListHostVulnerabilityImpactedHostsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListHostVulnerabilityImpactedHostsRequest.Builder>,
                        ListHostVulnerabilityImpactedHostsRequest>() {
                    @Override
                    public ListHostVulnerabilityImpactedHostsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListHostVulnerabilityImpactedHostsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilityImpactedHostsRequest,
                        ListHostVulnerabilityImpactedHostsResponse>() {
                    @Override
                    public ListHostVulnerabilityImpactedHostsResponse apply(
                            ListHostVulnerabilityImpactedHostsRequest request) {
                        return client.listHostVulnerabilityImpactedHosts(request);
                    }
                },
                new java.util.function.Function<
                        ListHostVulnerabilityImpactedHostsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .HostVulnerabilityImpactedHostSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .HostVulnerabilityImpactedHostSummary>
                            apply(ListHostVulnerabilityImpactedHostsResponse response) {
                        return response.getHostVulnerabilityImpactedHostSummaryCollection()
                                .getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listVulnerabilities operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListVulnerabilitiesResponse> listVulnerabilitiesResponseIterator(
            final ListVulnerabilitiesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListVulnerabilitiesRequest.Builder,
                ListVulnerabilitiesRequest,
                ListVulnerabilitiesResponse>(
                new java.util.function.Supplier<ListVulnerabilitiesRequest.Builder>() {
                    @Override
                    public ListVulnerabilitiesRequest.Builder get() {
                        return ListVulnerabilitiesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListVulnerabilitiesResponse, String>() {
                    @Override
                    public String apply(ListVulnerabilitiesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListVulnerabilitiesRequest.Builder>,
                        ListVulnerabilitiesRequest>() {
                    @Override
                    public ListVulnerabilitiesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListVulnerabilitiesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilitiesRequest, ListVulnerabilitiesResponse>() {
                    @Override
                    public ListVulnerabilitiesResponse apply(ListVulnerabilitiesRequest request) {
                        return client.listVulnerabilities(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.VulnerabilitySummary} objects contained in
     * responses from the listVulnerabilities operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.VulnerabilitySummary} objects contained in
     *     responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.VulnerabilitySummary>
            listVulnerabilitiesRecordIterator(final ListVulnerabilitiesRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListVulnerabilitiesRequest.Builder,
                ListVulnerabilitiesRequest,
                ListVulnerabilitiesResponse,
                com.oracle.bmc.vulnerabilityscanning.model.VulnerabilitySummary>(
                new java.util.function.Supplier<ListVulnerabilitiesRequest.Builder>() {
                    @Override
                    public ListVulnerabilitiesRequest.Builder get() {
                        return ListVulnerabilitiesRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListVulnerabilitiesResponse, String>() {
                    @Override
                    public String apply(ListVulnerabilitiesResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListVulnerabilitiesRequest.Builder>,
                        ListVulnerabilitiesRequest>() {
                    @Override
                    public ListVulnerabilitiesRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListVulnerabilitiesRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilitiesRequest, ListVulnerabilitiesResponse>() {
                    @Override
                    public ListVulnerabilitiesResponse apply(ListVulnerabilitiesRequest request) {
                        return client.listVulnerabilities(request);
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilitiesResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .VulnerabilitySummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model.VulnerabilitySummary>
                            apply(ListVulnerabilitiesResponse response) {
                        return response.getVulnerabilitySummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listVulnerabilityImpactedContainers operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListVulnerabilityImpactedContainersResponse>
            listVulnerabilityImpactedContainersResponseIterator(
                    final ListVulnerabilityImpactedContainersRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListVulnerabilityImpactedContainersRequest.Builder,
                ListVulnerabilityImpactedContainersRequest,
                ListVulnerabilityImpactedContainersResponse>(
                new java.util.function.Supplier<
                        ListVulnerabilityImpactedContainersRequest.Builder>() {
                    @Override
                    public ListVulnerabilityImpactedContainersRequest.Builder get() {
                        return ListVulnerabilityImpactedContainersRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedContainersResponse, String>() {
                    @Override
                    public String apply(ListVulnerabilityImpactedContainersResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListVulnerabilityImpactedContainersRequest.Builder>,
                        ListVulnerabilityImpactedContainersRequest>() {
                    @Override
                    public ListVulnerabilityImpactedContainersRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListVulnerabilityImpactedContainersRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedContainersRequest,
                        ListVulnerabilityImpactedContainersResponse>() {
                    @Override
                    public ListVulnerabilityImpactedContainersResponse apply(
                            ListVulnerabilityImpactedContainersRequest request) {
                        return client.listVulnerabilityImpactedContainers(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedContainerSummary} objects
     * contained in responses from the listVulnerabilityImpactedContainers operation. This iterable
     * will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedContainerSummary} objects
     *     contained in responses received from the service.
     */
    public Iterable<
                    com.oracle.bmc.vulnerabilityscanning.model
                            .VulnerabilityImpactedContainerSummary>
            listVulnerabilityImpactedContainersRecordIterator(
                    final ListVulnerabilityImpactedContainersRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListVulnerabilityImpactedContainersRequest.Builder,
                ListVulnerabilityImpactedContainersRequest,
                ListVulnerabilityImpactedContainersResponse,
                com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedContainerSummary>(
                new java.util.function.Supplier<
                        ListVulnerabilityImpactedContainersRequest.Builder>() {
                    @Override
                    public ListVulnerabilityImpactedContainersRequest.Builder get() {
                        return ListVulnerabilityImpactedContainersRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedContainersResponse, String>() {
                    @Override
                    public String apply(ListVulnerabilityImpactedContainersResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListVulnerabilityImpactedContainersRequest.Builder>,
                        ListVulnerabilityImpactedContainersRequest>() {
                    @Override
                    public ListVulnerabilityImpactedContainersRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListVulnerabilityImpactedContainersRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedContainersRequest,
                        ListVulnerabilityImpactedContainersResponse>() {
                    @Override
                    public ListVulnerabilityImpactedContainersResponse apply(
                            ListVulnerabilityImpactedContainersRequest request) {
                        return client.listVulnerabilityImpactedContainers(request);
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedContainersResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .VulnerabilityImpactedContainerSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .VulnerabilityImpactedContainerSummary>
                            apply(ListVulnerabilityImpactedContainersResponse response) {
                        return response.getVulnerabilityImpactedContainerSummaryCollection()
                                .getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listVulnerabilityImpactedHosts operation. This iterable will fetch more data from the server
     * as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListVulnerabilityImpactedHostsResponse>
            listVulnerabilityImpactedHostsResponseIterator(
                    final ListVulnerabilityImpactedHostsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListVulnerabilityImpactedHostsRequest.Builder,
                ListVulnerabilityImpactedHostsRequest,
                ListVulnerabilityImpactedHostsResponse>(
                new java.util.function.Supplier<ListVulnerabilityImpactedHostsRequest.Builder>() {
                    @Override
                    public ListVulnerabilityImpactedHostsRequest.Builder get() {
                        return ListVulnerabilityImpactedHostsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListVulnerabilityImpactedHostsResponse, String>() {
                    @Override
                    public String apply(ListVulnerabilityImpactedHostsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListVulnerabilityImpactedHostsRequest.Builder>,
                        ListVulnerabilityImpactedHostsRequest>() {
                    @Override
                    public ListVulnerabilityImpactedHostsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListVulnerabilityImpactedHostsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedHostsRequest,
                        ListVulnerabilityImpactedHostsResponse>() {
                    @Override
                    public ListVulnerabilityImpactedHostsResponse apply(
                            ListVulnerabilityImpactedHostsRequest request) {
                        return client.listVulnerabilityImpactedHosts(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedHostSummary} objects
     * contained in responses from the listVulnerabilityImpactedHosts operation. This iterable will
     * fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedHostSummary} objects
     *     contained in responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedHostSummary>
            listVulnerabilityImpactedHostsRecordIterator(
                    final ListVulnerabilityImpactedHostsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListVulnerabilityImpactedHostsRequest.Builder,
                ListVulnerabilityImpactedHostsRequest,
                ListVulnerabilityImpactedHostsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.VulnerabilityImpactedHostSummary>(
                new java.util.function.Supplier<ListVulnerabilityImpactedHostsRequest.Builder>() {
                    @Override
                    public ListVulnerabilityImpactedHostsRequest.Builder get() {
                        return ListVulnerabilityImpactedHostsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListVulnerabilityImpactedHostsResponse, String>() {
                    @Override
                    public String apply(ListVulnerabilityImpactedHostsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListVulnerabilityImpactedHostsRequest.Builder>,
                        ListVulnerabilityImpactedHostsRequest>() {
                    @Override
                    public ListVulnerabilityImpactedHostsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListVulnerabilityImpactedHostsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedHostsRequest,
                        ListVulnerabilityImpactedHostsResponse>() {
                    @Override
                    public ListVulnerabilityImpactedHostsResponse apply(
                            ListVulnerabilityImpactedHostsRequest request) {
                        return client.listVulnerabilityImpactedHosts(request);
                    }
                },
                new java.util.function.Function<
                        ListVulnerabilityImpactedHostsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model
                                        .VulnerabilityImpactedHostSummary>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model
                                            .VulnerabilityImpactedHostSummary>
                            apply(ListVulnerabilityImpactedHostsResponse response) {
                        return response.getVulnerabilityImpactedHostSummaryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listWorkRequestErrors operation. This iterable will fetch more data from the server as
     * needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListWorkRequestErrorsResponse> listWorkRequestErrorsResponseIterator(
            final ListWorkRequestErrorsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListWorkRequestErrorsRequest.Builder,
                ListWorkRequestErrorsRequest,
                ListWorkRequestErrorsResponse>(
                new java.util.function.Supplier<ListWorkRequestErrorsRequest.Builder>() {
                    @Override
                    public ListWorkRequestErrorsRequest.Builder get() {
                        return ListWorkRequestErrorsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestErrorsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestErrorsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestErrorsRequest.Builder>,
                        ListWorkRequestErrorsRequest>() {
                    @Override
                    public ListWorkRequestErrorsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestErrorsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestErrorsRequest, ListWorkRequestErrorsResponse>() {
                    @Override
                    public ListWorkRequestErrorsResponse apply(
                            ListWorkRequestErrorsRequest request) {
                        return client.listWorkRequestErrors(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.WorkRequestError} objects contained in responses
     * from the listWorkRequestErrors operation. This iterable will fetch more data from the server
     * as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.WorkRequestError} objects contained in
     *     responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.WorkRequestError>
            listWorkRequestErrorsRecordIterator(final ListWorkRequestErrorsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListWorkRequestErrorsRequest.Builder,
                ListWorkRequestErrorsRequest,
                ListWorkRequestErrorsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.WorkRequestError>(
                new java.util.function.Supplier<ListWorkRequestErrorsRequest.Builder>() {
                    @Override
                    public ListWorkRequestErrorsRequest.Builder get() {
                        return ListWorkRequestErrorsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestErrorsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestErrorsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestErrorsRequest.Builder>,
                        ListWorkRequestErrorsRequest>() {
                    @Override
                    public ListWorkRequestErrorsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestErrorsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestErrorsRequest, ListWorkRequestErrorsResponse>() {
                    @Override
                    public ListWorkRequestErrorsResponse apply(
                            ListWorkRequestErrorsRequest request) {
                        return client.listWorkRequestErrors(request);
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestErrorsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model.WorkRequestError>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model.WorkRequestError>
                            apply(ListWorkRequestErrorsResponse response) {
                        return response.getWorkRequestErrorCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listWorkRequestLogs operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListWorkRequestLogsResponse> listWorkRequestLogsResponseIterator(
            final ListWorkRequestLogsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListWorkRequestLogsRequest.Builder,
                ListWorkRequestLogsRequest,
                ListWorkRequestLogsResponse>(
                new java.util.function.Supplier<ListWorkRequestLogsRequest.Builder>() {
                    @Override
                    public ListWorkRequestLogsRequest.Builder get() {
                        return ListWorkRequestLogsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestLogsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestLogsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestLogsRequest.Builder>,
                        ListWorkRequestLogsRequest>() {
                    @Override
                    public ListWorkRequestLogsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestLogsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestLogsRequest, ListWorkRequestLogsResponse>() {
                    @Override
                    public ListWorkRequestLogsResponse apply(ListWorkRequestLogsRequest request) {
                        return client.listWorkRequestLogs(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntry} objects contained in
     * responses from the listWorkRequestLogs operation. This iterable will fetch more data from the
     * server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntry} objects contained in
     *     responses received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntry>
            listWorkRequestLogsRecordIterator(final ListWorkRequestLogsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListWorkRequestLogsRequest.Builder,
                ListWorkRequestLogsRequest,
                ListWorkRequestLogsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntry>(
                new java.util.function.Supplier<ListWorkRequestLogsRequest.Builder>() {
                    @Override
                    public ListWorkRequestLogsRequest.Builder get() {
                        return ListWorkRequestLogsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestLogsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestLogsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestLogsRequest.Builder>,
                        ListWorkRequestLogsRequest>() {
                    @Override
                    public ListWorkRequestLogsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestLogsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestLogsRequest, ListWorkRequestLogsResponse>() {
                    @Override
                    public ListWorkRequestLogsResponse apply(ListWorkRequestLogsRequest request) {
                        return client.listWorkRequestLogs(request);
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestLogsResponse,
                        java.util.List<
                                com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntry>>() {
                    @Override
                    public java.util.List<
                                    com.oracle.bmc.vulnerabilityscanning.model.WorkRequestLogEntry>
                            apply(ListWorkRequestLogsResponse response) {
                        return response.getWorkRequestLogEntryCollection().getItems();
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the responses received from the
     * listWorkRequests operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the responses
     *     received from the service.
     */
    public Iterable<ListWorkRequestsResponse> listWorkRequestsResponseIterator(
            final ListWorkRequestsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseIterable<
                ListWorkRequestsRequest.Builder, ListWorkRequestsRequest, ListWorkRequestsResponse>(
                new java.util.function.Supplier<ListWorkRequestsRequest.Builder>() {
                    @Override
                    public ListWorkRequestsRequest.Builder get() {
                        return ListWorkRequestsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestsRequest.Builder>,
                        ListWorkRequestsRequest>() {
                    @Override
                    public ListWorkRequestsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestsRequest, ListWorkRequestsResponse>() {
                    @Override
                    public ListWorkRequestsResponse apply(ListWorkRequestsRequest request) {
                        return client.listWorkRequests(request);
                    }
                });
    }

    /**
     * Creates a new iterable which will iterate over the {@link
     * com.oracle.bmc.vulnerabilityscanning.model.WorkRequest} objects contained in responses from
     * the listWorkRequests operation. This iterable will fetch more data from the server as needed.
     *
     * @param request a request which can be sent to the service operation
     * @return an {@link java.lang.Iterable} which can be used to iterate over the {@link
     *     com.oracle.bmc.vulnerabilityscanning.model.WorkRequest} objects contained in responses
     *     received from the service.
     */
    public Iterable<com.oracle.bmc.vulnerabilityscanning.model.WorkRequest>
            listWorkRequestsRecordIterator(final ListWorkRequestsRequest request) {
        return new com.oracle.bmc.paginator.internal.ResponseRecordIterable<
                ListWorkRequestsRequest.Builder,
                ListWorkRequestsRequest,
                ListWorkRequestsResponse,
                com.oracle.bmc.vulnerabilityscanning.model.WorkRequest>(
                new java.util.function.Supplier<ListWorkRequestsRequest.Builder>() {
                    @Override
                    public ListWorkRequestsRequest.Builder get() {
                        return ListWorkRequestsRequest.builder().copy(request);
                    }
                },
                new java.util.function.Function<ListWorkRequestsResponse, String>() {
                    @Override
                    public String apply(ListWorkRequestsResponse response) {
                        return response.getOpcNextPage();
                    }
                },
                new java.util.function.Function<
                        com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                ListWorkRequestsRequest.Builder>,
                        ListWorkRequestsRequest>() {
                    @Override
                    public ListWorkRequestsRequest apply(
                            com.oracle.bmc.paginator.internal.RequestBuilderAndToken<
                                            ListWorkRequestsRequest.Builder>
                                    input) {
                        if (input.getNextPageToken() == null) {
                            return input.getRequestBuilder().build();
                        } else {
                            return input.getRequestBuilder()
                                    .page(input.getNextPageToken().orElse(null))
                                    .build();
                        }
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestsRequest, ListWorkRequestsResponse>() {
                    @Override
                    public ListWorkRequestsResponse apply(ListWorkRequestsRequest request) {
                        return client.listWorkRequests(request);
                    }
                },
                new java.util.function.Function<
                        ListWorkRequestsResponse,
                        java.util.List<com.oracle.bmc.vulnerabilityscanning.model.WorkRequest>>() {
                    @Override
                    public java.util.List<com.oracle.bmc.vulnerabilityscanning.model.WorkRequest>
                            apply(ListWorkRequestsResponse response) {
                        return response.getWorkRequestCollection().getItems();
                    }
                });
    }
}
