package com.tschuchort.compiletesting

import java.io.File
import java.net.URLClassLoader
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi

/** Result of the compilation. */
@ExperimentalCompilerApi
sealed interface CompilationResult {
  /** The exit code of the compilation. */
  val exitCode: KotlinCompilation.ExitCode
  /** Messages that were printed by the compilation. */
  val messages: String
  /** Messages with captured diagnostic severity. */
  val diagnosticMessages: List<DiagnosticMessage>
  /** The directory where compiled files will be output to. */
  val outputDirectory: File

  /** Messages filtered by the given severities */
  fun messagesWithSeverity(vararg severities: DiagnosticSeverity): String =
    diagnosticMessages.filter { it.severity in severities }.joinToString("\n")
}

@ExperimentalCompilerApi
class JsCompilationResult(
  override val exitCode: KotlinCompilation.ExitCode,
  override val messages: String,
  override val diagnosticMessages: List<DiagnosticMessage>,
  private val compilation: KotlinJsCompilation,
) : CompilationResult {
  override val outputDirectory: File
    get() = compilation.outputDir

  /** JS files output by the compilation. */
  val jsFiles: List<File> = outputDirectory.listFilesRecursively()
}

/** Result of the compilation */
@ExperimentalCompilerApi
class JvmCompilationResult(
  override val exitCode: KotlinCompilation.ExitCode,
  override val messages: String,
  override val diagnosticMessages: List<DiagnosticMessage>,
  private val compilation: KotlinCompilation,
) : CompilationResult {
  override val outputDirectory: File
    get() = compilation.classesDir

  /** ClassLoader to load the compiled classes */
  val classLoader =
    URLClassLoader(
      // Include the original classpaths and the output directory to be able to load classes from
      // dependencies.
      compilation.classpaths.plus(outputDirectory).map { it.toURI().toURL() }.toTypedArray(),
      this::class.java.classLoader,
    )

  /** Compiled classes and resources output by the compilation. */
  val compiledClassAndResourceFiles: List<File> = outputDirectory.listFilesRecursively()

  /**
   * Intermediate source and resource files generated by the annotation processor that will be
   * compiled in the next steps.
   */
  val sourcesGeneratedByAnnotationProcessor: List<File> =
    compilation.kaptSourceDir.listFilesRecursively() +
      compilation.kaptKotlinGeneratedDir.listFilesRecursively()

  /** Stub files generated by kapt */
  val generatedStubFiles: List<File> = compilation.kaptStubsDir.listFilesRecursively()

  /**
   * The class, resource and intermediate source files generated during the compilation. Does not
   * include stub files and kapt incremental data.
   */
  val generatedFiles: Collection<File> =
    sourcesGeneratedByAnnotationProcessor + compiledClassAndResourceFiles + generatedStubFiles
}
