/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.server;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import eu.maydu.gwt.validation.client.InvalidValueSerializable;
import eu.maydu.gwt.validation.client.ValidationException;
import eu.maydu.gwt.validation.client.ValidationProcessor;



/**
 * This class is used to do some basic validation checks
 * on the server side. If validation errors occur it is able
 * to throw a ValidationException that can be consumed by
 * the UI-part of the maydu GWT Validation Framework.
 * 
 * Thus your interfaces that accept incoming data should declare
 * that they throw a ValidationException in order to talk the news
 * back to the client.
 * 
 * You can use it like ('Fail late mode' aka <code>throwOnFirstError=false</code>):
 * <pre>
 * <code>
 * 		ServerValidation validation = new ServerValidation(false);
 *		validation
 *		.notNull(g.getHomePlayer(), "homePlayer")
 *		.notNull(g.getAwayPlayer(), "awayPlayer")
 *		.notEqual(g.getAwayPlayer(), g.getHomePlayer(), "homePlayer,awayPlayer")
 *		.notNull(g.getHomeTeam(), "homeTeam")
 *		.notNull(g.getAwayTeam(), "awayTeam")
 *		.notEqual(g.getAwayTeam(), g.getHomeTeam(), "homeTeam,awayTeam")
 *		.notNull(score, "scores")
 *		.inRange(score.getHomeFirstHalf(), 0, 20, "homeFirstHalf")
 *		.inRange(score.getAwayFirstHalf(), 0, 20, "awayFirstHalf")
 *		.inRange(score.getHomeSecondHalf(), 0, 20, "homeSecondHalf")
 *		.inRange(score.getAwaySecondHalf(), 0, 20, "awaySecondHalf");
 *		if(score.getHadOvertime()) {
 *			validation
 *			.inRange(score.getHomeFirstOvertime(), 0, 20, "homeFirstOvertime")
 *			.inRange(score.getAwayFirstOvertime(), 0, 20, "awayFirstOvertime")
 *			.inRange(score.getHomeSecondOvertime(), 0, 20, "homeSecondOvertime")
 *			.inRange(score.getAwaySecondOvertime(), 0, 20, "awaySecondOvertime");
 *			if(score.getHadPenalty()) {
 *				validation
 *				.inRange(score.getHomePenalty(), 0, 99, "homePenalty")
 *				.inRange(score.getAwayPenalty(), 0, 99, "awayPenalty");
 *			}
 *		}
 *		validation.validate();
 * </code>
 * </pre>
 * Or like that in the 'fail fast mode' aka <code>throwOnFirstError=true</code>:
 * <pre>
 *  <code>
 *		new ServerValidation()
 *		.notNull(g.getHomePlayer(), "homePlayer")
 *		.notNull(g.getAwayPlayer(), "awayPlayer")
 *		.notEqual(g.getAwayPlayer(), g.getHomePlayer(), "homePlayer,awayPlayer")
 *		.notNull(g.getHomeTeam(), "homeTeam")
 *		.notNull(g.getAwayTeam(), "awayTeam")
 *		.notEqual(g.getAwayTeam(), g.getHomeTeam(), "homeTeam,awayTeam")
 *		.notNull(score, "scores")
 *		.inRange(score.getHomeFirstHalf(), 0, 20, "homeFirstHalf")
 *		.inRange(score.getAwayFirstHalf(), 0, 20, "awayFirstHalf")
 *		.inRange(score.getHomeSecondHalf(), 0, 20, "homeSecondHalf")
 *		.inRange(score.getAwaySecondHalf(), 0, 20, "awaySecondHalf");
 *  </code>
 *  </pre>
 *  Notice that there is no need to call the <code>validate()</code> method
 *  at the end. Because it will directly fail when any of the validations fail.
 *  So if it goes through to the end, there were no validation errors.
 * 
 * As you can see there are quite some ways you can validate the input from the
 * client on the server side.
 * 
 * If you still need additional checks that can't be covered by the existing functionality
 * there is an easy way to still get the benefits of easily talking your validation errors
 * back to the client and having him show the user that he did some wrong data entry:
 * 
 * You just need to do your custom validation, if it fails you can talk back the error to the
 * UI by invoking:
 * 
 * <code>
 * ServerValidation.exception("msgKey", "propertyName");
 * </code>
 * 
 * This creates a validation exception for you and throws it so the client UI can consume it.
 * The 'msgKey' parameter defines the key to your own custom localized message.
 * 
 * Which will on the client side result in the invocation of the <code>getCustomMessage(String)</code>
 * method of the <code>ValidationMessages</code> class which you should extend and then override
 * the <code>getCustomMessage(String)</code> method in order to return the correct localized string for
 * the key: 'msgKey'.
 * 
 * The 'propertyName' defines the name of the field that you gave it for validation that this
 * validation error should be bound to.
 * 
 * 
 * 
 * @author Anatol Mayen
 *
 */
public class ServerValidation {
	
	private List<InvalidValueSerializable> invalidPool = new LinkedList<InvalidValueSerializable>();
	boolean throwOnFirstError;
	
	/**
	 * Default constructor.
	 * Will throw immediately on first error,
	 * so no additional <code>validate();</code
	 * call after the last validation is needed.
	 */
	public ServerValidation() {
		this.throwOnFirstError = true;
	}
	
	/**
	 * Specify whether to throw on the first error or
	 * accumulate all errors and then throw at the end.
	 * When the additional call to <code>validate();</code>
	 * at the end of the validation chain occurs.
	 * 
	 * @param throwOnFirstError
	 */
	public ServerValidation(boolean throwOnFirstError) {
		this.throwOnFirstError = throwOnFirstError;
	}
	
	
	/**
	 * Checks if an object is not null.
	 * 
	 * @param a The object to check
	 * @param propertyName The property name of the field the object belonged to. Should be equal to the client side name for the validators.
	 * @return Returns the ServerValidation object in order to chain the validation calls
	 * @throws ValidationException Will be thrown if the given object <i>a</i> is null and this <code>ServerValidation</code> instance was initialized with <code>throwOnFirstError = true</code>. 
	 */
	public ServerValidation notNull(Object a, String propertyName) throws ValidationException {
		return notNull(a, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"notNull");
	}
	
	public ServerValidation notNull(Object a, String propertyName, String messageKey) throws ValidationException {
		if(a==null)
			throwValidationException(propertyName, messageKey);
		return this;
	}
	
	public ServerValidation isNull(Object a, String propertyName) throws ValidationException {
		return isNull(a, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"null");
	}
	
	public ServerValidation isNull(Object a, String propertyName, String messageKey) throws ValidationException {
		if(a != null)
			throwValidationException(propertyName, messageKey);
		
		return this;
	}
	
	public ServerValidation notEqual(Object a, Object b, String propertyName) throws ValidationException {
		return notEqual(a, b, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"notEqual" );
	}
	
	public ServerValidation notEqual(Object a, Object b, String propertyName, String messageKey) throws ValidationException {
		if(a == null && b == null) {
			throwValidationException(propertyName, messageKey);
			return this;
		}
		if(a == null || b == null)
			return this;
			
		if(a.equals(b))
			throwValidationException(propertyName, messageKey);
		
		return this;
	}

	
	public ServerValidation equal(Object a, Object b, String propertyName) throws ValidationException{
		
		return equal(a, b, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"equal");

	}
	
	public ServerValidation equal(Object a, Object b, String propertyName, String messageKey) throws ValidationException{
		if(a == null && b == null)
			return this;
		if(a == null || b == null) {
			throwValidationException(propertyName, messageKey);
			return this;
		}
			
		if(a.equals(b))
			return this;
		
		throwValidationException(propertyName, messageKey);
		return this;
	}
		
	public ServerValidation min(int value, int minValue, String propertyName) {
		return min(value, minValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"min_i");
	}
	
	public ServerValidation min(int value, int minValue, String propertyName, String messageKey) {
		if(value < minValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+value);
		return this;
	}
	
	public ServerValidation min(double value, double minValue, String propertyName) {
		return min(value, minValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"min_d");
	}
	
	public ServerValidation min(double value, double minValue, String propertyName, String messageKey) {
		if(value < minValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+value);
		return this;
	}
	
	public ServerValidation min(long value, long minValue, String propertyName) {
		return min(value, minValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"min_l");
	}
	
	public ServerValidation min(long value, long minValue, String propertyName, String messageKey) {
		if(value < minValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+value);
		return this;
	}
	
	public ServerValidation min(float value, float minValue, String propertyName) {
		return min(value, minValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"min_f");
	}
	
	public ServerValidation min(float value, float minValue, String propertyName, String messageKey) {
		if(value < minValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+value);
		return this;
	}
	
	
	public ServerValidation max(int value, int maxValue, String propertyName) {
		return max(value, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"max_i");
	}
	
	public ServerValidation max(int value, int maxValue, String propertyName, String messageKey) {
		if(value > maxValue)
			throwValidationException(propertyName, messageKey+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation max(double value, double maxValue, String propertyName) {
		return max(value, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"max_d");
	}
	
	public ServerValidation max(double value, double maxValue, String propertyName, String messageKey) {
		if(value > maxValue)
			throwValidationException(propertyName, messageKey+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation max(long value, long maxValue, String propertyName) {
		return max(value, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"max_l");
	}
	
	public ServerValidation max(long value, long maxValue, String propertyName, String messageKey) {
		if(value > maxValue)
			throwValidationException(propertyName, messageKey+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation max(float value, float maxValue, String propertyName) {
		return max(value, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"max_f");
	}
	
	public ServerValidation max(float value, float maxValue, String propertyName, String messageKey) {
		if(value > maxValue)
			throwValidationException(propertyName, messageKey+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation inRange(int value, int minValue, int maxValue, String propertyName) {
		return inRange(value, minValue, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"inRange_i"); 
	}
	
	public ServerValidation inRange(int value, int minValue, int maxValue, String propertyName, String messageKey) {
		if(value < minValue || value > maxValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation inRange(long value, long minValue, long maxValue, String propertyName) {
		return inRange(value, minValue, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"inRange_l"); 
	}
	
	public ServerValidation inRange(long value, long minValue, long maxValue, String propertyName, String messageKey) {
		if(value < minValue || value > maxValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation inRange(float value, float minValue, float maxValue, String propertyName) {
		return inRange(value, minValue, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"inRange_f");
	}
	
	public ServerValidation inRange(float value, float minValue, float maxValue, String propertyName, String messageKey) {
		if(value < minValue || value > maxValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation inRange(double value, double minValue, double maxValue, String propertyName) {
		return inRange(value, minValue, maxValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"inRange_d");
	}
	
	public ServerValidation inRange(double value, double minValue, double maxValue, String propertyName, String messageKey) {
		if(value < minValue || value > maxValue)
			throwValidationException(propertyName, messageKey+":"+minValue+":"+maxValue+":"+value);
		return this;
	}
	
	public ServerValidation length(String value, int minLength, int maxLength, String propertyName) {
		return length(value, minLength, maxLength, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"length");
	}	
	
	public ServerValidation length(String value, int minLength, int maxLength, String propertyName, String messageKey) {
		if(value.length() < minLength || value.length() > maxLength)
			throwValidationException(propertyName, messageKey+":"+minLength+":"+maxLength+":"+value.length());
		return this;
	}	
	
	
	public ServerValidation notExactly(long value, long forbiddenValue, String propertyName) throws ValidationException {
		return notExactly(value, forbiddenValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"notExactly"); 
	}
	
	public ServerValidation notExactly(long value, long forbiddenValue, String propertyName, String messageKey) throws ValidationException {
		if(value == forbiddenValue)
			throwValidationException(propertyName, messageKey+":"+forbiddenValue);
		return this;
	}
	
	public ServerValidation notExactly(int value, int forbiddenValue, String propertyName) throws ValidationException {
		return notExactly(value, forbiddenValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"notExactly");
	}
	
	public ServerValidation notExactly(int value, int forbiddenValue, String propertyName, String messageKey) throws ValidationException {
		if(value == forbiddenValue)
			throwValidationException(propertyName, messageKey+":"+forbiddenValue);
		return this;
	}
	
	public ServerValidation notExactly(double value, double forbiddenValue, String propertyName) throws ValidationException {
		return notExactly(value, forbiddenValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"notExactly");
	}
	
	public ServerValidation notExactly(double value, double forbiddenValue, String propertyName, String messageKey) throws ValidationException {
		if(value == forbiddenValue)
			throwValidationException(propertyName, messageKey+":"+forbiddenValue);
		return this;
	}
	
	public ServerValidation notExactly(float value, float forbiddenValue, String propertyName) throws ValidationException {
		return notExactly(value, forbiddenValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"notExactly");
	}
	
	public ServerValidation notExactly(float value, float forbiddenValue, String propertyName, String messageKey) throws ValidationException {
		if(value == forbiddenValue)
			throwValidationException(propertyName, messageKey+":"+forbiddenValue);
		return this;
	}
	
	public ServerValidation exactly(long value, long targetValue, String propertyName) throws ValidationException {
		return exactly(value, targetValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"exactly");
	}
	
	public ServerValidation exactly(long value, long targetValue, String propertyName, String messageKey) throws ValidationException {
		if(value != targetValue)
			throwValidationException(propertyName, messageKey+":"+targetValue);
		return this;
	}
	
	public ServerValidation exactly(int value, int targetValue, String propertyName) throws ValidationException {
		return exactly(value, targetValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"exactly");
	}
	
	public ServerValidation exactly(int value, int targetValue, String propertyName, String messageKey) throws ValidationException {
		if(value != targetValue)
			throwValidationException(propertyName, messageKey+":"+targetValue);
		return this;
	}
	
	public ServerValidation exactly(double value, double targetValue, String propertyName) throws ValidationException {
		return exactly(value, targetValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"exactly");
	}
	
	public ServerValidation exactly(double value, double targetValue, String propertyName, String messageKey) throws ValidationException {
		if(value != targetValue)
			throwValidationException(propertyName, messageKey+":"+targetValue);
		return this;
	}
	
	public ServerValidation exactly(float value, float targetValue, String propertyName) throws ValidationException {
		return exactly(value, targetValue, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"exactly");
	}
	
	public ServerValidation exactly(float value, float targetValue, String propertyName, String messageKey) throws ValidationException {
		if(value != targetValue)
			throwValidationException(propertyName, messageKey+":"+targetValue);
		return this;
	}
	
	public <T> ServerValidation contains(Collection<T> collection, T element, String propertyName) {
		return contains(collection, element, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"contains" );
	}
	
	public <T> ServerValidation contains(Collection<T> collection, T element, String propertyName, String messageKey) {
		if(!collection.contains(element))
			throwValidationException(propertyName, messageKey+":"+element);
		return this;
	}
	
	public <T> ServerValidation lacks(Collection<T> collection, T element, String propertyName) {
		return lacks(collection, element, propertyName, ValidationProcessor.SERVER_SIDE_STANDARD_PREFIX+"lacks" );
	}
	
	public <T> ServerValidation lacks(Collection<T> collection, T element, String propertyName, String messageKey) {
		if(collection.contains(element))
			throwValidationException(propertyName, messageKey+":"+element);
		return this;
	}
	
		
	private void throwValidationException(String propertyName, String message) throws ValidationException {
		InvalidValueSerializable iv = new InvalidValueSerializable();
		iv.setMessage(message);
		iv.setPropertyName(propertyName);
		if(this.throwOnFirstError) {
			ValidationException ex = new ValidationException();
			ex.getInvalidValues().add(iv);
			throw ex;
		}else {
			invalidPool.add(iv);
		}
	}
	
	/**
	 * If you are in 'fail late mode' aka <code>throwOnFirstError=false</code>
	 * then you need to call this method at the end of your validations to 
	 * get the result of the previous validations. Meaning it will throw
	 * an validation exception if one or more of the previous validations failed.
	 * If none failed, this method will do nothing. Also if the <code>ServerValidation</code>
	 * instance was not configured for 'fail late mode' (the default) it will do nothing.
	 * @return
	 * @throws ValidationException
	 */
	public ServerValidation validate() throws ValidationException {
		if(this.throwOnFirstError)
			return this;
		
		if(!invalidPool.isEmpty()) {
			ValidationException ex = new ValidationException();
			for(InvalidValueSerializable iv : invalidPool)
				ex.getInvalidValues().add(iv);
			throw ex;
		}
		return this;
	}
	
	/**
	 * This method is used to immediately and always throw an ValidationException.
	 * This can be used when custom validations failed. It will ignore the
	 * <code>throwOnFirstError</code> switch.
	 *  
	 * @param msgKey The key for the custom message
	 * @param propertyName The property name of the field the validation error belongs to
	 * @throws ValidationException The exception is always thrown by this method (thats its use ;>) 
	 */
	public static void exception(String msgKey, String propertyName, Object... params) throws ValidationException {
		InvalidValueSerializable iv = new InvalidValueSerializable();
		
		if(params != null && params.length > 0) {
			
			//merge parameters into message key
			for(Object o: params) {
				msgKey += ":"+o.toString();
			}
		}
		
		iv.setMessage(msgKey);
		iv.setPropertyName(propertyName);
		ValidationException ex = new ValidationException();
		ex.getInvalidValues().add(iv);
		throw ex;
	}
	
	/**
	 * This method adds a new error and quits without throwing a <code>ValidationException</code>
	 * when <code>throwOnFirstError</code> is <code>false</code>. If it is <code>true</code> it
	 * will behave like the <code>exception</code> method.
	 * 
	 * @param msgKey
	 * @param propertyName
	 * @param params
	 * @throws ValidationException
	 */
	public ServerValidation addException(String msgKey, String propertyName, Object... params) throws ValidationException {
		
		if(throwOnFirstError)
			exception(msgKey, propertyName, params);
		else {
			msgKey = addParamsToMessageKey(msgKey, params);
			InvalidValueSerializable iv = new InvalidValueSerializable();
			iv.setMessage(msgKey);
			iv.setPropertyName(propertyName);
			invalidPool.add(iv);
		}
		return this;
			
	}
	
	private String addParamsToMessageKey(String msgKey, Object... params) {
		String result = msgKey;
		if(params != null && params.length > 0) {
			
			//merge parameters into message key
			for(Object o: params) {
				result += ":"+o.toString();
			}
		}
		return result;
	}
	
	
	/**
	 * This method serializes an <code>ValidationException</code> into a string.
	 * 
	 * The generated String can be passed to the <code>ValidationProcessor</code>
	 * 
	 * @param ex
	 * @return
	 */
	public static String serializeValidationException(ValidationException ex) {
		
		StringBuffer serializedException = new StringBuffer("VE_SERIALIZE_");
		
		serializedException.append(ex.getInvalidValues().size());
		
		for(InvalidValueSerializable iv : ex.getInvalidValues()) {
			serializedException.append(":");
			serializedException.append("_[");
			serializedException.append(iv.getPropertyName());
			serializedException.append("_,_");
			serializedException.append(iv.getMessage());
			serializedException.append("]_");
		}
		
		return serializedException.toString();
		
		
	}

	
	
}
