package eu.maydu.gwt.validation.client.validators.datetime.algorithms;

import eu.maydu.gwt.validation.client.validators.ValidatorAlgorithm;
import eu.maydu.gwt.validation.client.validators.ValidatorAlgorithmResult;


/**
 * This validator algorithm validates that a given input is a time of day.
 * 
 * Currently it only validates the european 24 hour format with
 * seconds optionally (This is configurable).
 * 
 * 
 * @author Anatol Mayen
 *
 */
public class TimeValidatorAlgorithm implements ValidatorAlgorithm<String> {

	private boolean secondsAllowed;
	private boolean secondsRequired;
	private String withoutSecondsPattern = "[0-2]?[0-9]:[0-5][0-9]";
	private String withSecondsPattern = "[0-2]?[0-9]:[0-5][0-9]:[0-5][0-9]";
	
	public static final int NOT_A_VALID_TIME_WITH_SECONDS_REQUIRED = 1;
	public static final int NOT_A_VALID_TIME_WITH_SECONDS_OPTIONALLY = 2;
	public static final int NOT_A_VALID_TIME_WITHOUT_SECONDS = 4;
	
	public TimeValidatorAlgorithm() {
		this(false, false);
	}
	
	public TimeValidatorAlgorithm(boolean secondsAllowed, boolean secondsRequired) {
		this.secondsAllowed = secondsAllowed;
		this.secondsRequired = secondsRequired;
	}
	
	
	public ValidatorAlgorithmResult validate(String input) {
		
		if(!secondsRequired) {
			boolean matches = input.matches(withoutSecondsPattern);
			if(!matches) {
				if(secondsAllowed) {
					matches = input.matches(withSecondsPattern);
					if(!matches)
						return new ValidatorAlgorithmResult(NOT_A_VALID_TIME_WITH_SECONDS_OPTIONALLY, input);
					else
						return null;
				}
				return new ValidatorAlgorithmResult(NOT_A_VALID_TIME_WITHOUT_SECONDS, input);
				
			}
			return boundsCheck(input);
		}

		if(secondsRequired) {
			boolean matches = input.matches(withSecondsPattern);
			if(!matches) {
				return new ValidatorAlgorithmResult(NOT_A_VALID_TIME_WITH_SECONDS_REQUIRED, input);
			}
		}
		
		return boundsCheck(input);
		
	}

	private ValidatorAlgorithmResult boundsCheck(String input) {
		String[] values = input.split(":");
		for(int i=0; i<values.length; i++) {
			int max = i==0?23:59;
			int value = Integer.valueOf(values[i]);
			if(value > max) {
				if(secondsRequired)
					return new ValidatorAlgorithmResult(NOT_A_VALID_TIME_WITH_SECONDS_REQUIRED, input);
				else if(secondsAllowed)
					return new ValidatorAlgorithmResult(NOT_A_VALID_TIME_WITH_SECONDS_OPTIONALLY, input);
				else return new ValidatorAlgorithmResult(NOT_A_VALID_TIME_WITHOUT_SECONDS, input);
			}
		}
		return null;
	}
}
