/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.dom4j;

import info.informatica.doc.style.css.CSSStyleSheetFactory;
import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.StyleDeclarationFactory;
import info.informatica.doc.style.css.dom.BaseCSSRule;
import info.informatica.doc.style.css.dom.BaseCSSStyleDeclaration;

import org.dom4j.Attribute;
import org.dom4j.Element;
import org.dom4j.QName;
import org.dom4j.dom.DOMDocumentFactory;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSRuleList;
import org.w3c.dom.css.CSSStyleRule;
import org.w3c.dom.css.CSSStyleSheet;
import org.w3c.dom.stylesheets.MediaList;

/**
 * DocumentFactory for CSS-styled XHTML documents.
 * <p>
 * This factory creates XHTMLDocuments and other objects with support for CSS 
 * style sheets.
 * A default style sheet can be set with the <code>setDefaultStyleSheet</code>
 * method. If no default sheet is specified, an internal default sheet will be 
 * used.
 * <p>
 * It is possible to produce elements with the ability to cache its own 
 * computed style, if you call <code>setStyleCache(true)</code>. This should 
 * enhance performance for applicactions that may call the <code>getComputedStyle()</code> 
 * method of a single stylable element several times.
 * 
 * @author Carlos Amengual
 * 
 */
public class XHTMLDocumentFactory extends DOMDocumentFactory {

	private static final long serialVersionUID = 3L;

	private DOM4JCSSStyleSheet defStyleSheet = null;
	
	private BaseCSSStyleDeclaration userImportantStyle = null;

	private BaseCSSStyleDeclaration userNormalStyle = null;

	private DOM4JCSSStyleSheet uaStyleSheet = null;

	private DOM4JCSSStyleSheetFactory cssFactory = new DOM4JCSSStyleSheetFactory();
	
	private boolean styleCacheOn = false;

	private StyleDatabase styleDb = null;

	protected static transient XHTMLDocumentFactory singleton = new XHTMLDocumentFactory();

	public static DOMDocumentFactory getInstance() {
		return singleton;
	}

	/**
	 * Gets the User Agent default CSS style sheet to be used by this factory.
	 * 
	 * @return the default style sheet.
	 */
	public DOM4JCSSStyleSheet getUserAgentStyleSheet() {
		if(uaStyleSheet == null){
			try {
				uaStyleSheet = (DOM4JCSSStyleSheet) getCSSStyleSheetFactory()
					.loadXHTMLDefaultSheet();
			} catch (Exception e) {
				throw new IllegalStateException("Unable to load User Agent style sheet.", e);
			}
		}
		return uaStyleSheet;
	}
	
	protected DOM4JCSSStyleSheet getDefaultStyleSheet() {
		if(defStyleSheet == null){
			mergeUserSheets();
		}
		return defStyleSheet;
	}

	/**
	 * Gets the CSS style sheet factory.
	 * 
	 * @return the CSS style sheet factory.
	 */
	public DOM4JCSSStyleSheetFactory getCSSStyleSheetFactory() {
		return cssFactory;
	}

	/**
	 * Sets the default CSS style sheet to be used in documents created
	 * by this factory.
	 * 
	 * @param styleSheet the default style sheet.
	 * @deprecated use <code>setUserAgentStyleSheet</code> and <code>setUserStyleSheet</code>.
	 */
	public void setDefaultStyleSheet(DOM4JCSSStyleSheet styleSheet) {
		this.defStyleSheet = styleSheet;
	}

	/**
	 * Sets the default User Agent CSS style sheet to be used in documents created
	 * by this factory.<p>
	 * The sheet will be appropriately merged with the non-important part of the 
	 * user-preference style sheet to provide the document's default sheet.</p>
	 * 
	 * @param styleSheet the user agent style sheet.
	 */
	public void setUserAgentStyleSheet(DOM4JCSSStyleSheet styleSheet) {
		this.uaStyleSheet = styleSheet;
		defStyleSheet = null;
	}

	/**
	 * Sets the CSS style sheet defined by the end-user.<p>
	 * The supplied sheet should contain user preferences, and will be 
	 * appropriately merged with the other style sheets.</p>
	 * 
	 * @param styleSheet the user style sheet.
	 */
	public void setUserStyleSheet(DOM4JCSSStyleSheet styleSheet) {
		if(styleSheet != null) {
			this.userImportantStyle = (BaseCSSStyleDeclaration) StyleDeclarationFactory.createCSSStyleDeclaration();
			this.userNormalStyle = (BaseCSSStyleDeclaration) StyleDeclarationFactory.createCSSStyleDeclaration();
			CSSRuleList rules = styleSheet.getCssRules();
			int rl = rules.getLength();
			for(int i=0; i<rl; i++) {
				CSSRule r = rules.item(i);
				if (r.getType() == CSSRule.STYLE_RULE) {
					BaseCSSStyleDeclaration st = (BaseCSSStyleDeclaration) ((CSSStyleRule)r).getStyle();
					st.prioritySplit(userImportantStyle, userNormalStyle);
				}
			}
		} else {
			this.userImportantStyle = null;
			this.userNormalStyle = null;
		}
	}
	
	protected BaseCSSStyleDeclaration getUserImportantStyle() {
		return userImportantStyle;
	}
	
	private void mergeUserSheets() {
		defStyleSheet = getUserAgentStyleSheet().clone();
		if(userNormalStyle != null) {
			defStyleSheet.addRule((BaseCSSRule) userNormalStyle.getParentRule());
		}
	}

	/**
	 * Indicates whether the stylable elements currently produced by this 
	 * factory are cache-enabled or not.
	 * 
	 * @return true if the per-element cache is enabled, false otherwise.
	 */
	public boolean isStyleCacheOn() {
		return styleCacheOn;
	}

	/**
	 * Can turn on or off the per-Element style caching capability (by 
	 * default is off).
	 * <p>
	 * Only applications that repeatedly call the 
	 * <code>CSSStylableElement.getComputedStyle()</code> method on the same 
	 * Element should turn it on.
	 * 
	 * @param onOff set to true to turn on the cache capability, to false 
	 * to turn it off.
	 */
	public void setStyleCache(boolean onOff) {
		this.styleCacheOn = onOff;
	}

	@Override
	public Element createElement(QName qname) {
		/*
		   Since CSS4J 0.9, for XHTML the root element for formatting is 'html'
		   (which should be the only CSSStylableElement at its level). 
		 */
		String name = qname.getName();
		if ("base".equals(name)) {
			return new BaseURLElement(qname);
		} else if ("style".equals(name)) {
			return new StyleElement(qname);
		} else if ("link".equals(name)) {
			return new LinkElement(qname);
		} else if ("head".equals(name)) {
			return new HeadElement(qname);
		} else if ("title".equals(name) || "meta".equals(name)) {
			return new XHTMLElement(qname);
		} else if ("font".equals(name) || "basefont".equals(name)) {
			return new FontElement(qname);
		} else if ("center".equals(name)) {
			return new CenterElement(qname);
		} else if ("u".equals(name)) {
			return new UnderlineElement(qname);
		} else if ("s".equals(name) || "strike".equals(name)) {
			return new StrikeElement(qname);
		} else if(styleCacheOn) {
			if ("td".equals(name) || "th".equals(name)) {
				return new CachedTableCellElement(qname);
			} else if("tr".equals(name)) {
				return new CachedTableRowElement(qname);
			} else if("table".equals(name)) {
				return new CachedTableElement(qname);
			} else {
				// Produce a cached element
				return new CachedCSSStylableElement(qname);
			}
		} else {
			if ("td".equals(name) || "th".equals(name)) {
				return new TableCellElement(qname);
			} else if("tr".equals(name)) {
				return new TableRowElement(qname);
			} else if("table".equals(name)) {
				return new TableElement(qname);
			} else {
				// Produce a regular stylable element
				return new CSSStylableElement(qname);
			}
		}
	}

	@Override
	public Element createElement(QName qname, int attributeCount) {
		String name = qname.getName();
		if ("base".equals(name)) {
			return new BaseURLElement(qname, attributeCount);
		} else if ("style".equals(name)) {
			return new StyleElement(qname, attributeCount);
		} else if ("link".equals(name)) {
			return new LinkElement(qname, attributeCount);
		} else if ("head".equals(name)) {
			return new HeadElement(qname, attributeCount);
		} else if ("title".equals(name) || "meta".equals(name)) {
			return new XHTMLElement(qname, attributeCount);
		} else if ("font".equals(name) || "basefont".equals(name)) {
			return new FontElement(qname, attributeCount);
		} else if ("center".equals(name)) {
			return new CenterElement(qname, attributeCount);
		} else if ("u".equals(name)) {
			return new UnderlineElement(qname, attributeCount);
		} else if ("s".equals(name) || "strike".equals(name)) {
			return new StrikeElement(qname, attributeCount);
		} else if(styleCacheOn) {
			if ("td".equals(name) || "th".equals(name)) {
				return new CachedTableCellElement(qname, attributeCount);
			} else if("tr".equals(name)) {
				return new CachedTableRowElement(qname, attributeCount);
			} else if("table".equals(name)) {
				return new CachedTableElement(qname, attributeCount);
			} else {
				// Produce a cached element
				return new CachedCSSStylableElement(qname, attributeCount);
			}
		} else {
			if ("td".equals(name) || "th".equals(name)) {
				return new TableCellElement(qname, attributeCount);
			} else if("tr".equals(name)) {
				return new TableRowElement(qname, attributeCount);
			} else if("table".equals(name)) {
				return new TableElement(qname, attributeCount);
			} else {
				// Produce a regular stylable element
				return new CSSStylableElement(qname, attributeCount);
			}
		}
	}

	@Override
	public XHTMLDocument createDocument() {
		XHTMLDocument mydoc = new XHTMLDocument();
		mydoc.setDocumentFactory(this);
		mydoc.setDefaultStyleSheet(getDefaultStyleSheet());
		mydoc.setUserImportantStyleDeclaration(userImportantStyle);
		if(styleDb != null){
			mydoc.setStyleDatabase(styleDb);
		}
		return mydoc;
	}

	@Override
	public Attribute createAttribute(Element owner, QName qname, String value) {
		if(owner instanceof LinkElement){
			return new DocumentStyleEventAttribute(qname, value);
		} else if(owner instanceof BaseURLElement && 
				qname.getName().equals("href")){
			return new BaseHrefAttribute(qname, value);
		} else if(owner instanceof CachedCSSStylableElement && 
				qname.getName().equals("style")){
			return new StyleAttribute(qname, value);
		} else {
			return super.createAttribute(owner, qname, value);
		}
	}

	public StyleDatabase getStyleDatabase() {
		return styleDb ;
	}

	public void setStyleDatabase(StyleDatabase styleDb) {
		this.styleDb = styleDb;
	}

	/**
	 * CSS2 style sheet factory for DOM.
	 * 
	 * @author Carlos Amengual (amengual at informatica.info)
	 * 
	 */
	public class DOM4JCSSStyleSheetFactory extends CSSStyleSheetFactory {

		DOM4JCSSStyleSheetFactory() {
			super();
		}

		/**
		 * Creates a CSS style sheet.
		 * <p>
		 * 
		 * @param namespaceUri
		 *            the Namespace URI for the style sheet.
		 * @param mediaList
		 *            the target media list for style information.
		 * @return the style sheet.
		 */
		public CSSStyleSheet createStyleSheet(String namespaceUri, 
				MediaList mediaList) {
			return createStyleSheet(null, namespaceUri, mediaList);
		}

		/**
		 * Creates a CSS style sheet.
		 * <p>
		 * 
		 * @param ownerElement
		 *            the Element that associates the style sheet to the document. In
		 *            XHTML it can be a <code>link</code> or <code>style</code>
		 *            element. For style sheets that are included by other style
		 *            sheets, the value of this attribute is <code>null</code>.
		 * @param namespaceUri
		 *            the Namespace URI for the style sheet.
		 * @param mediaList
		 *            the target media list for style information.
		 * @return the style sheet.
		 */
		public DOM4JCSSStyleSheet createStyleSheet(XHTMLElement ownerElement,
				String namespaceUri, MediaList mediaList) {
			if(mediaList == null) {
				throw new NullPointerException("Null media list");
			}
			DOM4JCSSStyleSheet css = new DOM4JCSSStyleSheet(this, 
					ownerElement, mediaList);
			if(ownerElement != null) {
				String title = ownerElement.attributeValue("title");
				css.setTitle(title);
			}
			css.setNamespaceURI(namespaceUri);
			return css;
		}

	}

}
