/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.pdf.itext;

import info.informatica.doc.RenderingException;
import info.informatica.doc.dom4j.DOM4JCSSStyleDeclaration;
import info.informatica.doc.dom4j.DocumentFormatter;
import info.informatica.doc.dom4j.XHTMLDocument;
import info.informatica.doc.style.css.CSSStyleException;
import info.informatica.doc.style.css.visual.CSSBox;
import info.informatica.doc.style.css.visual.CSSContainerBox;
import info.informatica.doc.style.css.visual.CSSInlineBox;
import info.informatica.doc.style.css.visual.CSSTableCellBox;
import info.informatica.doc.style.css.visual.CSSTableRowBox;
import info.informatica.doc.style.css.visual.ReplacedElementBox;
import info.informatica.doc.style.css.visual.box.ListItemBox;
import info.informatica.doc.style.css.visual.box.RunInBox;
import info.informatica.doc.style.css.visual.container.CSSBlockBoxContainer;
import info.informatica.doc.style.css.visual.container.CSSBoxContainer;
import info.informatica.doc.style.css.visual.container.CSSInlineBoxContainer;
import info.informatica.doc.style.css.visual.container.CSSTableRowContainer;

import java.io.FileOutputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;

import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Element;
import com.itextpdf.text.Image;
import com.itextpdf.text.ListItem;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.Phrase;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.TextElementArray;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;

public class PDFRenderer {

	private CSSContainerBox rootBox;

	protected Document pdf = null;
	
	private PdfWriter wri = null;
	
	private PDFStyleFormatter pdfFormatter;

	static Logger log = Logger.getLogger(PDFRenderer.class.getName());

	public PDFRenderer() {
		this(new Document(PageSize.A4, 36, 36, 40, 40));
	}

	public PDFRenderer(Document pdf) {
		super();
		this.pdf = pdf;
		pdfFormatter = new PDFStyleFormatter(new PDFUserAgent(
				new PDFStyleDatabase(pdf.getPageSize())));
	}

	public void setDocument(XHTMLDocument xdoc) throws CSSStyleException {
		rootBox = new DocumentFormatter(new PDFUserAgent()).formatDocument(xdoc);
	}

	protected CSSContainerBox getRootBox() {
		return rootBox;
	}

	protected PdfWriter getPdfWriter() {
		return wri;
	}

	public synchronized void open(OutputStream out) throws RenderingException {
		if (pdf.isOpen()) {
			throw new IllegalStateException("Report output already opened");
		}
		// Writer
		try {
			wri = PdfWriter.getInstance(pdf, out);
		} catch (DocumentException e) {
			throw new RenderingException("Could not open PDF report", e);
		}
		wri.setStrictImageSequence(true);
		// Retrieve the document
		XHTMLDocument xhtmlDoc = (XHTMLDocument) ((DOM4JCSSStyleDeclaration)getRootBox()
				.getComputedStyle()).getPeerNode().getOwnerDocument();
		// Add meta-info
		org.dom4j.Element headEl = xhtmlDoc.getRootElement().element("head");
		String title = headEl.element("title").getTextTrim();
		if (title != null) {
			log.debug("Setting title: " + title);
			pdf.addTitle(title);
		}
		pdf.addCreationDate();
		pdf.addCreator(getClass().getPackage().toString());
		Iterator<org.dom4j.Element> it = headEl.elementIterator("meta");
		while(it.hasNext()) {
			org.dom4j.Element metaEl = it.next();
			String keyw = metaEl.attributeValue("http-equiv");
			if(keyw.equals("keywords")){
				keyw = metaEl.attributeValue("content");
				if (keyw != null && !keyw.equals("")) {
					pdf.addKeywords(keyw);
				}
			} else if(keyw.equals("author")){
				keyw = metaEl.attributeValue("content");
				if (keyw != null && !keyw.equals("")) {
					pdf.addAuthor(keyw);
				}
			} else if(keyw.equals("description")){
				keyw = metaEl.attributeValue("content");
				if (keyw != null && !keyw.equals("")) {
					pdf.addSubject(keyw);
				}
			}
		}
		// open the document
		pdf.open();
	}

	public synchronized void close() {
		log.debug("Closing the document.");
		pdf.close();
		pdf = null;
	}

	public PDFStyleFormatter getStyleFormatter() {
		return pdfFormatter;
	}

	public void print() throws RenderingException {
		renderStaticBlockBox(pdf, rootBox);
	}

	protected void renderStaticBlockBox(Object nativeContainer, CSSContainerBox box) {
		CSSBoxContainer container = box.asContainerBox();
		if(container instanceof CSSBlockBoxContainer){
			List<CSSContainerBox> blockBoxes = ((CSSBlockBoxContainer)container)
				.getStaticallyPositioned();
			Iterator<CSSContainerBox> it = blockBoxes.iterator();
			while(it.hasNext()){
				CSSContainerBox bbox = it.next();
				System.err.println("Rendering box: " + bbox.getComputedStyle().getPeerXPath());
				if(bbox instanceof ReplacedElementBox) {
					// TODO
					//addNativeObject(nativeContainer, elem);
				} else {
					Element elem = nativePeer(bbox);
					if(elem == null) {
						if(nativeContainer instanceof Element) {
							elem = (Element)nativeContainer;
						} else {
							renderStaticBlockBox(nativeContainer, bbox);
							continue;
						}
					}
					renderStaticBlockBox(elem, bbox);
					addNativeObject(nativeContainer, elem);
				}
			}
		} else if(container instanceof CSSInlineBoxContainer) {
			List<CSSBox> inBoxes = ((CSSInlineBoxContainer)container).getInlineBoxes();
			Iterator<CSSBox> it = inBoxes.iterator();
			while(it.hasNext()){
				CSSBox b = it.next();
				CSSInlineBox inBox;
				if(b instanceof CSSInlineBox) {
					inBox = (CSSInlineBox) b;
				} else {
					// Must be a run-in box
					b = ((RunInBox)b).finalBox();
					if(b instanceof CSSInlineBox) {
						inBox = (CSSInlineBox) b;
					} else {
						// FIXME
						continue;
					}
				}
				addNativeObject(nativeContainer, 
						renderInlineBox(inBox));
			}
		}
		format(nativeContainer, box);
	}

	protected Phrase renderInlineBox(CSSInlineBox box) {
		System.err.println("Rendering inline box: " + box.getComputedStyle().getPeerXPath());
		Chunk chunk = pdfFormatter.createChunk(box);
		Phrase phrase = pdfFormatter.createPhrase(box);
		phrase.add(chunk);
		return phrase;
	}

	protected void addNativeObject(Object nativeContainer, Element child) {
		if(nativeContainer instanceof TextElementArray) {
			((TextElementArray)nativeContainer).add(child);
		} else if(nativeContainer instanceof PdfPCell) {
			((PdfPCell)nativeContainer).addElement(child);
		} else if(nativeContainer instanceof Document) {
			try {
				((Document)nativeContainer).add(child);
			} catch (DocumentException e) {
				throw new IllegalStateException(e);
			}
		} else if(nativeContainer != null) {
			System.err.println("Lost child: " + child.getClass().getName()
					+ " for parent " + nativeContainer.getClass().getName());
		}
	}

	protected com.itextpdf.text.Element nativePeer(CSSContainerBox box) {
		CSSBoxContainer container = box.asContainerBox();
		Element wrapped = null;
		if(box instanceof CSSTableCellBox) {
			wrapped = new PdfPCell();
		} else if(container instanceof CSSInlineBoxContainer){
			wrapped = new Paragraph();
		} else if(box instanceof ListItemBox){
			wrapped = new ListItem();
		} else if(container instanceof CSSBlockBoxContainer){
			List<CSSContainerBox> childBlocks = ((CSSBlockBoxContainer)
					container).getStaticallyPositioned();
			if(!childBlocks.isEmpty()) {
				CSSContainerBox firstChild = childBlocks.get(0);
				if(firstChild instanceof ListItemBox) {
					wrapped = new com.itextpdf.text.List();
				}
			}
		} else if(container instanceof CSSTableRowContainer){
			List<CSSTableRowBox> rows = ((CSSTableRowContainer)container).getRows();
			if(!rows.isEmpty()) {
				int columns = rows.get(0).getColumnCount();
				wrapped = new PdfPTable(columns);
			}
		}
		return wrapped;
	}

	protected Object nativePeer(ReplacedElementBox<Image> box) {
		Object peer = null;
		if(box instanceof PDFImgReplacedBox) {
			peer = ((PDFImgReplacedBox)box).getNativeContent();
		} else if(box instanceof PDFInputTextReplacedBox) {
			peer = ((PDFInputTextReplacedBox)box).createPDFField(getPdfWriter());
		}
		return peer;
	}

	protected void format(Object o, CSSContainerBox box) {
		if(o instanceof Rectangle) {
			pdfFormatter.formatRectangle(box, (Rectangle)o);
		}
		if(o instanceof Paragraph) {
			pdfFormatter.formatParagraph(box, (Paragraph)o);
			if(o instanceof ListItem) {
				pdfFormatter.formatListItem(box, (ListItem)o);
			}
		} else if(o instanceof PdfPCell) {
			pdfFormatter.formatPCell((CSSTableCellBox) box, (PdfPCell)o);
		} else if(o instanceof PdfPTable) {
			pdfFormatter.formatPTable(box, (PdfPTable)o);
		} else if(o instanceof com.itextpdf.text.List) {
			pdfFormatter.formatList(box, (com.itextpdf.text.List)o);
		} else if(o instanceof Document) {
			pdfFormatter.formatDocument(box, (Document)o);
		}
	}

	public static void main(String[] args) throws Exception {
		//String urlString = "http://todo.com/";
		String urlString = "http://informatica.info/index";
		URL url = new URL(urlString);
		PDFUserAgent agent = new PDFUserAgent(); 
		XHTMLDocument xdoc = agent.readURL(url);
		PDFRenderer renderer = new PDFRenderer();
		renderer.setDocument(xdoc);
		String fileName = "c:/tmp/test.pdf";
		FileOutputStream out = new FileOutputStream(fileName);
		renderer.open(out);
		renderer.print();
		renderer.close();
		out.close();
	}

}
