/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css;

import info.informatica.doc.style.css.dom.BaseCSSStyleSheet;
import info.informatica.doc.style.css.dom.DOMMediaList;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;

import org.apache.log4j.Logger;
import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSStyleSheet;
import org.w3c.dom.stylesheets.MediaList;

/**
 * CSS2 style sheet factory.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
abstract public class CSSStyleSheetFactory {
	private ClassLoader classLoader = null;

	public final static String XHTML_NAMESPACE_URI = "http://www.w3.org/1999/xhtml";

	static Logger log = Logger.getLogger(CSSStyleSheetFactory.class.getName());

	public CSSStyleSheetFactory() {
		super();
	}

	/**
	 * Creates a CSS style sheet in XHTML namespace for all media.
	 * <p>
	 * 
	 * @return the style sheet.
	 */
	public CSSStyleSheet createStyleSheet() {
		MediaList mediaList = new DOMMediaList();
		return createStyleSheet(XHTML_NAMESPACE_URI, mediaList);
	}

	/**
	 * Creates a CSS style sheet for a target media.
	 * <p>
	 * 
	 * @param namespaceUri
	 *            the Namespace URI for the style sheet.
	 * @param media
	 *            the target media for style information.
	 * @return the style sheet.
	 */
	public CSSStyleSheet createStyleSheet(String namespaceUri, 
			String media) {
		if(media == null) {
			throw new NullPointerException("Null media list");
		}
		return createStyleSheet(namespaceUri, new DOMMediaList(media));
	}

	/**
	 * Creates a CSS style sheet.
	 * <p>
	 * 
	 * @param namespaceUri
	 *            the Namespace URI for the style sheet.
	 * @param mediaList
	 *            the target media list for the style information.
	 * @return the style sheet.
	 */
	abstract public CSSStyleSheet createStyleSheet(String namespaceUri, 
			MediaList mediaList);

	/**
	 * Parses a style sheet from a source.
	 * <p>
	 * If the given style sheet Object is not empty, the rules from 
	 * the parsed source will be added at the end of the rule list.
	 * 
	 * @param sheet 
	 *            the style sheet to be filled from the source.
	 * @param source
	 *            the SAC input source.
	 * @throws DOMException if a DOM problem is found parsing the sheet.
	 * @throws CSSException if a non-DOM problem is found parsing the sheet.
	 * @throws IOException if a problem is found reading the sheet.
	 */
	public static void parse(CSSStyleSheet sheet, InputSource source) 
	throws DOMException, IOException {
		((BaseCSSStyleSheet)sheet).parseCSSStyleSheet(source);
	}

	/**
	 * Sets the class loader to be used when loading internal style sheets.
	 * 
	 * @param loader
	 *            the class loader.
	 */
	public void setClassLoader(ClassLoader loader) {
		classLoader = loader;
	}

	InputStream readXHTMLDefaultSheet() {
		return loadCSSfromClasspath("html.css");
	}

	/**
	 * Loads a default CSS style sheet for (X)HTML documents.
	 * 
	 * @return the default style sheet for XHTML.
	 * @throws IOException if a problem arose when reading the 
	 * default sheet.
	 */
	public CSSStyleSheet loadXHTMLDefaultSheet() throws IOException {
		BaseCSSStyleSheet cssSheet;
		Reader re = null;
		try {
			re = new InputStreamReader(loadCSSfromClasspath("html.css"));
			InputSource source = new InputSource(re);
			cssSheet = (BaseCSSStyleSheet) createStyleSheet();
			cssSheet.parseCSSStyleSheet(source);
		} catch(IOException e) {
			throw e;
		} finally {
			try {
				re.close();
			}catch(IOException e) {}
		}
		return cssSheet;
	}

	protected InputStream loadCSSfromClasspath(final String cssFilename) {
		return java.security.AccessController
				.doPrivileged(new java.security.PrivilegedAction<InputStream>() {
					public InputStream run() {
						if (classLoader != null) {
							return classLoader
								.getResourceAsStream(resourcePath(
										CSSStyleSheetFactory.class.getPackage().getName(), 
										cssFilename));
						} else {
							return this.getClass()
								.getResourceAsStream(cssFilename);
						}
					}
				});
	}

	private static String resourcePath(String clase, String filename) {
		return '/' + clase.replace('.', '/')
			+ '/' + filename;
	}

}
