/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSStyleSheet;

/**
 * Abstract class to be inherited by all CSS rules.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
abstract public class BaseCSSRule implements CSSRule {

	/**
	 * The parent style sheet.
	 */
	private CSSStyleSheet parentSheet = null;

	/**
	 * The parent rule.
	 */
	private CSSRule parentRule = null;

	private final short ruleType;

	/**
	 * Contains the index at which this rule was inserted in parent sheet.
	 */
	protected int insertionIndex = 0;

	public BaseCSSRule(CSSStyleSheet parentSheet, short type) {
		super();
		this.parentSheet = parentSheet;
		ruleType = type;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.w3c.dom.css.CSSRule#getType()
	 */
	public short getType() {
		return ruleType;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.w3c.dom.css.CSSRule#getCssText()
	 */
	abstract public String getCssText();

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.w3c.dom.css.CSSRule#setCssText(java.lang.String)
	 */
	abstract public void setCssText(String cssText) throws DOMException;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.w3c.dom.css.CSSRule#getParentStyleSheet()
	 */
	public CSSStyleSheet getParentStyleSheet() {
		return parentSheet;
	}
	
	void setParentStyleSheet(CSSStyleSheet parentSheet) {
		this.parentSheet = parentSheet;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.w3c.dom.css.CSSRule#getParentRule()
	 */
	public CSSRule getParentRule() {
		return parentRule;
	}

	/**
	 * Sets the parent CSS rule, in case this rule is contained by another.
	 * 
	 * @param parent
	 *            the parent rule.
	 */
	void setParentRule(CSSRule parent) {
		parentRule = parent;
	}

	/**
	 * Factory method that creates a CSS rule according to given CSS type.
	 * 
	 * @param parentSheet
	 *            the parent CSS sheet of the rule.
	 * @param type
	 *            the type of rule according to the values defined in
	 *            <code>org.w3c.dom.css.CSSRule</code>.
	 * @return the rule.
	 */
	public static CSSRule createCSSRule(CSSStyleSheet parentSheet, int type) {
		CSSRule rule = null;
		switch (type) {
		case CSSRule.STYLE_RULE:
			rule = new DOMCSSStyleRule(parentSheet);
		case CSSRule.FONT_FACE_RULE:
			rule = new DOMCSSFontFaceRule(parentSheet);
		case CSSRule.IMPORT_RULE:
			rule = new DOMCSSImportRule(parentSheet);
		case CSSRule.MEDIA_RULE:
			rule = new DOMCSSMediaRule(parentSheet);
		case CSSRule.PAGE_RULE:
			rule = new DOMCSSPageRule(parentSheet);
		}
		return rule;
	}

	/**
	 * Sets the index at which this rule was inserted into its container,
	 * generally the parent sheet.
	 * 
	 * @param index
	 *            the index.
	 */
	void setInsertionIndex(int index) {
		insertionIndex = index;
	}
}
