/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import info.informatica.doc.style.css.AbstractStyleDatabase;
import info.informatica.doc.style.css.SACParserFactory;
import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.StyleDeclarationFactory;
import info.informatica.doc.style.css.property.AbstractCSSPrimitiveValue;
import info.informatica.doc.style.css.property.AbstractCSSValue;
import info.informatica.doc.style.css.property.CSSNumberValue;
import info.informatica.doc.style.css.property.CSSShorthandValue;
import info.informatica.doc.style.css.property.PropertyDatabase;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;
import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.Parser;
import org.w3c.css.sac.SelectorList;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSS2Properties;
import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSStyleDeclaration;
import org.w3c.dom.css.CSSValue;

/**
 * CSS Style Declaration.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class BaseCSSStyleDeclaration implements CSSStyleDeclaration,
		LexicalPropertyListener, Cloneable {

	/**
	 * The rule that contains this declaration block, if any.
	 */
	private CSSStyleDeclarationRule parentRule;

	private HashMap<String, CSSValue> propValue = null;

	private ArrayList<String> propertyList = null;

	private ArrayList<String> priorities = null;

	private StyleDeclarationDocumentHandler styleDeclarationDocumentHandler = null;

	private CSS2Properties css2properties = new CSS2PropertiesImpl();

	private StyleDatabase styleDb = null;

	static Logger log = Logger.getLogger(BaseCSSStyleDeclaration.class
			.getName());

	/**
	 * Constructor with parent CSS rule argument.
	 * 
	 * @param parentRule
	 *            the parent CSS rule.
	 */
	BaseCSSStyleDeclaration(CSSStyleDeclarationRule parentRule) {
		super();
		this.parentRule = parentRule;
		propValue = new HashMap<String, CSSValue>();
		propertyList = new ArrayList<String>();
		priorities = new ArrayList<String>();
	}

	public BaseCSSStyleDeclaration() {
		super();
		this.parentRule = null;
		propValue = new HashMap<String, CSSValue>();
		propertyList = new ArrayList<String>();
		priorities = new ArrayList<String>();
	}

	/**
	 * Copy constructor.
	 * 
	 * @param copiedObject
	 *            the DOMCSSStyleDeclaration to be copied.
	 */
	@SuppressWarnings("unchecked")
	protected BaseCSSStyleDeclaration(BaseCSSStyleDeclaration copiedObject) {
		super();
		this.parentRule = (CSSStyleDeclarationRule) copiedObject
				.getParentRule();
		priorities = (ArrayList<String>) copiedObject.priorities.clone();
		propertyList = (ArrayList<String>) copiedObject.propertyList.clone();
		propValue = (HashMap<String, CSSValue>) copiedObject.propValue.clone();
	}

    /**
     * Retrieves the parseable textual representation of the declaration block 
     * (excluding the surrounding curly braces).
     * 
     * @return the textual representation of the declaration block.
     * @exception DOMException
     *   SYNTAX_ERR: Raised if the specified CSS string value has a syntax 
     *   error and is unparsable.
     *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this declaration is 
     *   readonly or a property is readonly.
     */
	public String getCssText() {
		int sz = propertyList.size();
		StringBuilder sb = new StringBuilder(50 + sz * 10);
		for (int i = 0; i < sz; i++) {
			String ptyname = propertyList.get(i);
			CSSValue cssVal = propValue.get(ptyname);
			// Verify if the property is a subproperty of a previously set
			// shorthand
			if(cssVal instanceof AbstractCSSPrimitiveValue) {
				if(((AbstractCSSPrimitiveValue)cssVal).isSubproperty()){
					continue;
				}
			}
			// No subproperty of already printed shorthand, print it.
			String prio = priorities.get(i);
			sb.append(ptyname).append(':').append(' ').append(
					propValue.get(ptyname).getCssText());
			if (prio != null && "important".equals(prio)) {
				sb.append(" ! important");
			}
			sb.append(';').append(' ');
		}
		return sb.toString();
	}

    /**
     * Parses the given value and resets all the properties in the declaration
     * block, including the removal or addition of properties.
     * 
     * @param cssText the text with the style declaration.
     * @throws DOMException
     *   SYNTAX_ERR: Raised if the specified CSS string value has a syntax 
     *   error and is unparsable.
     *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this declaration is 
     *   readonly or a property is readonly.
     *   <br>NOT_SUPPORTED_ERR: if the system was unable to instantiate 
     *   parser.
     */
	public void setCssText(String cssText) throws DOMException {
		// XXX The following may cause a DOMException.NOT_SUPPORTED_ERR
		// (documented above) but the W3C API does not allow for that.
		Parser parser = SACParserFactory.createSACParser();
		InputSource source = new InputSource();
		Reader re = new StringReader(cssText);
		source.setCharacterStream(re);
		clear();
		StyleDeclarationDocumentHandler handler = getStyleDeclarationDocumentHandler();
		handler.setLexicalRuleListener(this);
		parser.setDocumentHandler(handler);
		try {
			parser.parseStyleDeclaration(source);
		} catch (CSSException e) {
			throw new DOMException(DOMException.SYNTAX_ERR, e
					.getMessage());
		} catch (IOException e) {
			// This should never happen!
			throw new DOMException(DOMException.SYNTAX_ERR, e
					.getMessage());
		}
	}

    /**
     *  Used to retrieve the value of a CSS property if it has been explicitly 
     * set within this declaration block. 
     * @param propertyName The name of the CSS property. See the CSS property 
     *   index. 
     * @return  Returns the value of the property if it has been explicitly 
     *   set for this declaration block. Returns the empty string if the 
     *   property has not been set. 
     */
	public String getPropertyValue(String propertyName) {
		CSSValue value = getCSSValue(propertyName);
		if (value != null) {
			if(value instanceof CSSPrimitiveValue && 
					((CSSPrimitiveValue)value).getPrimitiveType() == CSSPrimitiveValue.CSS_STRING) {
				return ((CSSPrimitiveValue)value).getStringValue();
			} else {
				return value.getCssText();
			}
		} else {
			return "";
		}
	}

    /**
     *  Used to retrieve the object representation of the value of a CSS 
     * property if it has been explicitly set within this declaration block. 
     * This method returns <code>null</code> if the property is a shorthand 
     * property. Shorthand property values can only be accessed and modified 
     * as strings, using the <code>getPropertyValue</code> and 
     * <code>setProperty</code> methods. 
     * @param propertyName The name of the CSS property. See the CSS property 
     *   index. 
     * @return  Returns the value of the property if it has been explicitly 
     *   set for this declaration block. Returns <code>null</code> if the 
     *   property has not been set. 
     */
	public CSSValue getPropertyCSSValue(String propertyName) {
		if (PropertyDatabase.getInstance().isShorthand(propertyName)) {
			return null;
		} else {
			return getCSSValue(propertyName);
		}
	}

	protected CSSValue getCSSValue(String propertyName) {
		return getDeclaredCSSValue(propertyName);
	}
	
	protected CSSValue getDeclaredCSSValue(String propertyName) {
		return propValue.get(propertyName);
	}

	/**
	 * Used to remove a CSS property if it has been explicitly set 
	 * within this declaration block.
	 * 
	 * @param propertyName name of the property to remove.
     * @return  Returns the value of the property if it has been explicitly 
     *   set for this declaration block. Returns the empty string if the 
     *   property has not been set or the property name does not correspond 
     *   to a known CSS property. 
	 * @throws DOMException if this declaration is readonly or the property 
	 * is readonly. This implementation is not expected to throw the
	 * Exception.
	 */
	public synchronized String removeProperty(String propertyName)
			throws DOMException {
		int idx = propertyList.indexOf(propertyName);
		if (idx >= 0) {
			propertyList.remove(idx);
			priorities.remove(idx);
			return propValue.remove(propertyName).getCssText();
		} else {
			return "";
		}
	}

    /**
     *  Used to retrieve the priority of a CSS property (e.g. the 
     * <code>"important"</code> qualifier) if the property has been 
     * explicitly set in this declaration block. 
     * @param propertyName The name of the CSS property. See the CSS property 
     *   index. 
     * @return  A string representing the priority (e.g. 
     *   <code>"important"</code>) if one exists. The empty string if none 
     *   exists.
     */
	public String getPropertyPriority(String propertyName) {
		int idx = propertyList.indexOf(propertyName);
		if(idx < 0){
			return "";
		}
		String prio = priorities.get(idx);
		if (prio != null) {
			return prio;
		} else {
			return "";
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see info.informatica.doc.style.css.dom.LexicalPropertyListener#setProperty(java.lang.String,
	 *      org.w3c.css.sac.LexicalUnit, java.lang.String)
	 */
	public void setProperty(String propertyName,
			LexicalUnit value, String priority) throws DOMException {
		if(priority != null) {
			priority = priority.intern();
		}
		// Check for shorthand properties
		if (PropertyDatabase.getInstance().isShorthand(propertyName)) {
			CSSShorthandValue shVal = new CSSShorthandValue();
			setProperty(propertyName, shVal, priority);
			String cssText = setSubproperties(propertyName, value, priority);
			shVal.setShorthandText(cssText);
		} else {
			CSSValue cssvalue = AbstractCSSValue.createCSSValue(value);
			setProperty(propertyName, cssvalue, priority);
		}
	}

    /**
     *  Used to set a property value and priority within this declaration 
     * block. 
     * @param propertyName The name of the CSS property. See the CSS property 
     *   index. 
     * @param value The new value of the property. 
     * @param priority The new priority of the property (e.g. 
     *   <code>"important"</code>).  
     * @throws DOMException
     *   SYNTAX_ERR: Raised if the specified value has a syntax error and is 
     *   unparsable.
     *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this declaration is 
     *   readonly or the property is readonly.
     */
	public void setProperty(String propertyName, String value, String priority)
			throws DOMException {
		Parser parser;
		try {
			parser = SACParserFactory.createSACParser();
		} catch (DOMException e) {
			// Could not create parser.
			throw new DOMException(DOMException.NO_MODIFICATION_ALLOWED_ERR, e
					.getMessage());
		}
		InputSource source = new InputSource();
		Reader re = new StringReader(value);
		source.setCharacterStream(re);
		LexicalUnit lunit = null;
		try {
			lunit = parser.parsePropertyValue(source);
		} catch (CSSException e) {
			throw new DOMException(DOMException.SYNTAX_ERR, e
					.getMessage());
		} catch (IOException e) {
			// This should never happen!
			throw new DOMException(DOMException.INVALID_STATE_ERR, e
					.getMessage());
		}
		setProperty(propertyName, lunit, priority);
	}

	void setProperty(String propertyName, CSSValue cssValue,
			String priority) {
		propertyName = propertyName.intern();
		if (!propertyList.contains(propertyName)) {
			addProperty(propertyName, cssValue, priority);
		} else {
			int idx = propertyList.indexOf(propertyName);
			if(!"important".equals(priorities.get(idx)) || 
					"important".equals(priority)) {
				propertyList.remove(idx);
				priorities.remove(idx);
				addProperty(propertyName, cssValue, priority);
			}
		}
	}
	
	private void addProperty(String propertyName, CSSValue cssValue,
			String priority) {
		propertyList.add(propertyName);
		priorities.add(priority);
		propValue.put(propertyName, cssValue);
	}

    /**
     *  The number of properties that have been explicitly set in this 
     * declaration block. The range of valid indices is 0 to length-1 
     * inclusive. 
     */
	public int getLength() {
		return propertyList.size();
	}

	/**
	 * Used to retrieve the properties that have been set in this declaration 
	 * block. The order of the properties retrieved using this method does not 
	 * have to be the order in which they were set. 
	 * This method can be used to iterate over all properties in this 
	 * declaration block.
	 * 
	 * @param index the property index.
	 * @return the name of the property at this ordinal position, or the 
	 * empty string if no property exists at this position.
	 */
	public String item(int index) {
		if(index < 0 || index > propertyList.size()) {
			return "";
		} else {
			return propertyList.get(index);
		}
	}
	
	void clear() {
		propValue.clear();
		propertyList.clear();
		priorities.clear();
	}

    /**
     * Retrieves the CSS rule that contains this declaration block.
     * 
     * @return the CSS rule that contains this declaration block or <code>null</code> 
     * if this <code>CSSStyleDeclaration</code> is not attached to a 
     * <code>CSSRule</code>. 
     */
	public CSSRule getParentRule() {
		return parentRule;
	}

	/**
	 * Add all the rules of the given style declaration to this one.
	 * 
	 * @param style
	 *            the style declaration whose rules have to be added.
	 */
	public void addStyle(BaseCSSStyleDeclaration style) {
		Iterator<String> it = style.propertyList.iterator();
		int i = 0;
		while (it.hasNext()) {
			String propertyName = it.next();
			if (!propertyList.contains(propertyName)) {
				propertyList.add(propertyName);
				priorities.add(style.priorities.get(i));
			} else if (!"important".equals(priorities.get(i))) {
				priorities.set(i, style.priorities.get(i));
			} else {
				continue;
			}
			propValue.put(propertyName, style.propValue.get(propertyName));
		}
	}

	/**
	 * Splits this style declaration in two: one for important properties only, and the other 
	 * with normal properties.
	 * 
	 * @param importantDecl
	 * @param normalDecl
	 */
	public void prioritySplit(BaseCSSStyleDeclaration importantDecl, BaseCSSStyleDeclaration normalDecl) {
		int psz = propertyList.size();
		for(int i=0; i<psz; i++) {
			String propertyName = propertyList.get(i);
			String priority = priorities.get(i);
			if("important".equals(priority)) {
				importantDecl.addProperty(
						propertyName, propValue.get(propertyName), priority);
			} else {
				normalDecl.addProperty(
						propertyName, propValue.get(propertyName), priority);
			}
		}
	}

	/**
	 * Gets the style database which is used to compute the style.
	 * 
	 * @return the style database.
	 */
	public StyleDatabase getStyleDatabase() {
		return styleDb;
	}

	/**
	 * Sets the style database used to compute the style.
	 * 
	 * @param styleDb
	 *            the style database.
	 */
	public void setStyleDatabase(StyleDatabase styleDb) {
		this.styleDb = styleDb;
	}

	/**
	 * Computes the inital (default) value for the given property.
	 * 
	 * @param propertyName
	 *            the name of the property.
	 * @return the initial value for the property, or null if none was found.
	 */
	protected CSSValue defaultPropertyValue(String propertyName) {
		CSSValue defval = PropertyDatabase.getInstance().getInitialValue(
				propertyName);
		if (defval == null) {
			if (propertyName.equals("color")) {
				defval = getStyleDatabase().getInitialColor();
			} else if (propertyName.equals("font-family")) {
				defval = StyleDeclarationFactory
						.parseProperty(getStyleDatabase()
								.getDefaultGenericFontFamily());
			} else if (propertyName.equals("text-align")) {
				String directionValue = getPropertyValue("direction");
				if (directionValue.equals("rtl")) {
					defval = StyleDeclarationFactory.parseProperty("right");
				} else {
					defval = StyleDeclarationFactory.parseProperty("left");
				}
			} else if (propertyName.equals("border-top-color")) {
				defval = getColor();
			} else if (propertyName.equals("border-right-color")) {
				defval = getColor();
			} else if (propertyName.equals("border-bottom-color")) {
				defval = getColor();
			} else if (propertyName.equals("border-left-color")) {
				defval = getColor();
			} else if (propertyName.equals("quotes")) {
				defval = StyleDeclarationFactory.parseProperty("\" \"");
			}
		}
		return defval;
	}

	public CSSPrimitiveValue getColor() {
		CSSPrimitiveValue color = (CSSPrimitiveValue) getCSSValue("color");
		if(color == null) {
			if(styleDb != null) {
				color = getStyleDatabase().getInitialColor();
			} else {
				color = AbstractStyleDatabase.DEFAULT_INITIAL_COLOR;
			}
		}
		return color;
	}

	/**
	 * Set the subproperties of a shorthand, and returns its text representation.
	 * 
	 * @param propertyName
	 *            the shorthand property name.
	 * @param value
	 *            the shorthand property value.
	 * @param priority
	 *            the shorthand property priority.
	 * @return the text representation of the shorthand, or null if propertyName 
	 * is not a shorthand.
	 * @throws DOMException
	 */
	protected String setSubproperties(String propertyName, LexicalUnit value,
			String priority) throws DOMException {
		PropertyDatabase pdb = PropertyDatabase.getInstance();
		if (pdb.isShorthand(propertyName)) {
			ShorthandSetter setter;
			if ("font".equals(propertyName)) {
				setter = new FontShorthandSetter();
			} else if ("margin".equals(propertyName)) {
				setter = new BoxShorthandSetter(propertyName);
			} else if ("padding".equals(propertyName)) {
				setter = new BoxShorthandSetter("padding");
			} else if ("border".equals(propertyName)) {
				setter = new BorderShorthandSetter();
			} else if ("border-width".equals(propertyName)) {
				setter = new BoxShorthandSetter("border-width");
			} else if ("border-style".equals(propertyName)) {
				setter = new BoxShorthandSetter("border-style");
			} else if ("border-color".equals(propertyName)) {
				setter = new BoxShorthandSetter("border-color");
			} else if ("border-top".equals(propertyName)) {
				setter = new BorderSideShorthandSetter(propertyName, "top");
			} else if ("border-right".equals(propertyName)) {
				setter = new BorderSideShorthandSetter(propertyName, "right");
			} else if ("border-bottom".equals(propertyName)) {
				setter = new BorderSideShorthandSetter(propertyName, "bottom");
			} else if ("border-left".equals(propertyName)) {
				setter = new BorderSideShorthandSetter(propertyName, "left");
			} else {
				setter = new ShorthandSetter(propertyName);
			}
			setter.init(value, priority);
			setter.assignSubproperties();
			return setter.getCssText();
		} else {
			return null;
		}
	}

	@Override
	public BaseCSSStyleDeclaration clone() {
		return new BaseCSSStyleDeclaration(this);
	}

	/**
	 * Gets the SAC document handler used by this style declaration.
	 * 
	 * @return the SAC document handler.
	 */
	StyleDeclarationDocumentHandler getStyleDeclarationDocumentHandler() {
		if (styleDeclarationDocumentHandler == null) {
			styleDeclarationDocumentHandler = new StyleDeclarationDocumentHandler();
		}
		return styleDeclarationDocumentHandler;
	}

	// XXX
	public class StyleDeclarationDocumentHandler extends
			PropertyDocumentHandler {
		public StyleDeclarationDocumentHandler() {
			super();
		}

		public void startSelector(SelectorList selectors) throws CSSException {
			parentRule.setSelectorList(selectors);
		}

		public void endSelector(SelectorList selectors) throws CSSException {

		}
	}

	/**
	 * Generic class that attempts to set the subproperties of shorthand
	 * properties.
	 * 
	 * @author Carlos Amengual
	 * 
	 */
	class ShorthandSetter {
		private String shorthandName;

		private String priority = null;

		private PropertyDatabase pdb = PropertyDatabase.getInstance();

		protected LexicalUnit currentValue = null;

		private short valueCount = 0;

		private short propertyCount = 0;

		protected short lookupCount = 0;
		
		private StringBuilder valueBuffer = new StringBuilder(32);

		/**
		 * The values in the shorthand are attempted to set subproperty 
		 * values in a certain order. The properties that failed to be set 
		 * to the tested value are stored here.
		 */
		private List<String> unassignedProperties = new ArrayList<String>(6);

		private List<LexicalUnit> unassignedValues = new ArrayList<LexicalUnit>(
				8);

		ShorthandSetter(String shorthandName) {
			super();
			this.shorthandName = shorthandName;
		}

		public String getShorthandName() {
			return shorthandName;
		}

		public String getPriority() {
			return priority;
		}

		public final PropertyDatabase getPropertyDatabase() {
			return pdb;
		}

		public short getValueCount() {
			return valueCount;
		}

		public short getPropertyCount() {
			return propertyCount;
		}

		protected void addUnassignedProperty(String propertyName) {
			unassignedProperties.add(propertyName);
			int sz = unassignedValues.size();
			if (sz > 0) {
				if (!unassignedValues.get(sz - 1).equals(currentValue)) {
					unassignedValues.add(currentValue);
				}
			}
		}

		protected List<String> getUnassignedProperties() {
			return unassignedProperties;
		}

		protected void nextCurrentValue() {
			if(currentValue != null) {
				currentValue = currentValue.getNextLexicalUnit();
				// This should not be found here, but just in case (wrong styles,
				// etc.):
				// if it is a comma, get next
				if (currentValue != null
						&& currentValue.getLexicalUnitType() == LexicalUnit.SAC_OPERATOR_COMMA) {
					currentValue = currentValue.getNextLexicalUnit();
				}
				// Add the value string. We do this here in case the shorthand 
				// setter decides to stop setting values, to avoid having -in the text 
				// representation- values that were not used as subproperties.
				appendValueString();
			}
		}

		public void init(LexicalUnit shorthandValue, String priority) {
			this.currentValue = shorthandValue;
			this.priority = priority;
			// Reset the variables
			unassignedValues.clear();
			valueCount = 0;
			propertyCount = 0;
			valueBuffer.setLength(0);
			appendValueString();
			// Count values
			for (LexicalUnit value = shorthandValue; value != null; value = value
					.getNextLexicalUnit()) {
				valueCount++;
				if (value.getLexicalUnitType() != LexicalUnit.SAC_OPERATOR_COMMA) {
					propertyCount++;
				} else {
					propertyCount--;
				}
			}
		}

		/**
		 * Reset subproperties not explicitly set by this shorthand.
		 */
		protected void resetSubproperties() {
			List<String> props = getUnassignedProperties();
			Iterator<String> it = props.iterator();
			while(it.hasNext()) {
				String pname = it.next();
				if(!getPropertyDatabase().isShorthand(pname)) {
					setPropertyDefault(pname);
				} else {
					String[] sh = getPropertyDatabase().getShorthandSubproperties(pname);
					for(int i=0; i<sh.length; i++) {
						setPropertyDefault(sh[i]);
					}
				}
			}
		}

		private void setPropertyDefault(String pname) {
			CSSValue cssVal = defaultPropertyValue(pname);
			if(cssVal != null) {
				((AbstractCSSPrimitiveValue)cssVal).setSubproperty(true);
				setProperty(pname, cssVal, getPriority());
			} else {
				log.info("No default for property " + pname);
			}
		}

		public void assignSubproperties() {
			String[] subparray = getPropertyDatabase()
					.getShorthandSubproperties(getShorthandName());
			List<String> subp = new ArrayList<String>(subparray.length);
			subp.addAll(Arrays.asList(subparray.clone()));
			while (currentValue != null && lookupCount <= getValueCount()) {
				unassignedProperties.clear();
				for (int i = 0; i < subp.size(); i++) {
					String pname = subp.get(i);
					if (currentValue != null) {
						if (!assignSubproperty(pname)) {
							addUnassignedProperty(pname);
						} else {
							subp.remove(i);
							i--;
						}
					} else {
						break;
					}
					lookupCount++;
				}
			}
			if(subp.size() > 0) {
				// Add remaining unassigned properties
				Iterator<String> it = subp.iterator();
				while(it.hasNext()) {
					addUnassignedProperty(it.next());
				}
			}
			// Reset subproperties not set by this shorthand
			resetSubproperties();
		}

		protected boolean assignSubproperty(String subproperty) {
			if (LexicalUnit.SAC_IDENT == currentValue.getLexicalUnitType()) {
				if (assignIdentifiers(subproperty)) {
					return true;
				}
			}
			return false;
		}

		protected boolean assignIdentifiers(String subproperty) {
			if (testIdentifiers(subproperty)) {
				CSSValue cssValue = createCSSValue(currentValue);
				setSubpropertyValue(subproperty, cssValue);
				nextCurrentValue();
				return true;
			}
			return false;
		}

		protected boolean testIdentifiers(String subproperty) {
			return getPropertyDatabase().isIdentifierValue(subproperty,
					currentValue.getStringValue());
		}

		protected void setSubpropertyValue(String subproperty, CSSValue cssValue) {
			setProperty(subproperty, cssValue, getPriority());
			if (log.isDebugEnabled()) {
				log.debug("Assigning shorthand subproperty " + subproperty
						+ ": " + cssValue.getCssText());
			}
			// Check if the just assigned value was in the "unassigned" list
			// and erase if that is the case.
			int szm1 = unassignedValues.size() - 1;
			if (szm1 >= 0) {
				if (unassignedValues.get(szm1).equals(currentValue)) {
					unassignedValues.remove(szm1);
					szm1--;
				}
				// Check if something is still unassigned
				if (szm1 >= 0) {
					// Try to force-assign the unassigned values,
					// taking as base the last assigned one.
					// TODO
					StringBuilder sb = new StringBuilder(64);
					sb.append("Unable to assign properties: ");
					Iterator<String> it = unassignedProperties.iterator();
					while (it.hasNext()) {
						sb.append(' ').append(it.next());
					}
					log.warn(sb.toString());
				}
			}
		}
		
		public CSSValue createCSSValue(LexicalUnit lunit) {
			return AbstractCSSValue.createCSSValue(currentValue, true);
		}

		/**
		 * Adds the text representation of the subproperty, to be used in 
		 * setting the css text for the shorthand.
		 * 
		 * @param cssValue
		 */
		protected void appendValueString() {
			if (currentValue != null) {
				CSSValue currentCss = createCSSValue(currentValue);
				if(currentCss != null) {
					String cssText = currentCss.getCssText();
					if(valueBuffer.length() > 0) {
						valueBuffer.append(' ');
					}
					valueBuffer.append(cssText);
				}
			}
		}
		
		public String getCssText() {
			return valueBuffer.toString();
		}

		protected boolean testColor(String subproperty) {
			if (LexicalUnit.SAC_IDENT == currentValue.getLexicalUnitType()) {
				return getPropertyDatabase().isIdentifierValue(subproperty,
						currentValue.getStringValue());
			} else if (LexicalUnit.SAC_RGBCOLOR == currentValue
					.getLexicalUnitType()) {
				return true;
			} else if (LexicalUnit.SAC_STRING_VALUE == currentValue
					.getLexicalUnitType()
					&& currentValue.getStringValue().charAt(0) == '#') {
				return true;
			}
			return false;
		}

	}

	/**
	 * Shorthand setter for the <code>font</code> shorthand property.
	 */
	class FontShorthandSetter extends ShorthandSetter {

		// Always set to the next lexical unit.
		private LexicalUnit nextLex = null;

		FontShorthandSetter() {
			super("font");
		}

		protected void nextCurrentValue() {
			super.nextCurrentValue();
			if (currentValue != null) {
				nextLex = currentValue.getNextLexicalUnit();
			} else {
				nextLex = null;
			}
		}

		protected boolean assignSubproperty(String subproperty) {
			// font-size is a special case
			if (subproperty.equals("font-size")) {
				if (assignFontSize()) {
					// Check for line-height
					if (currentValue != null
							&& currentValue.getLexicalUnitType() == LexicalUnit.SAC_OPERATOR_SLASH) {
						// the line-height value
						nextCurrentValue();
						CSSValue cssValue = createCSSValue(currentValue);
						setProperty("line-height", cssValue, getPriority());
						if (log.isDebugEnabled()) {
							log
									.debug("Assigning shorthand subproperty line-height: "
											+ cssValue.getCssText());
						}
					}
					nextCurrentValue();
					return true;
				}
			}
			// Rest of properties
			switch (currentValue.getLexicalUnitType()) {
			case LexicalUnit.SAC_IDENT:
				if (super.assignSubproperty(subproperty)) {
					return true;
				} else if (subproperty.equals("font-family")) {
					// Check for font-family
					// CSS Spec, 15.3: Font family names must either be given quoted as strings, 
					// or unquoted as a sequence of one or more identifiers.
					if (nextLex == null
							|| nextLex.getLexicalUnitType() == LexicalUnit.SAC_OPERATOR_COMMA) {
						CSSValue cssValue = createCSSValue(currentValue);
						setSubpropertyValue(subproperty, cssValue);
						currentValue = null;
						nextLex = null;
						return true;
					}
				}
				break;
			case LexicalUnit.SAC_STRING_VALUE:
				if (subproperty.equals("font-family")) {
					CSSValue cssValue = createCSSValue(currentValue);
					setSubpropertyValue(subproperty, cssValue);
					nextCurrentValue();
					return true;
				}
			}
			return false;
		}

		protected boolean assignFontSize() {
			if (currentValue.getLexicalUnitType() == LexicalUnit.SAC_IDENT) {
				return super.assignSubproperty("font-size");
			} else {
				CSSValue cssValue = createCSSValue(currentValue);
				if (cssValue instanceof CSSNumberValue) {
					setSubpropertyValue("font-size", cssValue);
					nextCurrentValue();
					return true;
				}
			}
			return false;
		}

	}

	/**
	 * Shorthand setter for box-style properties, with top, right, bottom and
	 * left possible subproperties.
	 */
	class BoxShorthandSetter extends ShorthandSetter {

		BoxShorthandSetter(String shorthand) {
			super(shorthand);
		}

		public void assignSubproperties() {
			String[] subparray = getPropertyDatabase()
					.getShorthandSubproperties(getShorthandName());
			switch (getPropertyCount()) {
			case 1:
				CSSValue cssValue = createCSSValue(currentValue);
				for (int i = 0; i < subparray.length; i++) {
					setSubpropertyValue(subparray[i], cssValue);
				}
				break;
			case 2:
				cssValue = createCSSValue(currentValue);
				// top
				setSubpropertyValue(subparray[0], cssValue);
				// bottom
				setSubpropertyValue(subparray[2], cssValue);
				nextCurrentValue();
				cssValue = createCSSValue(currentValue);
				// right
				setSubpropertyValue(subparray[1], cssValue);
				// left
				setSubpropertyValue(subparray[3], cssValue);
				break;
			case 3:
				cssValue = createCSSValue(currentValue);
				// top
				setSubpropertyValue(subparray[0], cssValue);
				nextCurrentValue();
				cssValue = createCSSValue(currentValue);
				// right
				setSubpropertyValue(subparray[1], cssValue);
				// left
				setSubpropertyValue(subparray[3], cssValue);
				nextCurrentValue();
				cssValue = createCSSValue(currentValue);
				// bottom
				setSubpropertyValue(subparray[2], cssValue);
				break;
			case 4:
				for (int i = 0; i < subparray.length; i++) {
					cssValue = createCSSValue(currentValue);
					setSubpropertyValue(subparray[i], cssValue);
					nextCurrentValue();
				}
				break;
			default:
				log.error("Found " + Integer.toString(getPropertyCount())
						+ " values for '" + getShorthandName()
						+ "' shorthand property");
			}
		}

	}

	/**
	 * Shorthand setter for the <code>border</code> property.
	 */
	class BorderShorthandSetter extends ShorthandSetter {

		BorderShorthandSetter() {
			super("border");
		}

		protected boolean assignSubproperty(String subproperty) {
			if ("border-width".equals(subproperty)) {
				CSSValue cssValue = createCSSValue(currentValue);
				if ((LexicalUnit.SAC_IDENT == currentValue.getLexicalUnitType()
						&& testIdentifiers(subproperty))
						|| cssValue instanceof CSSNumberValue) {
					String[] subparray = getPropertyDatabase()
							.getShorthandSubproperties(subproperty);
					for (int i = 0; i < subparray.length; i++) {
						setSubpropertyValue(subparray[i], cssValue);
					}
					nextCurrentValue();
					return true;
				}
			} else if ("border-style".equals(subproperty)) {
				if (LexicalUnit.SAC_IDENT == currentValue.getLexicalUnitType()
						&& testIdentifiers(subproperty)) {
					CSSValue cssValue = createCSSValue(currentValue);
					String[] subparray = getPropertyDatabase()
							.getShorthandSubproperties(subproperty);
					for (int i = 0; i < subparray.length; i++) {
						setSubpropertyValue(subparray[i], cssValue);
					}
					nextCurrentValue();
					return true;
				}
			} else if ("border-color".equals(subproperty)
					&& testColor(subproperty)) {
				CSSValue cssValue = createCSSValue(currentValue);
				String[] subparray = getPropertyDatabase()
						.getShorthandSubproperties(subproperty);
				for (int i = 0; i < subparray.length; i++) {
					setSubpropertyValue(subparray[i], cssValue);
				}
				// border-color must be last
				currentValue = null;
				return true;
			}
			return false;
		}

	}

	/**
	 * Shorthand setter for the <code>border-top</code>, <code>border-right</code>,  
	 * <code>border-bottom</code> and <code>border-left</code> properties.
	 */
	class BorderSideShorthandSetter extends ShorthandSetter {
		private String pnameWidth;
		private String pnameStyle;
		private String pnameColor;

		BorderSideShorthandSetter(String shorthandName, String side) {
			super(shorthandName);
			pnameWidth = "border-" + side + "-width";
			pnameStyle = "border-" + side + "-style";
			pnameColor = "border-" + side + "-color";
		}

		protected boolean assignSubproperty(String subproperty) {
			if (pnameWidth.equals(subproperty)) {
				CSSValue cssValue = createCSSValue(currentValue);
				if ((LexicalUnit.SAC_IDENT == currentValue.getLexicalUnitType() 
						&& testIdentifiers("border-width"))
						|| cssValue instanceof CSSNumberValue) {
					setSubpropertyValue(subproperty, cssValue);
					nextCurrentValue();
					return true;
				}
			} else if (pnameStyle.equals(subproperty)) {
				if (LexicalUnit.SAC_IDENT == currentValue.getLexicalUnitType()
						&& testIdentifiers("border-style")) {
					CSSValue cssValue = createCSSValue(currentValue);
					setSubpropertyValue(subproperty, cssValue);
					nextCurrentValue();
					return true;
				}
			} else if (pnameColor.equals(subproperty)
					&& testColor(subproperty)) {
				CSSValue cssValue = createCSSValue(currentValue);
				setSubpropertyValue(subproperty, cssValue);
				// border-side-color must be last
				currentValue = null;
				return true;
			}
			return false;
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.w3c.dom.css.CSS2Properties Not really sure this interface is
	 *      useful. TODO: complete property getter / setters, test it.
	 */
	class CSS2PropertiesImpl implements CSS2Properties {

		public String getAzimuth() {
			return getPropertyValue("azimuth");
		}

		public void setAzimuth(String azimuth) throws DOMException {
			getPropertyCSSValue("azimuth").setCssText(azimuth);
		}

		public String getBackground() {
			return getPropertyValue("background");
		}

		public void setBackground(String background) throws DOMException {
			getPropertyCSSValue("background").setCssText(background);
		}

		public String getBackgroundAttachment() {
			return getPropertyValue("background-attachment");
		}

		public void setBackgroundAttachment(String backgroundAttachment)
				throws DOMException {
			getPropertyCSSValue("background-attachment").setCssText(
					backgroundAttachment);
		}

		public String getBackgroundColor() {
			return getPropertyValue("background-color");
		}

		public void setBackgroundColor(String backgroundColor)
				throws DOMException {
			getPropertyCSSValue("background-color").setCssText(backgroundColor);
		}

		public String getBackgroundImage() {
			return getPropertyValue("background-image");
		}

		public void setBackgroundImage(String backgroundImage)
				throws DOMException {
			getPropertyCSSValue("background-attachment").setCssText(
					backgroundImage);
		}

		public String getBackgroundPosition() {
			return getPropertyValue("background-position");
		}

		public void setBackgroundPosition(String backgroundPosition)
				throws DOMException {
			getPropertyCSSValue("background-position").setCssText(
					backgroundPosition);
		}

		public String getBackgroundRepeat() {
			return getPropertyValue("background-repeat");
		}

		public void setBackgroundRepeat(String backgroundRepeat)
				throws DOMException {
			getPropertyCSSValue("background-repeat").setCssText(
					backgroundRepeat);
		}

		public String getBorder() {
			return getPropertyValue("border");
		}

		public void setBorder(String border) throws DOMException {
			getPropertyCSSValue("border").setCssText(border);
		}

		public String getBorderCollapse() {
			return getPropertyValue("border-collapse");
		}

		public void setBorderCollapse(String borderCollapse)
				throws DOMException {
			getPropertyCSSValue("border-collapse").setCssText(borderCollapse);
		}

		public String getBorderColor() {
			return getPropertyValue("border-color");
		}

		public void setBorderColor(String borderColor) throws DOMException {
			getPropertyCSSValue("border-color").setCssText(borderColor);
		}

		public String getBorderSpacing() {
			return getPropertyValue("border-spacing");
		}

		public void setBorderSpacing(String borderSpacing) throws DOMException {
			getPropertyCSSValue("border-spacing").setCssText(borderSpacing);
		}

		public String getBorderStyle() {
			return getPropertyValue("border-style");
		}

		public void setBorderStyle(String borderStyle) throws DOMException {
			getPropertyCSSValue("border-style").setCssText(borderStyle);
		}

		public String getBorderTop() {
			return getPropertyValue("border-top");
		}

		public void setBorderTop(String borderTop) throws DOMException {
			getPropertyCSSValue("border-top").setCssText(borderTop);
		}

		public String getBorderRight() {
			return getPropertyValue("border-right");
		}

		public void setBorderRight(String borderRight) throws DOMException {
			getPropertyCSSValue("border-right").setCssText(borderRight);
		}

		public String getBorderBottom() {
			return getPropertyValue("border-bottom");
		}

		public void setBorderBottom(String borderBottom) throws DOMException {
			getPropertyCSSValue("border-bottom").setCssText(borderBottom);
		}

		public String getBorderLeft() {
			return getPropertyValue("border-left");
		}

		public void setBorderLeft(String borderLeft) throws DOMException {
			getPropertyCSSValue("border-left").setCssText(borderLeft);
		}

		public String getBorderTopColor() {
			return getPropertyValue("border-top-color");
		}

		public void setBorderTopColor(String borderTopColor)
				throws DOMException {
			getPropertyCSSValue("border-top-color").setCssText(borderTopColor);
		}

		public String getBorderRightColor() {
			return getPropertyValue("border-right-color");
		}

		public void setBorderRightColor(String borderRightColor)
				throws DOMException {
			getPropertyCSSValue("border-right-color").setCssText(
					borderRightColor);
		}

		public String getBorderBottomColor() {
			return getPropertyValue("border-bottom-color");
		}

		public void setBorderBottomColor(String borderBottomColor)
				throws DOMException {
			getPropertyCSSValue("border-bottom-color").setCssText(
					borderBottomColor);
		}

		public String getBorderLeftColor() {
			return getPropertyValue("border-left-color");
		}

		public void setBorderLeftColor(String borderLeftColor)
				throws DOMException {
			getPropertyCSSValue("border-left-color")
					.setCssText(borderLeftColor);
		}

		public String getBorderTopStyle() {
			return getPropertyValue("border-top-style");
		}

		public void setBorderTopStyle(String borderTopStyle)
				throws DOMException {
			getPropertyCSSValue("border-top-style").setCssText(borderTopStyle);
		}

		public String getBorderRightStyle() {
			return getPropertyValue("border-right-style");
		}

		public void setBorderRightStyle(String borderRightStyle)
				throws DOMException {
			getPropertyCSSValue("border-right-style").setCssText(
					borderRightStyle);
		}

		public String getBorderBottomStyle() {
			return getPropertyValue("border-bottom-style");
		}

		public void setBorderBottomStyle(String borderBottomStyle)
				throws DOMException {
			getPropertyCSSValue("border-bottom-style").setCssText(
					borderBottomStyle);
		}

		public String getBorderLeftStyle() {
			return getPropertyValue("border-left-style");
		}

		public void setBorderLeftStyle(String borderLeftStyle)
				throws DOMException {
			getPropertyCSSValue("border-left-style")
					.setCssText(borderLeftStyle);
		}

		public String getBorderTopWidth() {
			return getPropertyValue("border-top-width");
		}

		public void setBorderTopWidth(String borderTopWidth)
				throws DOMException {
			getPropertyCSSValue("border-top-width").setCssText(borderTopWidth);
		}

		public String getBorderRightWidth() {
			return getPropertyValue("border-right-width");
		}

		public void setBorderRightWidth(String borderRightWidth)
				throws DOMException {
			getPropertyCSSValue("border-right-width").setCssText(
					borderRightWidth);
		}

		public String getBorderBottomWidth() {
			return getPropertyValue("border-bottom-width");
		}

		public void setBorderBottomWidth(String borderBottomWidth)
				throws DOMException {
			getPropertyCSSValue("border-bottom-width").setCssText(
					borderBottomWidth);
		}

		public String getBorderLeftWidth() {
			return getPropertyValue("border-left-width");
		}

		public void setBorderLeftWidth(String borderLeftWidth)
				throws DOMException {
			getPropertyCSSValue("border-left-width")
					.setCssText(borderLeftWidth);
		}

		public String getBorderWidth() {
			return getPropertyValue("border-width");
		}

		public void setBorderWidth(String borderWidth) throws DOMException {
			getPropertyCSSValue("border-width").setCssText(borderWidth);
		}

		public String getBottom() {
			return getPropertyValue("bottom");
		}

		public void setBottom(String bottom) throws DOMException {
			getPropertyCSSValue("bottom").setCssText(bottom);
		}

		public String getCaptionSide() {
			return getPropertyValue("caption-side");
		}

		public void setCaptionSide(String captionSide) throws DOMException {
			getPropertyCSSValue("caption-side").setCssText(captionSide);
		}

		public String getClear() {
			return getPropertyValue("clear");
		}

		public void setClear(String clear) throws DOMException {
			getPropertyCSSValue("clear").setCssText(clear);
		}

		public String getClip() {
			return getPropertyValue("clip");
		}

		public void setClip(String clip) throws DOMException {
			getPropertyCSSValue("clip").setCssText(clip);
		}

		public String getColor() {
			return getPropertyValue("color");
		}

		public void setColor(String color) throws DOMException {
			getPropertyCSSValue("color").setCssText(color);
		}

		public String getContent() {
			return getPropertyValue("content");
		}

		public void setContent(String content) throws DOMException {
			getPropertyCSSValue("content").setCssText(content);
		}

		public String getCounterIncrement() {
			return getPropertyValue("counter-increment");
		}

		public void setCounterIncrement(String counterIncrement)
				throws DOMException {
			getPropertyCSSValue("counter-increment").setCssText(counterIncrement);
		}

		public String getCounterReset() {
			return getPropertyValue("counter-reset");
		}

		public void setCounterReset(String counterReset) throws DOMException {
			getPropertyCSSValue("counter-reset").setCssText(counterReset);
		}

		public String getCue() {
			return getPropertyValue("cue");
		}

		public void setCue(String cue) throws DOMException {
			getPropertyCSSValue("cue").setCssText(cue);
		}

		public String getCueAfter() {
			return getPropertyValue("cue-after");
		}

		public void setCueAfter(String cueAfter) throws DOMException {
			getPropertyCSSValue("cue-after").setCssText(cueAfter);
		}

		public String getCueBefore() {
			return getPropertyValue("cue-before");
		}

		public void setCueBefore(String cueBefore) throws DOMException {
			getPropertyCSSValue("cue-before").setCssText(cueBefore);
		}

		public String getCursor() {
			return getPropertyValue("cursor");
		}

		public void setCursor(String cursor) throws DOMException {
			getPropertyCSSValue("cursor").setCssText(cursor);
		}

		public String getDirection() {
			return getPropertyValue("direction");
		}

		public void setDirection(String direction) throws DOMException {
			getPropertyCSSValue("direction").setCssText(direction);
		}

		public String getDisplay() {
			return getPropertyValue("display");
		}

		public void setDisplay(String display) throws DOMException {
			getPropertyCSSValue("display").setCssText(display);
		}

		public String getElevation() {
			return getPropertyValue("elevation");
		}

		public void setElevation(String elevation) throws DOMException {
			getPropertyCSSValue("elevation").setCssText(elevation);
		}

		public String getEmptyCells() {
			return getPropertyValue("empty-cells");
		}

		public void setEmptyCells(String emptyCells) throws DOMException {
			getPropertyCSSValue("empty-cells").setCssText(emptyCells);
		}

		public String getCssFloat() {
			return getPropertyValue("css-float");
		}

		public void setCssFloat(String cssFloat) throws DOMException {
			getPropertyCSSValue("css-float").setCssText(cssFloat);
		}

		public String getFont() {
			return getPropertyValue("font");
		}

		public void setFont(String font) throws DOMException {
			getPropertyCSSValue("font").setCssText(font);
		}

		public String getFontFamily() {
			return getPropertyValue("font-family");
		}

		public void setFontFamily(String fontFamily) throws DOMException {
			getPropertyCSSValue("font-family").setCssText(fontFamily);
		}

		public String getFontSize() {
			return getPropertyValue("font-size");
		}

		public void setFontSize(String fontSize) throws DOMException {
			getPropertyCSSValue("font-size").setCssText(fontSize);
		}

		public String getFontSizeAdjust() {
			return getPropertyValue("font-size-adjust");
		}

		public void setFontSizeAdjust(String fontSizeAdjust)
				throws DOMException {
			getPropertyCSSValue("font-size-adjust").setCssText(fontSizeAdjust);
		}

		public String getFontStretch() {
			return getPropertyValue("font-stretch");
		}

		public void setFontStretch(String fontStretch) throws DOMException {
			getPropertyCSSValue("font-stretch").setCssText(fontStretch);
		}

		public String getFontStyle() {
			return getPropertyValue("font-style");
		}

		public void setFontStyle(String fontStyle) throws DOMException {
			getPropertyCSSValue("font-style").setCssText(fontStyle);
		}

		public String getFontVariant() {
			return getPropertyValue("font-variant");
		}

		public void setFontVariant(String fontVariant) throws DOMException {
			getPropertyCSSValue("font-variant").setCssText(fontVariant);
		}

		public String getFontWeight() {
			return getPropertyValue("font-weight");
		}

		public void setFontWeight(String fontWeight) throws DOMException {
			getPropertyCSSValue("font-weight").setCssText(fontWeight);
		}

		public String getHeight() {
			return getPropertyValue("height");
		}

		public void setHeight(String height) throws DOMException {
			getPropertyCSSValue("height").setCssText(height);
		}

		public String getLeft() {
			return getPropertyValue("left");
		}

		public void setLeft(String left) throws DOMException {
			getPropertyCSSValue("left").setCssText(left);
		}

		public String getLetterSpacing() {
			return getPropertyValue("letter-spacing");
		}

		public void setLetterSpacing(String letterSpacing) throws DOMException {
			getPropertyCSSValue("letter-spacing").setCssText(letterSpacing);
		}

		public String getLineHeight() {
			return getPropertyValue("line-height");
		}

		public void setLineHeight(String lineHeight) throws DOMException {
			getPropertyCSSValue("line-height").setCssText(lineHeight);
		}

		public String getListStyle() {
			return getPropertyValue("list-style");
		}

		public void setListStyle(String listStyle) throws DOMException {
			getPropertyCSSValue("list-style").setCssText(listStyle);
		}

		public String getListStyleImage() {
			return getPropertyValue("list-style-image");
		}

		public void setListStyleImage(String listStyleImage)
				throws DOMException {
			getPropertyCSSValue("list-style-image").setCssText(listStyleImage);
		}

		public String getListStylePosition() {
			return getPropertyValue("list-style-position");
		}

		public void setListStylePosition(String listStylePosition)
				throws DOMException {
			getPropertyCSSValue("list-style-position").setCssText(listStylePosition);
		}

		public String getListStyleType() {
			return getPropertyValue("list-style-type");
		}

		public void setListStyleType(String listStyleType) throws DOMException {
			getPropertyCSSValue("list-style-type").setCssText(listStyleType);
		}

		public String getMargin() {
			return getPropertyValue("margin");
		}

		public void setMargin(String margin) throws DOMException {
			getPropertyCSSValue("margin").setCssText(margin);
		}

		public String getMarginTop() {
			return getPropertyValue("margin-top");
		}

		public void setMarginTop(String marginTop) throws DOMException {
			getPropertyCSSValue("margin-top").setCssText(marginTop);
		}

		public String getMarginRight() {
			return getPropertyValue("margin-right");
		}

		public void setMarginRight(String marginRight) throws DOMException {
			getPropertyCSSValue("margin-right").setCssText(marginRight);
		}

		public String getMarginBottom() {
			return getPropertyValue("margin-bottom");
		}

		public void setMarginBottom(String marginBottom) throws DOMException {
			getPropertyCSSValue("margin-bottom").setCssText(marginBottom);
		}

		public String getMarginLeft() {
			return getPropertyValue("margin-left");
		}

		public void setMarginLeft(String marginLeft) throws DOMException {
			getPropertyCSSValue("margin-left").setCssText(marginLeft);
		}

		public String getMarkerOffset() {
			return getPropertyValue("marker-offset");
		}

		public void setMarkerOffset(String markerOffset) throws DOMException {
			getPropertyCSSValue("marker-offset").setCssText(markerOffset);
		}

		public String getMarks() {
			return getPropertyValue("marks");
		}

		public void setMarks(String marks) throws DOMException {
			getPropertyCSSValue("marks").setCssText(marks);
		}

		public String getMaxHeight() {
			return getPropertyValue("max-height");
		}

		public void setMaxHeight(String maxHeight) throws DOMException {
			getPropertyCSSValue("max-height").setCssText(maxHeight);
		}

		public String getMaxWidth() {
			return getPropertyValue("max-width");
		}

		public void setMaxWidth(String maxWidth) throws DOMException {
			getPropertyCSSValue("max-width").setCssText(maxWidth);
		}

		public String getMinHeight() {
			return getPropertyValue("min-height");
		}

		public void setMinHeight(String minHeight) throws DOMException {
			getPropertyCSSValue("min-height").setCssText(minHeight);
		}

		public String getMinWidth() {
			return getPropertyValue("min-width");
		}

		public void setMinWidth(String minWidth) throws DOMException {
			getPropertyCSSValue("min-width").setCssText(minWidth);
		}

		public String getOrphans() {
			return getPropertyValue("orphans");
		}

		public void setOrphans(String orphans) throws DOMException {
			getPropertyCSSValue("orphans").setCssText(orphans);
		}

		public String getOutline() {
			return getPropertyValue("outline");
		}

		public void setOutline(String outline) throws DOMException {
			getPropertyCSSValue("outline").setCssText(outline);
		}

		public String getOutlineColor() {
			return getPropertyValue("outline-color");
		}

		public void setOutlineColor(String outlineColor) throws DOMException {
			getPropertyCSSValue("outline-color").setCssText(outlineColor);
		}

		public String getOutlineStyle() {
			return getPropertyValue("outline-style");
		}

		public void setOutlineStyle(String outlineStyle) throws DOMException {
			getPropertyCSSValue("outline-style").setCssText(outlineStyle);
		}

		public String getOutlineWidth() {
			return getPropertyValue("outline-width");
		}

		public void setOutlineWidth(String outlineWidth) throws DOMException {
			getPropertyCSSValue("outline-width").setCssText(outlineWidth);
		}

		public String getOverflow() {
			return getPropertyValue("overflow");
		}

		public void setOverflow(String overflow) throws DOMException {
			getPropertyCSSValue("overflow").setCssText(overflow);
		}

		public String getPadding() {
			return getPropertyValue("padding");
		}

		public void setPadding(String padding) throws DOMException {
			getPropertyCSSValue("padding").setCssText(padding);
		}

		public String getPaddingTop() {
			return getPropertyValue("padding-top");
		}

		public void setPaddingTop(String paddingTop) throws DOMException {
			getPropertyCSSValue("padding-top").setCssText(paddingTop);
		}

		public String getPaddingRight() {
			return getPropertyValue("padding-right");
		}

		public void setPaddingRight(String paddingRight) throws DOMException {
			getPropertyCSSValue("padding-right").setCssText(paddingRight);
		}

		public String getPaddingBottom() {
			return getPropertyValue("padding-bottom");
		}

		public void setPaddingBottom(String paddingBottom) throws DOMException {
			getPropertyCSSValue("padding-bottom").setCssText(paddingBottom);
		}

		public String getPaddingLeft() {
			return getPropertyValue("padding-left");
		}

		public void setPaddingLeft(String paddingLeft) throws DOMException {
			getPropertyCSSValue("padding-left").setCssText(paddingLeft);
		}

		public String getPage() {
			return getPropertyValue("page");
		}

		public void setPage(String page) throws DOMException {
			getPropertyCSSValue("page").setCssText(page);
		}

		public String getPageBreakAfter() {
			return getPropertyValue("page-break-after");
		}

		public void setPageBreakAfter(String pageBreakAfter)
				throws DOMException {
			getPropertyCSSValue("page-break-after").setCssText(pageBreakAfter);
		}

		public String getPageBreakBefore() {
			return getPropertyValue("page-break-before");
		}

		public void setPageBreakBefore(String pageBreakBefore)
				throws DOMException {
			getPropertyCSSValue("page-break-before").setCssText(pageBreakBefore);
		}

		public String getPageBreakInside() {
			return getPropertyValue("page-break-inside");
		}

		public void setPageBreakInside(String pageBreakInside)
				throws DOMException {
			getPropertyCSSValue("page-break-inside").setCssText(pageBreakInside);
		}

		public String getPause() {
			return getPropertyValue("pause");
		}

		public void setPause(String pause) throws DOMException {
			getPropertyCSSValue("pause").setCssText(pause);
		}

		public String getPauseAfter() {
			return getPropertyValue("pause-after");
		}

		public void setPauseAfter(String pauseAfter) throws DOMException {
			getPropertyCSSValue("pause-after").setCssText(pauseAfter);
		}

		public String getPauseBefore() {
			return getPropertyValue("pause-before");
		}

		public void setPauseBefore(String pauseBefore) throws DOMException {
			getPropertyCSSValue("pause-before").setCssText(pauseBefore);
		}

		public String getPitch() {
			return getPropertyValue("pitch");
		}

		public void setPitch(String pitch) throws DOMException {
			getPropertyCSSValue("pitch").setCssText(pitch);
		}

		public String getPitchRange() {
			return getPropertyValue("pitch-range");
		}

		public void setPitchRange(String pitchRange) throws DOMException {
			getPropertyCSSValue("pitch-range").setCssText(pitchRange);
		}

		public String getPlayDuring() {
			return getPropertyValue("play-during");
		}

		public void setPlayDuring(String playDuring) throws DOMException {
			getPropertyCSSValue("play-during").setCssText(playDuring);
		}

		public String getPosition() {
			return getPropertyValue("position");
		}

		public void setPosition(String position) throws DOMException {
			getPropertyCSSValue("position").setCssText(position);
		}

		public String getQuotes() {
			return getPropertyValue("quotes");
		}

		public void setQuotes(String quotes) throws DOMException {
			getPropertyCSSValue("quotes").setCssText(quotes);
		}

		public String getRichness() {
			return getPropertyValue("richness");
		}

		public void setRichness(String richness) throws DOMException {
			getPropertyCSSValue("richness").setCssText(richness);
		}

		public String getRight() {
			return getPropertyValue("right");
		}

		public void setRight(String right) throws DOMException {
			getPropertyCSSValue("right").setCssText(right);
		}

		public String getSize() {
			return getPropertyValue("size");
		}

		public void setSize(String size) throws DOMException {
			getPropertyCSSValue("size").setCssText(size);
		}

		public String getSpeak() {
			return getPropertyValue("speak");
		}

		public void setSpeak(String speak) throws DOMException {
			getPropertyCSSValue("speak").setCssText(speak);
		}

		public String getSpeakHeader() {
			return getPropertyValue("speak-header");
		}

		public void setSpeakHeader(String speakHeader) throws DOMException {
			getPropertyCSSValue("speak-header").setCssText(speakHeader);
		}

		public String getSpeakNumeral() {
			return getPropertyValue("speak-numeral");
		}

		public void setSpeakNumeral(String speakNumeral) throws DOMException {
			getPropertyCSSValue("speak-numeral").setCssText(speakNumeral);
		}

		public String getSpeakPunctuation() {
			return getPropertyValue("speak-punctuation");
		}

		public void setSpeakPunctuation(String speakPunctuation)
				throws DOMException {
			getPropertyCSSValue("speak-punctuation").setCssText(speakPunctuation);
		}

		public String getSpeechRate() {
			return getPropertyValue("speech-rate");
		}

		public void setSpeechRate(String speechRate) throws DOMException {
			getPropertyCSSValue("speech-rate").setCssText(speechRate);
		}

		public String getStress() {
			return getPropertyValue("stress");
		}

		public void setStress(String stress) throws DOMException {
			getPropertyCSSValue("stress").setCssText(stress);
		}

		public String getTableLayout() {
			return getPropertyValue("table-layout");
		}

		public void setTableLayout(String tableLayout) throws DOMException {
			getPropertyCSSValue("table-layout").setCssText(tableLayout);
		}

		public String getTextAlign() {
			return getPropertyValue("text-align");
		}

		public void setTextAlign(String textAlign) throws DOMException {
			getPropertyCSSValue("text-align").setCssText(textAlign);
		}

		public String getTextDecoration() {
			return getPropertyValue("text-decoration");
		}

		public void setTextDecoration(String textDecoration)
				throws DOMException {
			getPropertyCSSValue("text-decoration").setCssText(textDecoration);
		}

		public String getTextIndent() {
			return getPropertyValue("text-indent");
		}

		public void setTextIndent(String textIndent) throws DOMException {
			getPropertyCSSValue("text-indent").setCssText(textIndent);
		}

		public String getTextShadow() {
			return getPropertyValue("text-shadow");
		}

		public void setTextShadow(String textShadow) throws DOMException {
			getPropertyCSSValue("text-shadow").setCssText(textShadow);
		}

		public String getTextTransform() {
			return getPropertyValue("text-transform");
		}

		public void setTextTransform(String textTransform) throws DOMException {
			getPropertyCSSValue("text-transform").setCssText(textTransform);
		}

		public String getTop() {
			return getPropertyValue("top");
		}

		public void setTop(String top) throws DOMException {
			getPropertyCSSValue("top").setCssText(top);
		}

		public String getUnicodeBidi() {
			return getPropertyValue("unicode-bidi");
		}

		public void setUnicodeBidi(String unicodeBidi) throws DOMException {
			getPropertyCSSValue("unicode-bidi").setCssText(unicodeBidi);
		}

		public String getVerticalAlign() {
			return getPropertyValue("vertical-align");
		}

		public void setVerticalAlign(String verticalAlign) throws DOMException {
			getPropertyCSSValue("vertical-align").setCssText(verticalAlign);
		}

		public String getVisibility() {
			return getPropertyValue("visibility");
		}

		public void setVisibility(String visibility) throws DOMException {
			getPropertyCSSValue("visibility").setCssText(visibility);
		}

		public String getVoiceFamily() {
			return getPropertyValue("voice-family");
		}

		public void setVoiceFamily(String voiceFamily) throws DOMException {
			getPropertyCSSValue("voice-family").setCssText(voiceFamily);
		}

		public String getVolume() {
			return getPropertyValue("volume");
		}

		public void setVolume(String volume) throws DOMException {
			getPropertyCSSValue("volume").setCssText(volume);
		}

		public String getWhiteSpace() {
			return getPropertyValue("white-space");
		}

		public void setWhiteSpace(String whiteSpace) throws DOMException {
			getPropertyCSSValue("white-space").setCssText(whiteSpace);
		}

		public String getWidows() {
			return null;
		}

		public void setWidows(String widows) throws DOMException {
			getPropertyCSSValue("widows").setCssText(widows);
		}

		public String getWidth() {
			return getPropertyValue("width");
		}

		public void setWidth(String width) throws DOMException {
			getPropertyCSSValue("width").setCssText(width);
		}

		public String getWordSpacing() {
			return getPropertyValue("word-spacing");
		}

		public void setWordSpacing(String wordSpacing) throws DOMException {
			getPropertyCSSValue("word-spacing").setCssText(wordSpacing);
		}

		public String getZIndex() {
			return getPropertyValue("z-index");
		}

		public void setZIndex(String zIndex) throws DOMException {
			getPropertyCSSValue("z-index").setCssText(zIndex);
		}
	}

	public CSS2Properties getCSS2Properties() {
		return css2properties;
	}

}
