/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import java.util.ArrayList;
import java.util.Collection;

import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSRuleList;

/**
 * Stores a list of CSS rules, implementing CSSRuleList.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class CSSRuleArrayList extends ArrayList<CSSRule> implements
		CSSRuleList, RuleStore {

	private static final long serialVersionUID = 1L;

	public CSSRuleArrayList(int initialCapacity) {
		super(initialCapacity);
	}

	public CSSRuleArrayList() {
		super();
	}

	public CSSRuleArrayList(Collection<? extends CSSRule> c) {
		super(c);
	}

    /**
     * Gives the number of <code>CSSRules</code> in the list.
     * 
     * @return the number of <code>CSSRules</code> in the list. The range of valid 
     * child rule indices is <code>0</code> to <code>length-1</code> 
     * inclusive.
     */
	public int getLength() {
		return size();
	}

    /**
     * Used to retrieve a CSS rule by ordinal index. The order in this 
     * collection represents the order of the rules in the CSS style sheet. 
     * If index is greater than or equal to the number of rules in the list, 
     * this returns <code>null</code>. 
     * @param index Index into the collection.
     * @return The style rule at the <code>index</code> position in the 
     *   <code>CSSRuleList</code>, or <code>null</code> if that is not a 
     *   valid index. 
     */
	public CSSRule item(int index) {
		return get(index);
	}

	/**
	 * Insert the given CSS rule at the given index.
	 * 
	 * @param cssrule
	 *            the rule.
	 * @param index
	 *            the index at which to insert the rule.
	 * @return the index at which the rule was finally inserted.
	 */
	public int insertRule(CSSRule cssrule, int index) {
		if (index >= size()) {
			index = size();
		} else if (index < 0) {
			index = 0;
		}
		add(index, cssrule);
		((BaseCSSRule) cssrule).setInsertionIndex(index);
		for (int i = index + 1; i < size(); i++) {
			((BaseCSSRule) get(i)).setInsertionIndex(i);
		}
		return index;
	}

	@Override
	public boolean add(CSSRule o) {
		((BaseCSSRule) o).setInsertionIndex(size());
		return super.add(o);
	}

	public String toString() {
		int sz = size();
		StringBuilder sb = new StringBuilder(sz * 20);
		for (int i = 0; i < sz; i++) {
			sb.append(item(i).getCssText()).append('\n');
		}
		return sb.toString();
	}
}
