/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import java.io.IOException;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.SACMediaList;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSImportRule;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSStyleSheet;
import org.w3c.dom.stylesheets.MediaList;

/**
 * Implementation of CSSImportRule.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class DOMCSSImportRule extends BaseCSSRule implements CSSImportRule {

	private CSSStyleSheet loadedSheet = null;

	private String styleSheetURI = null;

	private DOMMediaList mediaList;

	public DOMCSSImportRule(CSSStyleSheet parentSheet) {
		super(parentSheet, CSSRule.IMPORT_RULE);
		mediaList = new DOMMediaList();
	}

	public String getHref() {
		return styleSheetURI;
	}

	public MediaList getMedia() {
		return mediaList;
	}

	public CSSStyleSheet getStyleSheet() {
		return loadedSheet;
	}

	@Override
	public void setCssText(String cssText) throws DOMException {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getCssText() {
		StringBuilder sb = new StringBuilder(80);
		sb.append("@import url(\"").append(getStyleSheet().getHref()).append(
				"\")");
		if (mediaList.getLength() > 0) {
			sb.append(' ').append(mediaList.getMediaText());
		}
		return sb.append(';').toString();
	}

	/**
	 * Loads and parses an imported CSS style sheet.
	 * 
	 * @param uri
	 *            the URI to import the sheet.
	 * @param title
	 *            the advisory title of the imported sheet. If not set, will try
	 *            to get the title from the parent style sheet.
	 * @param media
	 *            the destination SAC media list for the style information.
	 * @throws CSSException
	 * @throws IOException
	 * @throws DOMException
	 */
	public void loadStyleSheet(String uri, String title, SACMediaList media)
			throws CSSException, IOException, DOMException {
		styleSheetURI = uri;
		mediaList.appendSACMediaList(media);
		BaseCSSStyleSheet parentSS = (BaseCSSStyleSheet) getParentStyleSheet();
		if (title == null) {
			if (parentSS != null) {
				title = parentSS.getTitle();
			}
		}
		// Create & Parse
		BaseCSSStyleSheet css = (BaseCSSStyleSheet) parentSS
			.getStyleSheetFactory().createStyleSheet(parentSS.getNamespaceURI(), mediaList);
		css.setTitle(title);
		css.parseCSSStyleSheet(new InputSource(uri));
		loadedSheet = css;
	}
}
