/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.property.CSSStringValue;
import info.informatica.doc.style.css.visual.NonStaticallyPositioned;
import info.informatica.doc.style.css.visual.container.CSSBoxContainer;

import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * CSS absolutely positioned Box.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public class AbsolutelyPositionedBox extends BlockBox 
		implements NonStaticallyPositioned {

	private float left;
	
	private float marginRight;
	
	private float marginLeft;

	private float width;

	private float right;

	public AbsolutelyPositionedBox(CSS2ComputedProperties style) {
		super(style);
		computeDimension(0, 0);
	}

	public void computeDimension() {
		CSSBoxContainer container = asContainerBox();
		computeDimension(container.getMinimumWidth(), container.getContentWidth());
	}

	public void computeDimension(float preferredMinimumWidth, float preferredWidth) {
		CSSPrimitiveValue cssLeft = getCSSValue("left");
		CSSPrimitiveValue cssMarginLeft = getCSSValue("margin-left");
		CSSPrimitiveValue cssWidth = getCSSValue("width");
		CSSPrimitiveValue cssMarginRight = getCSSValue("margin-right");
		CSSPrimitiveValue cssRight = getCSSValue("right");
		// What properties are auto ?
		boolean autoLeft = cssLeft instanceof CSSStringValue;
		boolean autoMarginLeft = cssMarginLeft instanceof CSSStringValue;
		boolean autoWidth = cssWidth instanceof CSSStringValue;
		boolean autoMarginRight = cssMarginRight instanceof CSSStringValue;
		boolean autoRight = cssRight instanceof CSSStringValue;
		// Compute directly specified values
		if(!autoLeft){
			left = computeNumberOrZero(cssLeft);
		}
		marginLeft = computeNumberOrZero(cssMarginLeft);
		if(!autoWidth){
			width = computeNumberOrZero(cssWidth);
		}
		marginRight = computeNumberOrZero(cssMarginRight);
		if(!autoRight){
			right = computeNumberOrZero(cssRight);
		}
		// Tell the cases
		if(autoLeft && autoWidth && autoRight) {
			computeLeftWidthRightAreAuto(preferredMinimumWidth, preferredWidth);
		} else if (!autoLeft && !autoWidth && !autoRight) {
			if(autoMarginLeft) {
				if(autoMarginRight) {
					// both margin-left and margin-right are 'auto'.
					marginLeft = (getContainerWidth() - left - getBorderLeftWidth() 
					- getPaddingLeft() - width - getPaddingRight() - getBorderRightWidth() 
					- right - computeScrollbarWidth()) / 2;
					if(marginLeft < 0) {
						/* TODO: when direction of the containing block 
						 * is 'ltr' ('rtl'), set 'margin-left' ('margin-right') to zero 
						 * and solve for 'margin-right' ('margin-left') 
						 */
						marginLeft = 0;
					}
					marginRight = marginLeft;
				} else {
					// margin-left is 'auto', margin-right is not.
					marginLeft = getContainerWidth() - left - getBorderLeftWidth() 
							- getPaddingLeft() - width - getPaddingRight() - getBorderRightWidth() 
							- marginRight - right - computeScrollbarWidth();
				}
			} else if(autoMarginRight) {
				// margin-right is 'auto', margin-left is not.
				marginRight = getContainerWidth() - left - marginLeft - getBorderLeftWidth() 
						- getPaddingLeft() - width - getPaddingRight() - getBorderRightWidth() 
						- right - computeScrollbarWidth();
			} // Neither margin-left nor margin-right are 'auto'
		} else if (autoLeft) {
			if(autoWidth) {
				// left and width are 'auto', right is not.
				// shrink-to-fit
				width = getContainerWidth() - marginLeft - getBorderLeftWidth() 
					- getPaddingLeft() - right - getPaddingRight() - getBorderRightWidth() 
					- marginRight - computeScrollbarWidth();
				if(preferredWidth != 0) {
					width = shrinkToFit(preferredMinimumWidth, width, preferredWidth);
				}
			} // left is 'auto', right and width are not.
			left = getContainerWidth() - marginLeft - getBorderLeftWidth() 
				- getPaddingLeft() - width -right - getPaddingRight() 
				- getBorderRightWidth() - marginRight - computeScrollbarWidth();
		} else if (autoRight) {
			if(autoWidth) {
				// right and width are 'auto', left is not.
				// shrink-to-fit
				width = getContainerWidth() - left - marginLeft - getBorderLeftWidth() 
					- getPaddingLeft() - getPaddingRight() - getBorderRightWidth() 
					- marginRight - computeScrollbarWidth();
				if(preferredWidth != 0) {
					width = shrinkToFit(preferredMinimumWidth, width, preferredWidth);
				}
			} // right is 'auto', left and width are not.
			right = getContainerWidth() - left - marginLeft - getBorderLeftWidth() 
				- getPaddingLeft() - width - getPaddingRight() - getBorderRightWidth() 
				- marginRight - computeScrollbarWidth();
		} else {
			// width is 'auto', left and right are not.
			width = getContainerWidth() - left - marginLeft - getBorderLeftWidth() 
				- getPaddingLeft() - right - getPaddingRight() - getBorderRightWidth() 
				- marginRight - computeScrollbarWidth();
		}
	}

	private void computeLeftWidthRightAreAuto(float preferredMinimumWidth, float preferredWidth) {
		String direction = getCSSValue("direction").getStringValue();
		if(direction.equals("ltr")) {
			// Set left to static position
			// TODO
			left = 0;  // assume static position is 0
			// shrink-to-fit
			width = getContainerWidth() - left - marginLeft - getBorderLeftWidth() 
			- getPaddingLeft() - getPaddingRight() - getBorderRightWidth() 
			- marginRight - computeScrollbarWidth();
			if(preferredWidth != 0) {
				width = shrinkToFit(preferredMinimumWidth, width, preferredWidth);
			}
		} else {
			// Set right to static position
			// TODO
			right = 0;  // assume static position is 0
			// shrink-to-fit
			width = getContainerWidth() - marginLeft - getBorderLeftWidth() 
			- getPaddingLeft() - right - getPaddingRight() - getBorderRightWidth() 
			- marginRight - computeScrollbarWidth();
			if(preferredWidth != 0) {
				width = shrinkToFit(preferredMinimumWidth, width, preferredWidth);
			}
		}
	}

	/**
	 * Gets the value of the margin-right property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-right property.
	 */
	public float getMarginRight() {
		return marginRight;
	}

	/**
	 * Gets the value of the margin-left property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-left property.
	 */
	public float getMarginLeft() {
		return marginLeft;
	}

	public float getLeft() {
		return left;
	}

	public float getRight() {
		return right;
	}

	public float getWidth() {
		return width;
	}

}
