/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.property.CSSNumberValue;
import info.informatica.doc.style.css.visual.CSSContainerBox;
import info.informatica.doc.style.css.visual.NonStaticallyPositioned;

import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Box that floats.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
abstract public class FloatingBox extends AbstractCSSBox 
		implements NonStaticallyPositioned {

	private float boxWidth;

	FloatingBox(CSS2ComputedProperties style) {
		super(style);
		computeDimension(0, 0);
	}

	public void computeDimension() {
		computeDimension(0, 0);
	}

	public void computeDimension(float preferredMinimumWidth, 
			float preferredWidth) {
		CSSPrimitiveValue cssValue = getCSSValue("width");
		short primiType = cssValue.getPrimitiveType();
		if(primiType == getStyleDatabase().getNaturalUnit()) {
			boxWidth = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else if(primiType == CSSPrimitiveValue.CSS_PERCENTAGE) {
			boxWidth = getContainerWidth() * 
				cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
		} else if(cssValue instanceof CSSNumberValue) {
			// Expressed in non-natural unit
			boxWidth = cssValue.getFloatValue(primiType);
		} else {
			// Even if this was not suppiled as 'auto', will be handled as such.
			// Available width
			boxWidth = getContainerWidth() - getMarginLeft() - getBorderLeftWidth() 
				- getMarginRight() - getBorderRightWidth();
			// Should we take scroll into account ?
			if(overflowsWithScroll()) {
				boxWidth -= getStyleDatabase().getScrollbarWidth(); 
			}
			boxWidth -= getPaddingLeft();
			boxWidth -= getPaddingRight();
			if(preferredWidth != 0f) {
				// shrink-to-fit
				boxWidth = shrinkToFit(preferredMinimumWidth, boxWidth, preferredWidth);
			}
		}
	}

	public float getWidth() {
		return boxWidth;
	}

	public float getHeight() {
		float boxHeight;
		CSSPrimitiveValue cssValue = getCSSValue("height");
		short primiType = cssValue.getPrimitiveType();
		if(primiType == getStyleDatabase().getNaturalUnit()) {
			boxHeight = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else if(primiType == CSSPrimitiveValue.CSS_PERCENTAGE) {
			if(isContainerHeightAuto()) {
				boxHeight =  computeAutoHeight(getWidth());
			} else {
				// Compute container height
				float containerHeight;
				CSSContainerBox container = getContainingBlock();
				if(container != null){
					containerHeight = container.getHeight();
				} else {
					// root block (should not happen here!!)
					containerHeight = getStyleDatabase().getDocumentHeight();
				}
				boxHeight = containerHeight * 
					cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
			}
		} else if(cssValue instanceof CSSNumberValue) {
			// Expressed in non-natural unit
			boxHeight = cssValue.getFloatValue(primiType);
		} else {
			// Even if this was not supplied as 'auto', will be handled as such.
			boxHeight = computeAutoHeight(getWidth());
		}
		return boxHeight;
	}

	abstract protected float computeAutoHeight(float containerWidth);

}
