/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.appium.java_client.android.options.context;

import io.appium.java_client.remote.options.BaseOptions;
import io.appium.java_client.remote.options.CanSetCapability;
import org.openqa.selenium.Capabilities;

import java.util.Optional;

import static io.appium.java_client.internal.CapabilityHelpers.toSafeBoolean;

public interface SupportsExtractChromeAndroidPackageFromContextNameOption
        <T extends BaseOptions<T>> extends Capabilities, CanSetCapability<T> {
    String EXTRACT_CHROME_ANDROID_PACKAGE_FROM_CONTEXT_NAME_OPTION =
            "extractChromeAndroidPackageFromContextName";

    /**
     * Tell chromedriver to attach to the android package we have associated
     * with the context name, rather than the package of the application under test.
     *
     * @return self instance for chaining.
     */
    default T extractChromeAndroidPackageFromContextName() {
        return amend(EXTRACT_CHROME_ANDROID_PACKAGE_FROM_CONTEXT_NAME_OPTION, true);
    }

    /**
     * If set to true, tell chromedriver to attach to the android package we have associated
     * with the context name, rather than the package of the application under test.
     * false by default.
     *
     * @param value Whether to use the android package identifier associated with the context name.
     * @return self instance for chaining.
     */
    default T setExtractChromeAndroidPackageFromContextName(boolean value) {
        return amend(EXTRACT_CHROME_ANDROID_PACKAGE_FROM_CONTEXT_NAME_OPTION, value);
    }

    /**
     * Get whether to use the android package identifier associated with the context name.
     *
     * @return True or false.
     */
    default Optional<Boolean> doesExtractChromeAndroidPackageFromContextName() {
        return Optional.ofNullable(
                toSafeBoolean(getCapability(EXTRACT_CHROME_ANDROID_PACKAGE_FROM_CONTEXT_NAME_OPTION))
        );
    }
}
