/*
 * Decompiled with CFR 0.152.
 */
package io.basc.framework.http;

import io.basc.framework.lang.Constants;
import io.basc.framework.lang.Nullable;
import io.basc.framework.util.Assert;
import io.basc.framework.util.CollectionUtils;
import io.basc.framework.util.ObjectUtils;
import io.basc.framework.value.Value;
import java.io.ByteArrayOutputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public final class ContentDisposition {
    private static final String INVALID_HEADER_FIELD_PARAMETER_FORMAT = "Invalid header field parameter format (as defined in RFC 5987)";
    private final String type;
    private final String name;
    private final String filename;
    private final Charset charset;
    private final Map<String, Value> attributes;

    private ContentDisposition(String type, String name, String filename, Charset charset, Map<String, Value> attributes) {
        this.type = type;
        this.name = name;
        this.filename = filename;
        this.charset = charset;
        this.attributes = attributes;
    }

    public String getType() {
        return this.type;
    }

    public String getName() {
        return this.name;
    }

    public String getFilename() {
        return this.filename;
    }

    public Charset getCharset() {
        return this.charset;
    }

    public Map<String, Value> getAttributes() {
        if (this.attributes == null) {
            return Collections.emptyMap();
        }
        return this.attributes;
    }

    public Value getAttribute(String name) {
        return this.getAttributes().get(name);
    }

    public boolean equals(@Nullable Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof ContentDisposition)) {
            return false;
        }
        ContentDisposition otherCd = (ContentDisposition)other;
        return ObjectUtils.equals(this.type, otherCd.type) && ObjectUtils.equals(this.name, otherCd.name) && ObjectUtils.equals(this.filename, otherCd.filename) && ObjectUtils.equals(this.charset, otherCd.charset);
    }

    public int hashCode() {
        int result = ObjectUtils.hashCode(this.type);
        result = 31 * result + ObjectUtils.hashCode(this.name);
        result = 31 * result + ObjectUtils.hashCode(this.filename);
        result = 31 * result + ObjectUtils.hashCode(this.charset);
        return result;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        if (this.type != null) {
            sb.append(this.type);
        }
        if (this.name != null) {
            sb.append("; name=\"");
            sb.append(this.name).append('\"');
        }
        if (this.filename != null) {
            if (this.charset == null || Constants.US_ASCII.equals(this.charset)) {
                sb.append("; filename=\"");
                sb.append(ContentDisposition.escapeQuotationsInFilename(this.filename)).append('\"');
            } else {
                sb.append("; filename*=");
                sb.append(ContentDisposition.encodeFilename(this.filename, this.charset));
            }
        }
        if (!CollectionUtils.isEmpty(this.attributes)) {
            for (Map.Entry<String, Value> entry : this.attributes.entrySet()) {
                sb.append("; ").append(entry.getKey()).append("=");
                if (entry.getValue().isNumber()) {
                    sb.append(entry.getValue().getAsNumber());
                    continue;
                }
                sb.append("\"").append(entry.getValue().getAsString()).append("\"");
            }
        }
        return sb.toString();
    }

    public static Builder builder(String type) {
        return new BuilderImpl(type);
    }

    public static ContentDisposition empty() {
        return new ContentDisposition("", null, null, null, null);
    }

    public static ContentDisposition parse(String contentDisposition) {
        List<String> parts = ContentDisposition.tokenize(contentDisposition);
        String type = parts.get(0);
        String name = null;
        String filename = null;
        Charset charset = null;
        LinkedHashMap<String, Value> attributes = new LinkedHashMap<String, Value>(8);
        for (int i = 1; i < parts.size(); ++i) {
            String part = parts.get(i);
            int eqIndex = part.indexOf(61);
            if (eqIndex != -1) {
                String value;
                String attribute = part.substring(0, eqIndex);
                boolean isText = part.startsWith("\"", eqIndex + 1) && part.endsWith("\"");
                String string = value = isText ? part.substring(eqIndex + 2, part.length() - 1) : part.substring(eqIndex + 1);
                if (attribute.equals("name")) {
                    name = value;
                    continue;
                }
                if (attribute.equals("filename*")) {
                    int idx1 = value.indexOf(39);
                    int idx2 = value.indexOf(39, idx1 + 1);
                    if (idx1 != -1 && idx2 != -1) {
                        charset = Charset.forName(value.substring(0, idx1).trim());
                        Assert.isTrue(Constants.UTF_8.equals(charset) || Constants.ISO_8859_1.equals(charset), "Charset should be UTF-8 or ISO-8859-1");
                        filename = ContentDisposition.decodeFilename(value.substring(idx2 + 1), charset);
                        continue;
                    }
                    filename = ContentDisposition.decodeFilename(value, Constants.US_ASCII);
                    continue;
                }
                if (attribute.equals("filename") && filename == null) {
                    filename = value;
                    continue;
                }
                attributes.put(attribute, Value.of(value));
                continue;
            }
            throw new IllegalArgumentException("Invalid content disposition format");
        }
        return new ContentDisposition(type, name, filename, charset, attributes);
    }

    private static List<String> tokenize(String headerValue) {
        int index = headerValue.indexOf(59);
        String type = (index >= 0 ? headerValue.substring(0, index) : headerValue).trim();
        if (type.isEmpty()) {
            throw new IllegalArgumentException("Content-Disposition header must not be empty");
        }
        ArrayList<String> parts = new ArrayList<String>();
        parts.add(type);
        if (index >= 0) {
            int nextIndex;
            do {
                String part;
                boolean quoted = false;
                boolean escaped = false;
                for (nextIndex = index + 1; nextIndex < headerValue.length(); ++nextIndex) {
                    char ch = headerValue.charAt(nextIndex);
                    if (ch == ';') {
                        if (!quoted) {
                            break;
                        }
                    } else if (!escaped && ch == '\"') {
                        quoted = !quoted;
                    }
                    escaped = !escaped && ch == '\\';
                }
                if ((part = headerValue.substring(index + 1, nextIndex).trim()).isEmpty()) continue;
                parts.add(part);
            } while ((index = nextIndex) < headerValue.length());
        }
        return parts;
    }

    private static String decodeFilename(String filename, Charset charset) {
        Assert.notNull((Object)filename, "'input' String` should not be null");
        Assert.notNull((Object)charset, "'charset' should not be null");
        byte[] value = filename.getBytes(charset);
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        int index = 0;
        while (index < value.length) {
            byte b = value[index];
            if (ContentDisposition.isRFC5987AttrChar(b)) {
                bos.write((char)b);
                ++index;
                continue;
            }
            if (b == 37 && index < value.length - 2) {
                char[] array = new char[]{(char)value[index + 1], (char)value[index + 2]};
                try {
                    bos.write(Integer.parseInt(String.valueOf(array), 16));
                }
                catch (NumberFormatException ex) {
                    throw new IllegalArgumentException(INVALID_HEADER_FIELD_PARAMETER_FORMAT, ex);
                }
                index += 3;
                continue;
            }
            throw new IllegalArgumentException(INVALID_HEADER_FIELD_PARAMETER_FORMAT);
        }
        return new String(bos.toByteArray(), charset);
    }

    private static boolean isRFC5987AttrChar(byte c) {
        return c >= 48 && c <= 57 || c >= 97 && c <= 122 || c >= 65 && c <= 90 || c == 33 || c == 35 || c == 36 || c == 38 || c == 43 || c == 45 || c == 46 || c == 94 || c == 95 || c == 96 || c == 124 || c == 126;
    }

    private static String escapeQuotationsInFilename(String filename) {
        if (filename.indexOf(34) == -1 && filename.indexOf(92) == -1) {
            return filename;
        }
        boolean escaped = false;
        StringBuilder sb = new StringBuilder();
        for (char c : filename.toCharArray()) {
            sb.append(c == '\"' && !escaped ? "\\\"" : Character.valueOf(c));
            escaped = !escaped && c == '\\';
        }
        if (escaped) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    private static String encodeFilename(String input, Charset charset) {
        Assert.notNull((Object)input, "`input` is required");
        Assert.notNull((Object)charset, "`charset` is required");
        Assert.isTrue(!Constants.US_ASCII.equals(charset), "ASCII does not require encoding");
        Assert.isTrue(Constants.UTF_8.equals(charset) || Constants.ISO_8859_1.equals(charset), "Only UTF-8 and ISO-8859-1 supported.");
        byte[] source = input.getBytes(charset);
        int len = source.length;
        StringBuilder sb = new StringBuilder(len << 1);
        sb.append(charset.name());
        sb.append("''");
        for (byte b : source) {
            if (ContentDisposition.isRFC5987AttrChar(b)) {
                sb.append((char)b);
                continue;
            }
            sb.append('%');
            char hex1 = Character.toUpperCase(Character.forDigit(b >> 4 & 0xF, 16));
            char hex2 = Character.toUpperCase(Character.forDigit(b & 0xF, 16));
            sb.append(hex1);
            sb.append(hex2);
        }
        return sb.toString();
    }

    private static class BuilderImpl
    implements Builder {
        private String type;
        @Nullable
        private String name;
        @Nullable
        private String filename;
        @Nullable
        private Charset charset;
        private Map<String, Value> attributes;

        public BuilderImpl(String type) {
            Assert.hasText(type, "'type' must not be not empty");
            this.type = type;
        }

        @Override
        public Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public Builder filename(String filename) {
            Assert.hasText(filename, "No filename");
            this.filename = filename;
            return this;
        }

        @Override
        public Builder filename(String filename, Charset charset) {
            Assert.hasText(filename, "No filename");
            this.filename = filename;
            this.charset = charset;
            return this;
        }

        @Override
        public Builder attributes(Map<String, Value> attributes) {
            this.attributes = attributes;
            return this;
        }

        @Override
        public ContentDisposition build() {
            return new ContentDisposition(this.type, this.name, this.filename, this.charset, this.attributes);
        }
    }

    public static interface Builder {
        public Builder name(String var1);

        public Builder filename(String var1);

        public Builder filename(String var1, @Nullable Charset var2);

        public Builder attributes(Map<String, Value> var1);

        public ContentDisposition build();
    }
}

