/*
 * Copyright Camunda Services GmbH and/or licensed to Camunda Services GmbH under
 * one or more contributor license agreements. Licensed under a proprietary license. See the
 * License.txt file for more information. You may not use this file except in compliance with the
 * proprietary license.
 */

package io.camunda.identity.sdk.annotation;

import io.camunda.identity.sdk.IdentityConfiguration;
import io.camunda.identity.sdk.exception.InvalidConfigurationException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import org.apache.commons.lang3.Validate;

public class AnnotationProcessor {
  public static <T> T apply(
      final IdentityConfiguration configuration, final Class<T> inf, final T object
  ) {
    return (T) Proxy.newProxyInstance(
        inf.getClassLoader(),
        new Class[] {inf},
        (proxy, method, args) -> {
          try {
            if (method.isAnnotationPresent(RequiresOAuthCredentials.class)) {
              validateOAuthCredentials(method, configuration);
            }
            if (method.isAnnotationPresent(RequiresBaseUrl.class)) {
              validateBaseUrl(method, configuration);
            }

            return method.invoke(object, args);
          } catch (final InvocationTargetException e) {
            throw e.getCause();
          }
        });
  }

  private static void validateOAuthCredentials(
      final Method method, final IdentityConfiguration configuration
  ) {
    try {
      Validate.notBlank(configuration.getIssuer(), "issuer must not be empty");
      Validate.notBlank(configuration.getIssuerBackendUrl(), "issuerBackendUrl must not be empty");
      Validate.notBlank(configuration.getClientId(), "clientId must not be empty");
      Validate.notBlank(configuration.getClientSecret(), "clientSecret must not be empty");
    } catch (final Throwable throwable) {
      throw new InvalidConfigurationException(
          String.format("'%s' can only be used if OAuth credentials are provided",
                        method.getName()), throwable);
    }
  }

  private static void validateBaseUrl(
      final Method method, final IdentityConfiguration configuration
  ) {
    try {
      Validate.notBlank(configuration.getBaseUrl(), "baseUrl must not be empty");
    } catch (final Throwable throwable) {
      throw new InvalidConfigurationException(
          String.format("'%s' can only be used if baseUrl is provided", method.getName()),
          throwable);
    }
  }
}
