/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.maven.docker.service;

import com.google.common.collect.ImmutableMap;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import io.fabric8.maven.docker.access.BuildOptions;
import io.fabric8.maven.docker.access.DockerAccess;
import io.fabric8.maven.docker.access.DockerAccessException;
import io.fabric8.maven.docker.config.AssemblyConfiguration;
import io.fabric8.maven.docker.config.BuildImageConfiguration;
import io.fabric8.maven.docker.config.CleanupMode;
import io.fabric8.maven.docker.config.ImageConfiguration;
import io.fabric8.maven.docker.model.ImageArchiveManifest;
import io.fabric8.maven.docker.model.ImageArchiveManifestEntry;
import io.fabric8.maven.docker.service.ArchiveService;
import io.fabric8.maven.docker.service.ImagePullManager;
import io.fabric8.maven.docker.service.QueryService;
import io.fabric8.maven.docker.service.RegistryService;
import io.fabric8.maven.docker.util.DockerFileUtil;
import io.fabric8.maven.docker.util.EnvUtil;
import io.fabric8.maven.docker.util.ImageArchiveUtil;
import io.fabric8.maven.docker.util.ImageName;
import io.fabric8.maven.docker.util.Logger;
import io.fabric8.maven.docker.util.MojoParameters;
import io.fabric8.maven.docker.util.NamePatternUtil;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.PatternSyntaxException;
import org.apache.maven.plugin.MojoExecutionException;

public class BuildService {
    private final String argPrefix = "docker.buildArg.";
    private final DockerAccess docker;
    private final QueryService queryService;
    private final ArchiveService archiveService;
    private final RegistryService registryService;
    private final Logger log;

    BuildService(DockerAccess docker, QueryService queryService, RegistryService registryService, ArchiveService archiveService, Logger log) {
        this.docker = docker;
        this.queryService = queryService;
        this.registryService = registryService;
        this.archiveService = archiveService;
        this.log = log;
    }

    public void buildImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, BuildContext buildContext, File buildArchiveFile) throws DockerAccessException, MojoExecutionException {
        if (imagePullManager != null) {
            this.autoPullBaseImage(imageConfig, imagePullManager, buildContext);
            this.autoPullCacheFromImage(imageConfig, imagePullManager, buildContext);
        }
        this.buildImage(imageConfig, buildContext.getMojoParameters(), this.checkForNocache(imageConfig), this.checkForSquash(imageConfig), this.addBuildArgs(buildContext), buildArchiveFile);
    }

    public File buildArchive(ImageConfiguration imageConfiguration, BuildContext buildContext, String archivePath) throws MojoExecutionException {
        String imageName = imageConfiguration.getName();
        ImageName.validate(imageName);
        BuildImageConfiguration buildConfig = imageConfiguration.getBuildConfiguration();
        MojoParameters params = buildContext.getMojoParameters();
        if (buildConfig.getDockerArchive() != null) {
            return buildConfig.getAbsoluteDockerTarPath(params);
        }
        long time = System.currentTimeMillis();
        File dockerArchive = this.archiveService.createArchive(imageName, buildConfig, params, this.log);
        this.log.info("%s: Created %s in %s", imageConfiguration.getDescription(), dockerArchive.getName(), EnvUtil.formatDurationTill(time));
        try {
            this.copyDockerArchive(imageConfiguration, dockerArchive, archivePath);
        }
        catch (IOException exception) {
            throw new MojoExecutionException("Error while copying created tar to specified buildArchive path: " + archivePath, (Exception)exception);
        }
        return dockerArchive;
    }

    public void copyDockerArchive(ImageConfiguration imageConfiguration, File dockerArchive, String archivePath) throws IOException {
        if (archivePath != null && !archivePath.isEmpty()) {
            Files.copy(dockerArchive.toPath(), new File(archivePath, dockerArchive.getName()).toPath(), new CopyOption[0]);
            this.log.info("%s: Copied created tarball to %s", imageConfiguration.getDescription(), archivePath);
        }
    }

    public void tagImage(ImageConfiguration imageConfig) throws DockerAccessException {
        List<String> tags = imageConfig.getBuildConfiguration().getTags();
        if (!tags.isEmpty()) {
            String imageName = imageConfig.getName();
            this.log.info("%s: Tag with %s", imageConfig.getDescription(), EnvUtil.stringJoin(tags, ","));
            BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
            for (String tag : tags) {
                this.tagImage(imageName, tag, null, buildConfig.cleanupMode());
            }
        }
    }

    protected void buildImage(ImageConfiguration imageConfig, MojoParameters params, boolean noCache, boolean squash, Map<String, String> buildArgs, File dockerArchive) throws DockerAccessException, MojoExecutionException {
        String imageName = imageConfig.getName();
        ImageName.validate(imageName);
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        String oldImageId = null;
        CleanupMode cleanupMode = buildConfig.cleanupMode();
        if (cleanupMode.isRemove()) {
            oldImageId = this.queryService.getImageId(imageName);
        }
        if (buildConfig.getDockerArchive() != null) {
            File tarArchive = buildConfig.getAbsoluteDockerTarPath(params);
            String archiveImageName = this.getArchiveImageName(buildConfig, tarArchive);
            long time = System.currentTimeMillis();
            this.docker.loadImage(imageName, tarArchive);
            this.log.info("%s: Loaded tarball in %s", buildConfig.getDockerArchive(), EnvUtil.formatDurationTill(time));
            if (archiveImageName != null && !archiveImageName.equals(imageName)) {
                this.docker.tag(archiveImageName, imageName, true);
            }
            return;
        }
        Map<String, String> mergedBuildMap = this.prepareBuildArgs(buildArgs, buildConfig);
        BuildOptions opts = new BuildOptions(buildConfig.getBuildOptions()).dockerfile(this.getDockerfileName(buildConfig)).forceRemove(cleanupMode.isRemove()).noCache(noCache).squash(squash).cacheFrom(buildConfig.getCacheFrom()).network(buildConfig.getNetwork()).buildArgs(mergedBuildMap);
        String newImageId = this.doBuildImage(imageName, dockerArchive, opts);
        this.log.info("%s: Built image %s", imageConfig.getDescription(), newImageId);
        this.removeDanglingImage(imageName, oldImageId, newImageId, cleanupMode, true);
    }

    public void tagImage(String imageName, String tag, String repo, CleanupMode cleanupMode) throws DockerAccessException {
        if (tag != null) {
            String fullImageName = new ImageName(imageName, tag).getNameWithOptionalRepository(repo);
            String oldImageId = null;
            if (cleanupMode.isRemove()) {
                oldImageId = this.queryService.getImageId(fullImageName);
            }
            this.docker.tag(imageName, fullImageName, true);
            this.log.info("Tagging image %s successful!", fullImageName);
            String newImageId = this.queryService.getImageId(fullImageName);
            this.removeDanglingImage(fullImageName, oldImageId, newImageId, cleanupMode, false);
        }
    }

    private Map<String, String> prepareBuildArgs(Map<String, String> buildArgs, BuildImageConfiguration buildConfig) {
        ImmutableMap.Builder builder = ImmutableMap.builder().putAll(buildArgs);
        if (buildConfig.getArgs() != null) {
            builder.putAll(buildConfig.getArgs());
        }
        return builder.build();
    }

    private String getArchiveImageName(BuildImageConfiguration buildConfig, File tarArchive) throws MojoExecutionException {
        String archiveImageName;
        ImageArchiveManifest manifest;
        if (buildConfig.getLoadNamePattern() == null || buildConfig.getLoadNamePattern().length() == 0) {
            return null;
        }
        try {
            manifest = this.readArchiveManifest(tarArchive);
        }
        catch (JsonParseException | IOException e) {
            throw new MojoExecutionException("Unable to read image manifest in archive " + buildConfig.getDockerArchive(), (Exception)e);
        }
        try {
            archiveImageName = this.matchArchiveImagesToPattern(buildConfig.getLoadNamePattern(), manifest);
        }
        catch (PatternSyntaxException e) {
            throw new MojoExecutionException("Unable to interpret loadNamePattern " + buildConfig.getLoadNamePattern(), (Exception)e);
        }
        if (archiveImageName == null) {
            throw new MojoExecutionException("No image in the archive has a tag that matches pattern " + buildConfig.getLoadNamePattern());
        }
        return archiveImageName;
    }

    private ImageArchiveManifest readArchiveManifest(File tarArchive) throws IOException, JsonParseException {
        long time = System.currentTimeMillis();
        ImageArchiveManifest manifest = ImageArchiveUtil.readManifest(tarArchive);
        this.log.info("%s: Read archive manifest in %s", tarArchive, EnvUtil.formatDurationTill(time));
        if (this.log.isDebugEnabled()) {
            for (ImageArchiveManifestEntry entry : manifest.getEntries()) {
                this.log.debug("Entry ID: %s has %d repo tag(s)", entry.getId(), entry.getRepoTags().size());
                for (String repoTag : entry.getRepoTags()) {
                    this.log.debug("Repo Tag: %s", repoTag);
                }
            }
        }
        return manifest;
    }

    private String matchArchiveImagesToPattern(String imageNamePattern, ImageArchiveManifest manifest) {
        String imageNameRegex = NamePatternUtil.convertNamePattern(imageNamePattern);
        this.log.debug("Image name regex is %s", imageNameRegex);
        Map<String, ImageArchiveManifestEntry> entries = ImageArchiveUtil.findEntriesByRepoTagPattern(imageNameRegex, manifest);
        if (this.log.isDebugEnabled()) {
            for (Map.Entry<String, ImageArchiveManifestEntry> entry : entries.entrySet()) {
                this.log.debug("Repo tag pattern matched %s referring to image %s", entry.getKey(), entry.getValue().getId());
            }
        }
        if (!entries.isEmpty()) {
            Map.Entry<String, ImageArchiveManifestEntry> matchedEntry = entries.entrySet().iterator().next();
            if (ImageArchiveUtil.mapEntriesById(entries.values()).size() > 1) {
                this.log.warn("Multiple image ids matched pattern %s: using tag %s associated with id %s", imageNamePattern, matchedEntry.getKey(), matchedEntry.getValue().getId());
            } else {
                this.log.info("Using image tag %s from archive", matchedEntry.getKey());
            }
            return matchedEntry.getKey();
        }
        return null;
    }

    private String getDockerfileName(BuildImageConfiguration buildConfig) {
        if (buildConfig.isDockerFileMode()) {
            return buildConfig.getDockerFile().getName();
        }
        return null;
    }

    private String doBuildImage(String imageName, File dockerArchive, BuildOptions options) throws DockerAccessException, MojoExecutionException {
        this.docker.buildImage(imageName, dockerArchive, options);
        return this.queryService.getImageId(imageName);
    }

    private Map<String, String> addBuildArgs(BuildContext buildContext) {
        Map<String, String> buildArgsFromProject = this.addBuildArgsFromProperties(buildContext.getMojoParameters().getProject().getProperties());
        Map<String, String> buildArgsFromSystem = this.addBuildArgsFromProperties(System.getProperties());
        Map<String, String> buildArgsFromDockerConfig = this.addBuildArgsFromDockerConfig();
        return ImmutableMap.builder().putAll(buildArgsFromDockerConfig).putAll(buildContext.getBuildArgs() != null ? buildContext.getBuildArgs() : Collections.emptyMap()).putAll(buildArgsFromProject).putAll(buildArgsFromSystem).build();
    }

    private Map<String, String> addBuildArgsFromProperties(Properties properties) {
        HashMap<String, String> buildArgs = new HashMap<String, String>();
        for (Object keyObj : properties.keySet()) {
            String key = (String)keyObj;
            if (!key.startsWith("docker.buildArg.")) continue;
            String argKey = key.replaceFirst("docker.buildArg.", "");
            String value = properties.getProperty(key);
            if (this.isEmpty(value)) continue;
            buildArgs.put(argKey, value);
        }
        this.log.debug("Build args set %s", buildArgs);
        return buildArgs;
    }

    private Map<String, String> addBuildArgsFromDockerConfig() {
        JsonObject proxies;
        JsonObject dockerConfig = DockerFileUtil.readDockerConfig();
        if (dockerConfig == null) {
            return Collections.emptyMap();
        }
        HashMap<String, String> buildArgs = new HashMap<String, String>();
        if (dockerConfig.has("proxies") && (proxies = dockerConfig.getAsJsonObject("proxies")).has("default")) {
            JsonObject defaultProxyObj = proxies.getAsJsonObject("default");
            String[] proxyMapping = new String[]{"httpProxy", "http_proxy", "httpsProxy", "https_proxy", "noProxy", "no_proxy", "ftpProxy", "ftp_proxy"};
            for (int index = 0; index < proxyMapping.length; index += 2) {
                if (!defaultProxyObj.has(proxyMapping[index])) continue;
                buildArgs.put(proxyMapping[index + 1], defaultProxyObj.get(proxyMapping[index]).getAsString());
            }
        }
        this.log.debug("Build args set %s", buildArgs);
        return buildArgs;
    }

    private void autoPullBaseImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, BuildContext buildContext) throws DockerAccessException, MojoExecutionException {
        List<Object> fromImages;
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        CleanupMode cleanupMode = buildConfig.cleanupMode();
        if (buildConfig.getDockerArchive() != null) {
            return;
        }
        if (buildConfig.isDockerFileMode()) {
            fromImages = this.extractBaseFromDockerfile(buildConfig, buildContext);
        } else {
            fromImages = new LinkedList();
            String baseImage = this.extractBaseFromConfiguration(buildConfig);
            if (baseImage != null) {
                fromImages.add(this.extractBaseFromConfiguration(buildConfig));
            }
        }
        for (String string : fromImages) {
            if (string == null || "scratch".equals(string)) continue;
            String oldImageId = null;
            if (cleanupMode.isRemove()) {
                oldImageId = this.queryService.getImageId(string);
            }
            this.registryService.pullImageWithPolicy(string, imagePullManager, buildContext.getRegistryConfig(), buildConfig);
            String newImageId = this.queryService.getImageId(string);
            this.removeDanglingImage(string, oldImageId, newImageId, cleanupMode, false);
        }
    }

    private void autoPullCacheFromImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, BuildContext buildContext) throws DockerAccessException, MojoExecutionException {
        if (imageConfig.getBuildConfiguration().getCacheFrom() == null) {
            return;
        }
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        CleanupMode cleanupMode = buildConfig.cleanupMode();
        for (String cacheFromImage : buildConfig.getCacheFrom()) {
            String oldImageId = null;
            if (cleanupMode.isRemove()) {
                oldImageId = this.queryService.getImageId(cacheFromImage);
            }
            try {
                this.registryService.pullImageWithPolicy(cacheFromImage, imagePullManager, buildContext.getRegistryConfig(), buildConfig);
            }
            catch (DockerAccessException e) {
                this.log.warn("Could not pull cacheFrom image: '%s'. Reason: %s", cacheFromImage, e.getMessage());
            }
            String newImageId = this.queryService.getImageId(cacheFromImage);
            this.removeDanglingImage(cacheFromImage, oldImageId, newImageId, cleanupMode, false);
        }
    }

    private String extractBaseFromConfiguration(BuildImageConfiguration buildConfig) {
        List<AssemblyConfiguration> assemblyConfig;
        String fromImage = buildConfig.getFrom();
        if (fromImage == null && (assemblyConfig = buildConfig.getAssemblyConfigurations()).isEmpty()) {
            fromImage = "busybox:latest";
        }
        return fromImage;
    }

    private List<String> extractBaseFromDockerfile(BuildImageConfiguration buildConfig, BuildContext buildContext) {
        List<String> fromImage;
        try {
            File fullDockerFilePath = buildConfig.getAbsoluteDockerFilePath(buildContext.getMojoParameters());
            fromImage = DockerFileUtil.extractBaseImages(fullDockerFilePath, DockerFileUtil.createInterpolator(buildContext.getMojoParameters(), buildConfig.getFilter()), buildConfig.getArgs());
        }
        catch (IOException e) {
            return Collections.emptyList();
        }
        return fromImage;
    }

    private void removeDanglingImage(String oldImageName, String oldImageId, String newImageId, CleanupMode cleanupMode, boolean force) throws DockerAccessException {
        if (oldImageId != null && !oldImageId.equals(newImageId)) {
            if (force) {
                this.removeImage(oldImageName, oldImageId, cleanupMode, true);
            } else {
                List<String> oldImageTags = this.docker.getImageTags(oldImageId);
                if (oldImageTags != null) {
                    if (oldImageTags.isEmpty()) {
                        this.removeImage(oldImageName, oldImageId, cleanupMode, false);
                    } else {
                        this.log.warn("%s: Skipped removing image %s; still tagged with: ", oldImageName, oldImageId, String.join((CharSequence)",", oldImageTags));
                    }
                }
            }
        }
    }

    private void removeImage(String oldImageName, String oldImageId, CleanupMode cleanupMode, boolean force) throws DockerAccessException {
        try {
            this.docker.removeImage(oldImageId, force);
            this.log.info("%s: Removed dangling image %s", oldImageName, oldImageId);
        }
        catch (DockerAccessException exp) {
            if (cleanupMode == CleanupMode.TRY_TO_REMOVE) {
                this.log.warn("%s: %s (dangling image)%s", oldImageName, exp.getMessage(), exp.getCause() != null ? " [" + exp.getCause().getMessage() + "]" : "");
            }
            throw exp;
        }
    }

    private boolean checkForNocache(ImageConfiguration imageConfig) {
        String noCache = System.getProperty("docker.noCache");
        if (noCache == null) {
            noCache = System.getProperty("docker.nocache");
        }
        if (noCache != null) {
            return noCache.length() == 0 || Boolean.valueOf(noCache) != false;
        }
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        return buildConfig.noCache();
    }

    private boolean checkForSquash(ImageConfiguration imageConfig) {
        String squash = System.getProperty("docker.squash");
        if (squash != null) {
            return squash.length() == 0 || Boolean.valueOf(squash) != false;
        }
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        return buildConfig.squash();
    }

    private boolean isEmpty(String str) {
        return str == null || str.isEmpty();
    }

    public static class BuildContext
    implements Serializable {
        private MojoParameters mojoParameters;
        private Map<String, String> buildArgs;
        private RegistryService.RegistryConfig registryConfig;

        public MojoParameters getMojoParameters() {
            return this.mojoParameters;
        }

        public Map<String, String> getBuildArgs() {
            return this.buildArgs;
        }

        public RegistryService.RegistryConfig getRegistryConfig() {
            return this.registryConfig;
        }

        public static class Builder {
            private BuildContext context;

            public Builder() {
                this.context = new BuildContext();
            }

            public Builder(BuildContext context) {
                this.context = context;
            }

            public Builder mojoParameters(MojoParameters mojoParameters) {
                this.context.mojoParameters = mojoParameters;
                return this;
            }

            public Builder buildArgs(Map<String, String> buildArgs) {
                this.context.buildArgs = buildArgs;
                return this;
            }

            public Builder registryConfig(RegistryService.RegistryConfig registryConfig) {
                this.context.registryConfig = registryConfig;
                return this;
            }

            public BuildContext build() {
                return this.context;
            }
        }
    }
}

