/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.kev;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.github.jeremylong.openvulnerability.client.DataFeed;
import io.github.jeremylong.openvulnerability.client.HttpClientSupplier;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.impl.classic.BasicHttpClientResponseHandler;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;

import java.io.IOException;

/**
 * Data Feed for the CISA Known Exploited Vulnerabilities Catalog.
 *
 * @see <a href=
 * "https://www.cisa.gov/known-exploited-vulnerabilities-catalog">https://www.cisa.gov/known-exploited-vulnerabilities-catalog</a>
 */
public class KevDataFeed implements DataFeed<KevCatalog> {
    private final static String DEFAULT_LOCATION = "https://www.cisa.gov/sites/default/files/feeds/known_exploited_vulnerabilities.json";
    /**
     * Jackson object mapper.
     */
    private final ObjectMapper objectMapper;
    private final HttpClientSupplier httpClientSupplier;
    private final String downloadUrl;

    public KevDataFeed() {
        this(DEFAULT_LOCATION);
    }

    public KevDataFeed(String downloadUrl) {
        this(downloadUrl, null);
    }

    public KevDataFeed(String downloadUrl, HttpClientSupplier httpClientSupplier) {
        this.downloadUrl = downloadUrl;
        this.httpClientSupplier = httpClientSupplier != null ? httpClientSupplier : HttpClientSupplier.getDefault();
        objectMapper = new ObjectMapper();
        objectMapper.registerModule(new JavaTimeModule());
    }

    @Override
    public KevCatalog download() {
        HttpGet request = new HttpGet(downloadUrl);
        String json;
        try (CloseableHttpClient client = httpClientSupplier.get()) {
            json = client.execute(request, new BasicHttpClientResponseHandler());
        } catch (IOException e) {
            throw new KevException("Unable to download the Known Exploitable Vulnerability Catalog", e);
        }
        try {
            return objectMapper.readValue(json, KevCatalog.class);
        } catch (JsonProcessingException e) {
            throw new KevException("Failed to parse JSON starting with: \"" + json.substring(0, 100) + "\"", e);
        }
    }
}
