/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.features;

import io.helidon.common.NativeImageHelper;
import io.helidon.common.features.FeatureCatalog;
import io.helidon.common.features.FeatureDescriptor;
import io.helidon.common.features.api.HelidonFlavor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

public final class HelidonFeatures {
    static final AtomicBoolean PRINTED = new AtomicBoolean();
    static final AtomicReference<HelidonFlavor> CURRENT_FLAVOR = new AtomicReference();
    private static final System.Logger LOGGER = System.getLogger(HelidonFeatures.class.getName());
    private static final System.Logger INCUBATING = System.getLogger(HelidonFeatures.class.getName() + ".incubating");
    private static final System.Logger PREVIEW = System.getLogger(HelidonFeatures.class.getName() + ".preview");
    private static final System.Logger DEPRECATED = System.getLogger(HelidonFeatures.class.getName() + ".deprecated");
    private static final System.Logger INVALID = System.getLogger(HelidonFeatures.class.getName() + ".invalid");
    private static final AtomicBoolean SCANNED = new AtomicBoolean();
    private static final Map<HelidonFlavor, Set<FeatureDescriptor>> FEATURES = new EnumMap<HelidonFlavor, Set<FeatureDescriptor>>(HelidonFlavor.class);
    private static final Map<HelidonFlavor, Map<String, Node>> ROOT_FEATURE_NODES = new EnumMap<HelidonFlavor, Map<String, Node>>(HelidonFlavor.class);
    private static final List<FeatureDescriptor> ALL_FEATURES = new LinkedList<FeatureDescriptor>();

    private HelidonFeatures() {
    }

    public static void print(HelidonFlavor flavor, String version, boolean details) {
        new Thread(() -> HelidonFeatures.features(flavor, version, details), "features-thread").start();
    }

    public static void nativeBuildTime(ClassLoader classLoader) {
        HelidonFeatures.scan(classLoader);
        for (FeatureDescriptor feat : ALL_FEATURES) {
            if (!feat.nativeSupported()) {
                LOGGER.log(System.Logger.Level.ERROR, "Feature '" + feat.name() + "' for path '" + feat.stringPath() + "' IS NOT SUPPORTED in native image. Image may still build and run.");
                continue;
            }
            if (feat.nativeDescription().isBlank()) continue;
            LOGGER.log(System.Logger.Level.WARNING, "Feature '" + feat.name() + "' for path '" + feat.stringPath() + "' has limited support in native image: " + feat.nativeDescription());
        }
    }

    public static void flavor(HelidonFlavor flavor) {
        CURRENT_FLAVOR.compareAndSet(null, flavor);
    }

    static Node ensureNode(Map<String, Node> rootFeatureNodes, String ... path) {
        if (path.length == 1) {
            return rootFeatureNodes.computeIfAbsent(path[0], Node::new);
        }
        Node lastNode = HelidonFeatures.ensureNode(rootFeatureNodes, path[0]);
        for (int i = 1; i < path.length; ++i) {
            String pathElement = path[i];
            lastNode = HelidonFeatures.ensureNode(pathElement, lastNode);
        }
        return lastNode;
    }

    static Node ensureNode(String name, Node parent) {
        return parent.children.computeIfAbsent(name, it -> new Node(name));
    }

    private static void register(FeatureDescriptor featureDescriptor) {
        for (HelidonFlavor flavor : featureDescriptor.flavors()) {
            String[] path = featureDescriptor.path();
            if (path.length == 1) {
                FEATURES.computeIfAbsent(flavor, key -> new TreeSet<FeatureDescriptor>(Comparator.comparing(FeatureDescriptor::name))).add(featureDescriptor);
            }
            Map rootFeatures = ROOT_FEATURE_NODES.computeIfAbsent(flavor, it -> new TreeMap(String.CASE_INSENSITIVE_ORDER));
            Node node = HelidonFeatures.ensureNode(rootFeatures, path);
            node.descriptor(featureDescriptor);
        }
        ALL_FEATURES.add(featureDescriptor);
    }

    static void features(HelidonFlavor flavor, String version, boolean details) {
        CURRENT_FLAVOR.compareAndSet(null, HelidonFlavor.SE);
        HelidonFlavor currentFlavor = CURRENT_FLAVOR.get();
        if (currentFlavor != flavor) {
            return;
        }
        if (!PRINTED.compareAndSet(false, true)) {
            return;
        }
        HelidonFeatures.scan(Thread.currentThread().getContextClassLoader());
        Set<FeatureDescriptor> features = FEATURES.get(currentFlavor);
        if (null == features) {
            LOGGER.log(System.Logger.Level.INFO, "Helidon " + String.valueOf(currentFlavor) + " " + version + " has no registered features");
        } else {
            String featureString = "[" + features.stream().map(FeatureDescriptor::name).collect(Collectors.joining(", ")) + "]";
            LOGGER.log(System.Logger.Level.INFO, "Helidon " + String.valueOf(currentFlavor) + " " + version + " features: " + featureString);
        }
        List<FeatureDescriptor> invalidFeatures = ALL_FEATURES.stream().filter(feature -> feature.not(currentFlavor)).collect(Collectors.toList());
        if (!invalidFeatures.isEmpty()) {
            INVALID.log(System.Logger.Level.WARNING, "Invalid modules are used:");
            invalidFeatures.forEach(HelidonFeatures::logInvalid);
        }
        if (details) {
            LOGGER.log(System.Logger.Level.INFO, "Detailed feature tree:");
            if (FEATURES.containsKey(currentFlavor)) {
                FEATURES.get(currentFlavor).forEach(feature -> HelidonFeatures.printDetails(feature.name(), ROOT_FEATURE_NODES.get(currentFlavor).get(feature.path()[0]), 0));
            }
        } else {
            ArrayList allIncubating = new ArrayList();
            ArrayList allDeprecated = new ArrayList();
            ArrayList allPreview = new ArrayList();
            if (ROOT_FEATURE_NODES.containsKey(currentFlavor) && FEATURES.containsKey(currentFlavor)) {
                FEATURES.get(currentFlavor).forEach(feature -> {
                    HelidonFeatures.gatherIncubating(allIncubating, ROOT_FEATURE_NODES.get(currentFlavor).get(feature.path()[0]));
                    HelidonFeatures.gatherDeprecated(allDeprecated, ROOT_FEATURE_NODES.get(currentFlavor).get(feature.path()[0]));
                    HelidonFeatures.gatherPreview(allPreview, ROOT_FEATURE_NODES.get(currentFlavor).get(feature.path()[0]));
                });
            }
            if (!allIncubating.isEmpty()) {
                INCUBATING.log(System.Logger.Level.WARNING, "You are using incubating features. These APIs are not production ready!");
                allIncubating.forEach(it -> INCUBATING.log(System.Logger.Level.INFO, "\tIncubating feature: " + it.name() + " since " + it.since() + " (" + it.stringPath() + ")"));
            }
            if (!allDeprecated.isEmpty()) {
                DEPRECATED.log(System.Logger.Level.WARNING, "You are using deprecated features. These APIs will be removed from Helidon!");
                allDeprecated.forEach(it -> DEPRECATED.log(System.Logger.Level.INFO, "\tDeprecated feature: " + it.name() + " since " + it.deprecatedSince() + " (" + it.stringPath() + ")"));
            }
            if (!allDeprecated.isEmpty()) {
                PREVIEW.log(System.Logger.Level.INFO, "You are using preview features. These APIs are production ready, yet may change more frequently. Please follow Helidon release changelog!");
                allPreview.forEach(it -> PREVIEW.log(System.Logger.Level.INFO, "\tPreview feature: " + it.name() + " since " + it.since() + " (" + it.stringPath() + ")"));
            }
        }
    }

    private static void logInvalid(FeatureDescriptor feature) {
        INVALID.log(System.Logger.Level.WARNING, "\tModule \"" + feature.module() + "\" (" + feature.stringPath() + ") is not designed for Helidon " + String.valueOf(CURRENT_FLAVOR.get()) + ", it should only be used in Helidon " + Arrays.toString(feature.flavors()));
    }

    private static void gatherIncubating(List<FeatureDescriptor> allIncubating, Node node) {
        if (node.descriptor != null && node.descriptor.incubating()) {
            allIncubating.add(node.descriptor);
        }
        node.children().values().forEach(it -> HelidonFeatures.gatherIncubating(allIncubating, it));
    }

    private static void gatherPreview(List<FeatureDescriptor> allPreview, Node node) {
        if (node.descriptor != null && node.descriptor.preview()) {
            allPreview.add(node.descriptor);
        }
        node.children().values().forEach(it -> HelidonFeatures.gatherPreview(allPreview, it));
    }

    private static void gatherDeprecated(List<FeatureDescriptor> allDeprecated, Node node) {
        if (node.descriptor != null && node.descriptor.deprecated()) {
            allDeprecated.add(node.descriptor);
        }
        node.children().values().forEach(it -> HelidonFeatures.gatherDeprecated(allDeprecated, it));
    }

    private static void scan(ClassLoader classLoader) {
        if (!SCANNED.compareAndSet(false, true)) {
            return;
        }
        FeatureCatalog.features(classLoader).forEach(HelidonFeatures::register);
        if (NativeImageHelper.isRuntime()) {
            for (FeatureDescriptor feature : ALL_FEATURES) {
                String desc = feature.nativeDescription();
                if (feature.nativeSupported()) {
                    if (desc == null || desc.isBlank()) continue;
                    LOGGER.log(System.Logger.Level.WARNING, "Native image for feature " + feature.name() + "(" + feature.stringPath() + "): " + desc);
                    continue;
                }
                if (desc == null || desc.isBlank()) {
                    LOGGER.log(System.Logger.Level.ERROR, "You are using a feature not supported in native image: " + feature.name() + "(" + feature.stringPath() + ")");
                    continue;
                }
                LOGGER.log(System.Logger.Level.ERROR, "You are using a feature not supported in native image: " + feature.name() + "(" + feature.stringPath() + "): " + desc);
            }
        }
    }

    private static void printDetails(String name, Node node, int level) {
        FeatureDescriptor feat = node.descriptor;
        if (feat == null) {
            System.out.println("  ".repeat(level) + name);
        } else {
            String prefix = " ".repeat(level * 2);
            int len = prefix.length() + name.length();
            String suffix = len <= 18 ? " ".repeat(20 - len) : "\t";
            String preview = feat.preview() ? "Preview - " : "";
            String incubating = feat.incubating() ? "Incubating - " : "";
            String deprecated = feat.deprecated() ? "Deprecated since " + feat.deprecatedSince() + " - " : "";
            Object nativeDesc = "";
            if (!feat.nativeSupported()) {
                nativeDesc = " (NOT SUPPORTED in native image)";
            } else if (!feat.nativeDescription().isBlank()) {
                nativeDesc = " (Native image: " + feat.nativeDescription() + ")";
            }
            System.out.println(prefix + name + suffix + deprecated + incubating + preview + feat.description() + (String)nativeDesc);
        }
        node.children.forEach((childName, childNode) -> {
            FeatureDescriptor descriptor = childNode.descriptor;
            String actualName = descriptor == null ? childName : descriptor.name();
            HelidonFeatures.printDetails(actualName, childNode, level + 1);
        });
    }

    static final class Node {
        private final Map<String, Node> children = new TreeMap<String, Node>(String.CASE_INSENSITIVE_ORDER);
        private final String name;
        private FeatureDescriptor descriptor;

        Node(String name) {
            this.name = name;
        }

        String name() {
            return this.name;
        }

        Map<String, Node> children() {
            return this.children;
        }

        void descriptor(FeatureDescriptor featureDescriptor) {
            this.descriptor = featureDescriptor;
        }
    }
}

