'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.defaultLngDetector = undefined;
exports.i18nTranslator = i18nTranslator;
exports.enableMocksForI18n = enableMocksForI18n;
exports.disableMocksForI18n = disableMocksForI18n;

var _i18next = require('i18next');

var _i18next2 = _interopRequireDefault(_i18next);

var _i18nextBrowserLanguagedetector = require('i18next-browser-languagedetector');

var _i18nextBrowserLanguagedetector2 = _interopRequireDefault(_i18nextBrowserLanguagedetector);

var _jsExtensions = require('@jenkins-cd/js-extensions');

var _i18nextXhrBackend = require('i18next-xhr-backend');

var _i18nextXhrBackend2 = _interopRequireDefault(_i18nextXhrBackend);

var _urlconfig = require('../urlconfig');

var _logging = require('../logging');

var _fetch = require('../fetch');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var logger = _logging.logging.logger('io.jenkins.blueocean.i18n');

/**
 * Init language detector, we are going to use first queryString and then the navigator prefered language
 */
var defaultLngDetector = exports.defaultLngDetector = new _i18nextBrowserLanguagedetector2.default(null, {
    // order and from where user language should be detected
    order: ['querystring', 'htmlTag', 'navigator'],
    // keys or params to lookup language from
    lookupQuerystring: 'language',
    // Don't use the default (document.documentElement) because that can
    // trigger the browsers auto-translate, which is quite annoying.
    htmlTag: window.document ? window.document.head : undefined
});
var prefix = _urlconfig.UrlConfig.getJenkinsRootURL() || '';
var FALLBACK_LANG = '';

function newPluginXHR(pluginName) {
    var pluginVersion = _jsExtensions.store.getPluginVersion(pluginName);

    if (!pluginVersion) {
        // if we do not have a version we may have an alias to resolve a resourceBUndle
        throw new Error('Unable to create an i18n instance for plugin "' + pluginName + '". This plugin is not currently installed, or is disabled.');
    }

    pluginVersion = encodeURIComponent(pluginVersion);

    var loadPath = prefix + '/blue/rest/i18n/' + pluginName + '/' + pluginVersion + '/{ns}/{lng}';
    return new _i18nextXhrBackend2.default(null, {
        loadPath: loadPath,
        allowMultiLoading: false,
        ajax: function ajax(url, options, callback) {
            if (logger.isDebugEnabled()) {
                logger.debug('loading data for', url);
            }
            var fetchOptions = {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json'
                }
            };
            _fetch.Fetch.fetchJSON(url, { disableCapabilities: true, disableLoadingIndicator: true, ignoreRefreshHeader: true, fetchOptions: fetchOptions }).then(function (data) {
                callback(data, { status: 200 });
            });
        },
        parse: function parse(response) {
            if (logger.isDebugEnabled()) {
                logger.debug('Received i18n resource bundle for plugin "%s".', pluginName, response.data);
            }
            return response.data;
        }
    });
}

/**
 * Create a instance of i18next and init it
 * in case we are in test mode and run unit test, we deliver a i18next instance that are not using any backend nor language detection
 * @param backend  {object} - the backend XHR invoker we want to use
 * @param lngDetector {object} - the component that detects which language we want to display
 * @param options {object} - general options for i18next
 * @see defaultOptions
 */
var i18nextInstance = function i18nextInstance(backend) {
    var lngDetector = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : defaultLngDetector;
    var options = arguments[2];

    if (!backend) {
        throw new Error('Invalid call to create a new i18next instance. No backend XHR invoker supplied.');
    }
    if (!options) {
        throw new Error('Invalid call to create a new i18next instance. No i18next options supplied.');
    }
    return _i18next2.default.createInstance().use(backend).use(lngDetector).init(options);
};

var translatorCache = {};
var useMockFallback = false;

var assertPluginNameDefined = function assertPluginNameDefined(pluginName) {
    if (!pluginName) {
        throw new Error('"pluginName" arg cannot be null/blank');
    }
};

var toDefaultNamespace = function toDefaultNamespace(pluginName) {
    assertPluginNameDefined(pluginName);
    // Replace all hyphen chars with a dot.
    return 'jenkins.plugins.' + pluginName.replace(/-/g, '.') + '.Messages';
};

/**
 * Create an i18next instance for accessing i18n resource bundles
 * in the named plugin namespace.
 * @param pluginName The name of the plugin.
 * @param namespace The resource bundle namespace. Optional, defaulting to
 * the plugin's default resource bundle e.g. "jenkins.plugins.blueocean.web.Messages"
 * for the "blueocean-web" plugin and "jenkins.plugins.blueocean.dashboard.Messages"
 * for the "blueocean-dashboard" plugin.
 * @return An i18n instance.
 */
var pluginI18next = function pluginI18next(pluginName) {
    var namespace = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : toDefaultNamespace(pluginName);

    assertPluginNameDefined(pluginName);

    var initOptions = {
        ns: [namespace],
        defaultNS: namespace,
        keySeparator: false, // we do not have any nested keys in properties files
        debug: false,
        fallbackLng: FALLBACK_LANG,
        load: 'currentOnly',
        interpolation: {
            prefix: '{',
            suffix: '}',
            escapeValue: false // not needed for react!!
        }
    };

    return i18nextInstance(newPluginXHR(pluginName), defaultLngDetector, initOptions);
};

function buildCacheKey(pluginName) {
    var namespace = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : toDefaultNamespace(pluginName);

    return pluginName + ':' + namespace;
}

/**
 * Create an i18n Translator instance for accessing i18n resource bundles
 * in the named plugin namespace.
 * @param pluginName The name of the plugin.
 * @param namespace The resource bundle namespace. Optional, defaulting to
 * the plugin's default resource bundle e.g. "jenkins.plugins.blueocean.web.Messages"
 * for the "blueocean-web" plugin and "jenkins.plugins.blueocean.dashboard.Messages"
 * for the "blueocean-dashboard" plugin.
 * @return An i18n Translator instance.
 */
function i18nTranslator(pluginName, namespace, onLoad) {
    assertPluginNameDefined(pluginName);

    var translatorCacheKey = buildCacheKey(pluginName, namespace);
    var translator = translatorCache[translatorCacheKey];

    if (translator) {
        if (typeof onLoad === 'function') {
            onLoad();
        }
        return translator;
    }

    // Lazily construct what we need instead of on creation
    return function translate(key, params) {
        if (useMockFallback) {
            return params && params.defaultValue || key;
        }

        if (!translator) {
            var I18n = pluginI18next(pluginName, namespace);
            if (typeof onLoad === 'function') {
                I18n.on('loaded', onLoad);
            }

            // Create and cache the translator instance.
            var detectedLang = void 0;
            try {
                detectedLang = defaultLngDetector.detect();
            } catch (e) {
                detectedLang = FALLBACK_LANG;
            }

            if (logger.isLogEnabled()) {
                logger.log('Translator instance created for "%s". Language detected as "%s".', translatorCacheKey, detectedLang);
            }

            var fixedT = I18n.getFixedT(detectedLang, namespace);
            translator = function _translate(i18nKey, i18nParams) {
                var normalizedKey = i18nKey.replace(/[\W]/g, '.');
                var passedParams = i18nParams;
                if (normalizedKey !== i18nKey) {
                    if (!passedParams) {
                        passedParams = {};
                    }
                    if (!passedParams.defaultValue) {
                        passedParams.defaultValue = i18nKey;
                    }
                    if (logger.isDebugEnabled()) {
                        logger.debug('Normalized i18n key "' + i18nKey + '" to "' + normalizedKey + '".');
                    }
                }
                return fixedT(normalizedKey, passedParams);
            };
            translatorCache[translatorCacheKey] = translator;
        }

        if (key) {
            return translator(key, params);
        }
        return undefined;
    };
}

function enableMocksForI18n() {
    useMockFallback = true;
}

function disableMocksForI18n() {
    useMockFallback = false;
}
//# sourceMappingURL=i18n.js.map
