"use strict";
/*********************************************************************************************
 **********************************************************************************************

 Builders to specific screens, REST resources, and classic functionality

 For general URL-wrangling helpers please see the UrlUtils module.

 **********************************************************************************************
 *********************************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
var config_1 = require("./config");
var UrlUtils_1 = require("./utils/UrlUtils");
/**
 * Return a new array with leading and trailing whitespace elements removed.
 */
function trimEmptyTokens(tokens) {
    var copy = tokens.slice();
    if (copy[0] === '') {
        copy.shift();
    }
    if (copy[copy.length - 1] === '') {
        copy.pop();
    }
    return copy;
}
/**
 * Parses the REST link to a pipeline run, extracts and unencodes the separate parts
 *
 * @param restUrl
 * @returns {{organizationName: string; fullName: string; detailName: string; runId: string}}
 */
function parseRestRunUrl(restUrl) {
    // FIXME: before exporting, make this more flexible so it can be a complete compliment to buildRestUrl
    var tokens = trimEmptyTokens(restUrl.split('/'));
    // given the following URL '/blue/rest/organizations/jenkins/pipelines/folder1/pipelines/folder2/pipelines/folder3/pipelines/jdl-2
    // /branches/experiment%252Fbuild-locally-docker/runs/21/
    var organizationName = decodeURIComponent(tokens[3]);
    var isMultiBranch = tokens[tokens.length - 4] === 'branches';
    var fullNameStart = 4;
    var fullNameEnd = isMultiBranch ? tokens.length - 4 : tokens.length - 2;
    // grab the tokens that make up the full name, then filter out the even values ('/pipelines')
    // so the clean folder path is returned, e.g. f1/pipelines/f2/pipelines/f3/pipelines/foo => f1/f2/f3/foo
    var pipelineFullName = tokens
        .slice(fullNameStart, fullNameEnd)
        .filter(function (name, index) { return index % 2 === 1; })
        .join('/');
    var pipelineName = tokens[fullNameEnd - 1];
    var branchName = isMultiBranch ? tokens[tokens.length - 3] : '';
    var runId = tokens[tokens.length - 1];
    var detailName = decodeURIComponent(isMultiBranch ? decodeURIComponent(branchName) : pipelineName);
    // fail fast
    if (!organizationName || !pipelineFullName || !detailName || !runId) {
        throw new Error('Could not extract URI components');
    }
    return {
        organizationName: organizationName,
        pipelineFullName: pipelineFullName,
        detailName: detailName,
        runId: runId,
    };
}
/**
 * Builds a run details view url from a RunDetails object
 */
function buildRunUrlForDetails(runDetails) {
    var restUrl = runDetails._links.self.href;
    return buildRunUrlForRestUrl(restUrl);
}
exports.buildRunUrlForDetails = buildRunUrlForDetails;
/**
 * Builds a run details view url from a run's REST link URL
 */
function buildRunUrlForRestUrl(restUrl) {
    var _a = parseRestRunUrl(restUrl), organizationName = _a.organizationName, pipelineFullName = _a.pipelineFullName, detailName = _a.detailName, runId = _a.runId;
    return buildRunUrl(organizationName, pipelineFullName, detailName, runId);
}
exports.buildRunUrlForRestUrl = buildRunUrlForRestUrl;
/**
 * Builds a run details view url from identifiers
 */
function buildRunUrl(organizationName, pipelineFullName, branchOrPipelineName, runId, tabName) {
    if (tabName === void 0) { tabName = 'pipeline'; }
    var baseUrl = "/organizations/" + encodeURIComponent(organizationName) +
        ("/" + encodeURIComponent(pipelineFullName)) +
        ("/detail/" + encodeURIComponent(branchOrPipelineName)) +
        ("/" + encodeURIComponent(runId));
    return tabName === null ? baseUrl : baseUrl + ("/" + tabName);
}
exports.buildRunUrl = buildRunUrl;
/**
 * Build a root-relative URL to the organization's pipeline list screen.
 * @param organization
 */
function buildOrganizationUrl(organization) {
    return "/organizations/" + encodeURIComponent(organization);
}
exports.buildOrganizationUrl = buildOrganizationUrl;
/**
 * Build a root-relative URL to the pipeline details screen.
 * @param organizationName
 * @param pipelineFullName
 * @param tabName
 * @returns {string}
 */
function buildPipelineUrl(organizationName, pipelineFullName, tabName) {
    var baseUrl = "/organizations/" + encodeURIComponent(organizationName) + "/" + ("" + encodeURIComponent(pipelineFullName));
    return tabName ? baseUrl + "/" + tabName : baseUrl;
}
exports.buildPipelineUrl = buildPipelineUrl;
/**
 * Gives classic jenkins job path prefix.
 * For organization group '/folder1/org1', job prefix is: /job/folder1/job/org1
 * For root organization group '/', there is no prefix: ''.
 * @param organizationGroupName organization group
 * @returns {string}
 */
function classicOrganizationRoot(organizationGroupName) {
    if (organizationGroupName && organizationGroupName !== '/') {
        return "" + organizationGroupName.split('/').join('/job/');
    }
    return '';
}
exports.classicOrganizationRoot = classicOrganizationRoot;
/**
 * The root of a classic URL for a specific pipeline using the default org group
 * @param pipelineFullName
 * @returns {string}
 */
function classicJobRoot(pipelineFullName) {
    var jenkinsUrl = config_1.AppConfig.getJenkinsRootURL();
    var orgRoot = classicOrganizationRoot(config_1.AppConfig.getOrganizationGroup());
    var classicFolderPath = pipelineFullName.split('/').join('/job/');
    return "" + jenkinsUrl + orgRoot + "/job/" + classicFolderPath;
}
exports.classicJobRoot = classicJobRoot;
/**
 * URL to "create job" page in classic
 */
function buildClassicCreateJobUrl() {
    var jenkinsUrl = config_1.AppConfig.getJenkinsRootURL();
    return "" + jenkinsUrl + classicOrganizationRoot(config_1.AppConfig.getOrganizationGroup()) + "/newJob";
}
exports.buildClassicCreateJobUrl = buildClassicCreateJobUrl;
/**
 * URL to the pipeline configuration screen in classic
 */
function buildClassicConfigUrl(pipelineDetails) {
    if (pipelineDetails && pipelineDetails.fullName) {
        return classicJobRoot(pipelineDetails.fullName) + "/configure";
    }
    return null;
}
exports.buildClassicConfigUrl = buildClassicConfigUrl;
/**
 * URL to the classic input sceen for a specific run
 */
function buildClassicInputUrl(pipelineDetails, branchName, runId) {
    if (pipelineDetails && pipelineDetails.fullName) {
        if (pipelineDetails.branchNames) {
            return classicJobRoot(pipelineDetails.fullName) + "/job/" + encodeURIComponent(branchName) + "/" + encodeURIComponent(runId) + "/input";
        }
        else {
            return classicJobRoot(pipelineDetails.fullName) + "/" + encodeURIComponent(runId) + "/input";
        }
    }
    return null;
}
exports.buildClassicInputUrl = buildClassicInputUrl;
/**
 * URL to classic to trigger a build
 *
 * example: http://localhost:8080/jenkins/job/scherler/job/Jenkins-40617-params/build?delay=0sec
 */
function buildClassicBuildUrl(pipelineDetails) {
    if (pipelineDetails && pipelineDetails.fullName) {
        return classicJobRoot(pipelineDetails.fullName) + "/build?delay=0sec";
    }
    return null;
}
exports.buildClassicBuildUrl = buildClassicBuildUrl;
/**
 * Check is the current Blue ocean page a pipeline page and if so,
 * decode it to the corresponding classic Jenkins Job page.
 * @returns {string|undefined} The classic job page, or undefined if
 * it was unable to decode the page URL.
 */
function toClassicJobPage(currentPageUrl, isMultibranch) {
    if (isMultibranch === void 0) { isMultibranch = false; }
    var pageUrlTokens = currentPageUrl.split('/').filter(function (token) { return typeof token === 'string' && token !== ''; });
    // Remove all path elements up to and including the Jenkins
    // organization name.
    var token = pageUrlTokens.shift();
    while (token !== undefined && token !== 'organizations') {
        token = pageUrlTokens.shift();
    }
    var classicJobFullName = classicOrganizationRoot(config_1.AppConfig.getOrganizationGroup());
    if (pageUrlTokens.length > 1) {
        // The next token is the actual organization name e.g. "jenkins".
        // Remove that since we don't need it.
        pageUrlTokens.shift();
        // The next token is the "full" job name, URL encoded.
        var fullJobName = decodeURIComponent(pageUrlTokens.shift());
        // If the URL comes from pipeline-editor then the "full" job name is the next token
        if (fullJobName === 'pipeline-editor' && pageUrlTokens.length === 2) {
            fullJobName = decodeURIComponent(pageUrlTokens.shift());
        }
        var fullJobNameTokens = fullJobName.split('/');
        if (fullJobName !== 'pipelines' && pageUrlTokens.length > 0) {
            classicJobFullName = classicJobFullName + '/job/' + fullJobNameTokens.join('/job/');
        }
        if (pageUrlTokens.length > 1) {
            // The next token being "detail" indicates that we're looking
            // at a branch.
            if (pageUrlTokens.shift() === 'detail') {
                // is going to be something like one of:
                // - detail/[freestyleA/activity]
                // - detail/[freestyleA/2/pipeline]
                if (isMultibranch) {
                    var branchName = pageUrlTokens.shift(); // "freestyleA"
                    var classicJobBranch = classicJobFullName + '/job/' + branchName;
                    // And if there's more than one token left then we have
                    // the detail/freestyleA/[2/pipeline] variant. The next
                    // token is the runId
                    if (pageUrlTokens.length > 1) {
                        return classicJobBranch + '/' + pageUrlTokens.shift(); // "2"
                    }
                    return classicJobBranch;
                }
                else if (pageUrlTokens.length > 2) {
                    // And if there's more than two tokens left then we have
                    // the detail/[freestyleA/2/pipeline] variant.
                    // Next token is the branch name - not really a branch name !!
                    // Ignoring it.
                    pageUrlTokens.shift(); // "freestyleA"
                    // And the next token is the runId.
                    var runId = pageUrlTokens.shift(); // "2"
                    return classicJobFullName + '/' + runId;
                }
            }
        }
    }
    return classicJobFullName;
}
exports.toClassicJobPage = toClassicJobPage;
/**
 * Build a REST resource URL from component identifiers.
 *
 * @param organizationName the name of the owning organisation, including org folders
 * @param pipelineFullName (optional) the full name of a pipeline, including folders
 * @param branchName (optional) branch name for multibranch projects
 * @param runId (optional) identifies an individual run
 * @returns a URL string
 */
function buildRestUrl(organizationName, pipelineFullName, branchName, runId) {
    var jenkinsUrl = config_1.AppConfig.getJenkinsRootURL();
    var url = jenkinsUrl + "/blue/rest/organizations/" + encodeURIComponent(organizationName);
    if (pipelineFullName) {
        // pipelineFullName might include a folder path, and final component is already encoded
        url += "/pipelines/" + pipelineFullName;
    }
    if (branchName) {
        // JENKINS-37712 branch needs to be double-encoded for some reason
        url += "/branches/" + UrlUtils_1.UrlUtils.doubleUriEncode(branchName);
    }
    if (runId) {
        url += "/runs/" + encodeURIComponent(runId);
    }
    return UrlUtils_1.UrlUtils.ensureTrailingSlash(url);
}
exports.buildRestUrl = buildRestUrl;
