/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.checks.status;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import hudson.model.Result;
import hudson.model.Run;
import io.jenkins.plugins.checks.api.ChecksOutput;
import io.jenkins.plugins.checks.api.TruncatedString;
import io.jenkins.plugins.checks.utils.FlowNodeUtils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Stack;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.commons.collections.iterators.ReverseListIterator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jenkinsci.plugins.displayurlapi.DisplayURLProvider;
import org.jenkinsci.plugins.workflow.actions.ArgumentsAction;
import org.jenkinsci.plugins.workflow.actions.ErrorAction;
import org.jenkinsci.plugins.workflow.actions.LabelAction;
import org.jenkinsci.plugins.workflow.actions.LogAction;
import org.jenkinsci.plugins.workflow.actions.ThreadNameAction;
import org.jenkinsci.plugins.workflow.actions.WarningAction;
import org.jenkinsci.plugins.workflow.flow.FlowExecution;
import org.jenkinsci.plugins.workflow.graph.BlockStartNode;
import org.jenkinsci.plugins.workflow.graph.FlowNode;
import org.jenkinsci.plugins.workflow.support.visualization.table.FlowGraphTable;

class FlowExecutionAnalyzer {
    private static final Logger LOGGER = Logger.getLogger(FlowExecutionAnalyzer.class.getName());
    private static final String TRUNCATED_MESSAGE = "\n\nOutput truncated.";
    private static final String TRUNCATED_MESSAGE_BUILD_LOG = "Build log truncated.\n\n";
    private static final int MAX_MESSAGE_SIZE_TO_CHECKS_API = 65535;
    private final Run<?, ?> run;
    private final FlowExecution execution;
    private final Stack<Integer> indentationStack = new Stack();
    private final boolean suppressLogs;

    FlowExecutionAnalyzer(Run<?, ?> run, FlowExecution execution, boolean suppressLogs) {
        this.run = run;
        this.execution = execution;
        this.suppressLogs = suppressLogs;
    }

    private static Optional<String> getStageOrBranchName(FlowNode node) {
        if (node instanceof BlockStartNode) {
            return FlowExecutionAnalyzer.getParallelName(node).or(() -> FlowExecutionAnalyzer.getStageName(node));
        }
        return Optional.empty();
    }

    private static Optional<String> getStageName(FlowNode node) {
        return Optional.ofNullable(node).filter(n -> n.getAction(ThreadNameAction.class) == null).map(n -> (LabelAction)n.getAction(LabelAction.class)).map(LabelAction::getDisplayName);
    }

    private static Optional<String> getParallelName(FlowNode node) {
        return Optional.ofNullable(node).filter(n -> n.getAction(LabelAction.class) != null).map(n -> (ThreadNameAction)n.getAction(ThreadNameAction.class)).map(ThreadNameAction::getThreadName);
    }

    private Pair<String, String> processStageOrBranchRow(FlowGraphTable.Row row, String stageOrBranchName) {
        StringBuilder nodeTextBuilder = new StringBuilder();
        while (!this.indentationStack.isEmpty() && row.getTreeDepth() < this.indentationStack.peek()) {
            this.indentationStack.pop();
        }
        if (this.indentationStack.isEmpty() || row.getTreeDepth() > this.indentationStack.peek()) {
            this.indentationStack.push(row.getTreeDepth());
        }
        nodeTextBuilder.append(String.join((CharSequence)"", Collections.nCopies(this.indentationStack.size(), "  ")));
        nodeTextBuilder.append("* ");
        nodeTextBuilder.append(stageOrBranchName);
        if (row.getNode().isActive()) {
            nodeTextBuilder.append(" *(running)*");
        } else {
            nodeTextBuilder.append(String.format(" *(%s)*", row.getDurationString()));
        }
        nodeTextBuilder.append("\n");
        return Pair.of((Object)nodeTextBuilder.toString(), (Object)"");
    }

    private Pair<String, String> processErrorOrWarningRow(FlowGraphTable.Row row, ErrorAction errorAction, WarningAction warningAction) {
        FlowNode flowNode = row.getNode();
        StringBuilder nodeSummaryBuilder = new StringBuilder();
        StringBuilder nodeTextBuilder = new StringBuilder();
        List location = flowNode.getEnclosingBlocks().stream().map(FlowExecutionAnalyzer::getStageOrBranchName).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        Collections.reverse(location);
        location.add(flowNode.getDisplayName());
        nodeSummaryBuilder.append(String.format("### `%s`%n", String.join((CharSequence)" / ", location)));
        nodeSummaryBuilder.append(String.format("%s in `%s` step", errorAction == null ? "Warning" : "Error", flowNode.getDisplayFunctionName()));
        String arguments = ArgumentsAction.getStepArgumentsAsString((FlowNode)flowNode);
        if (arguments == null) {
            nodeSummaryBuilder.append(".\n");
        } else {
            nodeSummaryBuilder.append(String.format(", with arguments `%s`.%n", arguments));
        }
        nodeTextBuilder.append(String.join((CharSequence)"", Collections.nCopies(this.indentationStack.size() + 1, "  ")));
        if (warningAction == null) {
            String displayName = errorAction == null ? "[no error action]" : errorAction.getDisplayName();
            nodeTextBuilder.append(String.format("**Error**: *%s*", displayName));
            nodeSummaryBuilder.append(String.format("```%n%s%n```%n", displayName));
            if (!this.suppressLogs) {
                String logTemplate = "<details>%n<summary>Build log</summary>%n%n```%n%s%n```%n</details>";
                int maxMessageSize = 65535 - nodeSummaryBuilder.length() - logTemplate.length() - 32;
                String log = FlowExecutionAnalyzer.getLog(flowNode, maxMessageSize);
                if (StringUtils.isNotBlank((CharSequence)log)) {
                    nodeSummaryBuilder.append(String.format(logTemplate, log));
                }
            }
        } else {
            nodeTextBuilder.append(String.format("**Unstable**: *%s*", warningAction.getMessage()));
            nodeSummaryBuilder.append(String.format("```%n%s%n```", warningAction.getMessage()));
        }
        try {
            String logsUrl = String.format("%s%slog", DisplayURLProvider.get().getRoot(), flowNode.getUrl());
            nodeTextBuilder.append(String.format(" - [logs](%s)", logsUrl));
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, String.format("Failed to get log url for step '%s'", flowNode.getDisplayName()).replaceAll("[\r\n]", ""), e);
        }
        nodeTextBuilder.append("\n");
        nodeSummaryBuilder.append("\n\n");
        return Pair.of((Object)nodeTextBuilder.toString(), (Object)nodeSummaryBuilder.toString());
    }

    ChecksOutput extractOutput() {
        FlowGraphTable table = new FlowGraphTable(this.execution);
        table.build();
        TruncatedString.Builder summaryBuilder = new TruncatedString.Builder().withTruncationText(TRUNCATED_MESSAGE);
        TruncatedString.Builder textBuilder = new TruncatedString.Builder().withTruncationText(TRUNCATED_MESSAGE);
        this.indentationStack.clear();
        String title = null;
        for (FlowGraphTable.Row row : table.getRows()) {
            FlowNode flowNode = row.getNode();
            Optional<String> stageOrBranchName = FlowExecutionAnalyzer.getStageOrBranchName(flowNode);
            ErrorAction errorAction = flowNode.getError();
            WarningAction warningAction = (WarningAction)flowNode.getPersistentAction(WarningAction.class);
            if (!stageOrBranchName.isPresent() && errorAction == null && warningAction == null) continue;
            Pair nodeInfo = stageOrBranchName.map(s -> this.processStageOrBranchRow(row, (String)s)).orElseGet(() -> this.processErrorOrWarningRow(row, errorAction, warningAction));
            if (stageOrBranchName.isEmpty()) {
                title = this.getPotentialTitle(flowNode, errorAction);
            }
            textBuilder.addText((String)nodeInfo.getLeft());
            summaryBuilder.addText((String)nodeInfo.getRight());
        }
        return new ChecksOutput.ChecksOutputBuilder().withTitle(this.extractOutputTitle(title)).withSummary(summaryBuilder.build()).withText(textBuilder.build()).build();
    }

    private String getPotentialTitle(FlowNode flowNode, ErrorAction errorAction) {
        String whereBuildFailed = String.format("%s in '%s' step", errorAction == null ? "warning" : "error", flowNode.getDisplayFunctionName());
        List<FlowNode> enclosingStagesAndParallels = FlowNodeUtils.getEnclosingStagesAndParallels(flowNode);
        List<String> enclosingBlockNames = FlowNodeUtils.getEnclosingBlockNames(enclosingStagesAndParallels);
        return StringUtils.join((Iterator)new ReverseListIterator(enclosingBlockNames), (String)"/") + ": " + whereBuildFailed;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @CheckForNull
    private static String getLog(FlowNode flowNode, int maxMessageSize) {
        LogAction logAction = (LogAction)flowNode.getAction(LogAction.class);
        if (logAction == null) {
            return null;
        }
        try (ByteArrayOutputStream out = new ByteArrayOutputStream();){
            if (logAction.getLogText().writeLogTo(0L, (OutputStream)out) == 0L) {
                String string = null;
                return string;
            }
            String outputString = out.toString(StandardCharsets.UTF_8);
            String log = outputString.replaceAll("\u001b\\[[;\\d]*m", "");
            String string = new TruncatedString.Builder().setChunkOnNewlines().setTruncateStart().withTruncationText(TRUNCATED_MESSAGE_BUILD_LOG).addText(log).build().build(maxMessageSize);
            return string;
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, String.format("Failed to extract logs for step '%s'", flowNode.getDisplayName()).replaceAll("[\r\n]", ""), e);
            return null;
        }
    }

    private String extractOutputTitle(String title) {
        Result result = this.run.getResult();
        if (result == null) {
            return "In progress";
        }
        if (result.isBetterOrEqualTo(Result.SUCCESS)) {
            return "Success";
        }
        if (title != null) {
            return title;
        }
        if (result.isBetterOrEqualTo(Result.UNSTABLE)) {
            return "Unstable";
        }
        if (result.isBetterOrEqualTo(Result.FAILURE)) {
            return "Failure";
        }
        if (result.isBetterOrEqualTo(Result.NOT_BUILT)) {
            return "Skipped";
        }
        if (result.isBetterOrEqualTo(Result.ABORTED)) {
            return "Aborted";
        }
        throw new IllegalStateException("Unsupported run result: " + String.valueOf(result));
    }
}

