/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.service;

import com.stripe.StripeClient;
import com.stripe.model.Price;
import com.stripe.model.Subscription;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubSettingService;
import io.meeds.billing.service.SubscriptionEmailReminderService;
import org.exoplatform.commons.ObjectAlreadyExistsException;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.portal.config.model.Application;
import org.exoplatform.portal.config.model.ModelObject;
import org.exoplatform.portal.config.model.PortalConfig;
import org.exoplatform.portal.config.model.TransientApplicationState;
import org.exoplatform.portal.config.serialize.model.SiteLayout;
import org.exoplatform.portal.mop.SiteKey;
import org.exoplatform.portal.mop.service.LayoutService;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.manager.IdentityManager;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.bean.override.mockito.MockitoBean;

import java.util.ArrayList;
import java.util.List;

import static junit.framework.TestCase.assertEquals;
import static junit.framework.TestCase.assertTrue;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertThrows;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@SpringBootTest(classes = { BillingService.class, })
@ExtendWith(MockitoExtension.class)
public class BillingServiceTest {

  @MockitoBean
  private SpaceService                     spaceService;

  @MockitoBean
  private HubSettingService                hubSettingService;

  @MockitoBean
  private SubscriptionEmailReminderService subscriptionEmailReminderService;

  @MockitoBean
  private IdentityManager                  identityManager;

  @MockitoBean
  private LayoutService                    layoutService;

  @MockitoBean
  private HubBilling                       hubBilling;

  @MockitoBean
  private StripeClient                     stripeClient;

  private BillingService                   billingService;

  @BeforeEach
  public void setup() {
    when(hubBilling.isEnabled()).thenReturn(true);
    when(hubBilling.getSecretKey()).thenReturn("secret");
    billingService = spy(new BillingService(hubBilling, subscriptionEmailReminderService, spaceService, hubSettingService, identityManager, layoutService));
  }

  @Test
  public void billingServiceEnablementTest() {
    assertTrue(billingService.isEnabled());
  }

  @Test
  public void enableExistingSpaceBillingTest() throws Exception {
    long spaceId = 1L;
    String priceId = "priceId";
    String spaceManager = "john";
    Space space = mock(Space.class);
    when(space.getId()).thenReturn(String.valueOf(spaceId));
    when(space.getGroupId()).thenReturn("groupId");
    when(spaceService.getSpaceById(spaceId)).thenReturn(null).thenReturn(space);
    when(spaceService.canManageSpace(space, spaceManager)).thenReturn(true);
    assertThrows(ObjectNotFoundException.class, () -> billingService.enableExistingSpaceBilling(spaceId, priceId, spaceManager));

    HubBillingSettings hubBillingSettings = mock(HubBillingSettings.class);
    when(hubSettingService.getSettingsBySpaceId(space.getId())).thenReturn(hubBillingSettings).thenReturn(null);
    assertThrows(ObjectAlreadyExistsException.class,
                 () -> billingService.enableExistingSpaceBilling(spaceId, priceId, spaceManager));

    assertThrows(IllegalArgumentException.class, () -> billingService.enableExistingSpaceBilling(spaceId, null, spaceManager));

    Identity identity = mock(Identity.class);
    when(identityManager.getOrCreateUserIdentity(spaceManager)).thenReturn(identity);
    Price price = mock(Price.class);
    Subscription subscription = mock(Subscription.class);
    when(price.getId()).thenReturn(priceId);
    doReturn(price).when(billingService).getPriceById(priceId);
    doReturn("customer_id").when(billingService).createCustomer(space, identity);
    doReturn(subscription).when(billingService).subscribe(space, "customer_id", priceId);
    doNothing().when(hubSettingService).createSettings(space, subscription, identity);
    doNothing().when(billingService).updateBillableSpaceLayout(any(SiteKey.class));

    billingService.enableExistingSpaceBilling(spaceId, priceId, spaceManager);
    verify(billingService, times(1)).getPriceById(priceId);
    verify(billingService, times(1)).createCustomer(space, identity);
    verify(billingService, times(1)).subscribe(space, "customer_id", priceId);
    verify(hubSettingService, times(1)).createSettings(space, subscription, identity);
    verify(billingService, times(1)).updateBillableSpaceLayout(any(SiteKey.class));
  }

  @Test
  void updateBillableSpaceLayoutTest() {
    SiteKey siteKey = mock(SiteKey.class);
    SiteLayout existingLayout = new SiteLayout();
    Application existingApp = new Application();
    existingLayout.setChildren(new ArrayList<>(List.of(existingApp)));
    PortalConfig portalConfig = new PortalConfig();
    portalConfig.setAccessPermissions(new String[] { "*:/spaces" });
    portalConfig.setPortalLayout(existingLayout);

    when(layoutService.getPortalConfig(siteKey)).thenReturn(portalConfig);

    billingService.updateBillableSpaceLayout(siteKey);

    List<ModelObject> children = portalConfig.getPortalLayout().getChildren();
    assertEquals(2, children.size());

    Application addedApp = (Application) children.get(1);
    assertEquals("billing/HubSubscriptionRestriction", ((TransientApplicationState) addedApp.getState()).getContentId());
    assertArrayEquals(portalConfig.getAccessPermissions(), addedApp.getAccessPermissions());
    verify(layoutService).save(portalConfig);
  }

}
