/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.scheduling.task;

import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubSettingService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * Scheduled job responsible for reporting the number of members in each hub
 * (space) that has metered billing enabled. This task runs automatically at a
 * configured interval (daily by default) to synchronize the member usage data
 * with Stripe by sending a metered billing event for each hub. The job runs
 * only if: the property {@code meeds.billing.enabled=true} and the property
 * {@code meeds.billing.secretKey} is defined and not blank The schedule is
 * defined by the cron expression configured under
 * {@code meeds.billing.hubBillingReport.job.cron}, which defaults to midnight
 * every day ({@code 0 0 0 * * *}).
 */
@Component
public class HubBillingReportJob {

  private static final Log  LOG = ExoLogger.getLogger(HubBillingReportJob.class);

  @Autowired
  private BillingService    billingService;

  @Autowired
  private HubSettingService hubSettingService;

  @Scheduled(cron = "${meeds.billing.hubBillingReport.job.cron:0 0 0 * * *}")
  public void reportMembersCount() {
    if (!billingService.isEnabled()) {
      return;
    }
    List<HubBillingSettings> hubBillingSettingsList = hubSettingService.getMeteredHubBillingSettingList();
    if (hubBillingSettingsList.isEmpty()) {
      return;
    }
    hubBillingSettingsList.forEach(hubBillingSettings -> {
      try {
        billingService.reportMembersCount(hubBillingSettings);
      } catch (Exception exception) {
        LOG.error("Error when reporting members count for space with id {}, {}",
                  hubBillingSettings.getSpaceId(),
                  exception.getMessage());
      }
    });
  }

}
