/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.rest;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.lenient;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.user;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.rest.BillingRest;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubSettingService;
import lombok.SneakyThrows;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.services.security.ConversationState;
import org.exoplatform.services.security.Identity;
import org.exoplatform.social.core.manager.IdentityManager;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.MockedStatic;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureWebMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.bean.override.mockito.MockitoBean;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.RequestPostProcessor;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.context.WebApplicationContext;

import com.fasterxml.jackson.core.json.JsonReadFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

import io.meeds.spring.web.security.PortalAuthenticationManager;
import io.meeds.spring.web.security.WebSecurityConfiguration;
import jakarta.servlet.Filter;

@SpringBootTest(classes = { BillingRest.class, PortalAuthenticationManager.class, })
@ContextConfiguration(classes = { WebSecurityConfiguration.class })
@AutoConfigureWebMvc
@AutoConfigureMockMvc(addFilters = false)
@ExtendWith(MockitoExtension.class)
public class BillingRestTest {

  private static final String BILLING_PATH  = "/billing";

  private static final String SIMPLE_USER   = "simple";

  private static final String ADMIN_USER    = "admin";

  private static final String TEST_PASSWORD = "testPassword";

  static final ObjectMapper   OBJECT_MAPPER;

  static {
    // Workaround when Jackson is defined in shared library with different
    // version and without artifact jackson-datatype-jsr310
    OBJECT_MAPPER = JsonMapper.builder()
                              .configure(JsonReadFeature.ALLOW_MISSING_VALUES, true)
                              .configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false)
                              .build();
    OBJECT_MAPPER.registerModule(new JavaTimeModule());
  }

  @Autowired
  private SecurityFilterChain   filterChain;

  @Autowired
  private WebApplicationContext context;

  @MockitoBean
  private IdentityManager       identityManager;

  @MockitoBean
  HubSettingService             hubSettingService;

  @MockitoBean
  BillingService                billingService;

  private MockMvc               mockMvc;

  @BeforeEach
  public void setUp() {
    mockMvc = MockMvcBuilders.webAppContextSetup(context).addFilters(filterChain.getFilters().toArray(new Filter[0])).build();
  }

  @Test
  public void shouldReturnNotFoundIfNoHubSettings() throws Exception {
    when(billingService.createCustomerPortalSession(anyLong(), anyString())).thenThrow(ObjectNotFoundException.class);
    try (MockedStatic<ConversationState> conversationStateMock = mockCurrentIdentityId()) {
      mockMvc.perform(get(BILLING_PATH + "/customer-portal/123").with(testSimpleUser())).andExpect(status().isNotFound());
    }
  }

  @Test
  public void shouldReturnForbiddenIfUserIsNotManager() throws Exception {
    when(billingService.createCustomerPortalSession(anyLong(), anyString())).thenThrow(IllegalAccessException.class);
    try (MockedStatic<ConversationState> conversationStateMock = mockCurrentIdentityId()) {
      mockMvc.perform(get(BILLING_PATH + "/customer-portal/123").with(testSimpleUser())).andExpect(status().isForbidden());
    }

  }

  @Test
  public void shouldReturnCustomerPortalUrl() throws Exception {
    when(billingService.createCustomerPortalSession(anyLong(), anyString())).thenReturn("customerPortalUrl");
    try (MockedStatic<ConversationState> conversationStateMock = mockCurrentIdentityId()) {
      mockMvc.perform(get(BILLING_PATH + "/customer-portal/123").with(testSimpleUser())).andExpect(status().isOk());
    }

  }

  @Test
  public void testCancelSubscription() throws Exception {
    doThrow(new IllegalAccessException())
            .when(billingService)
            .cancelSubscription(anyLong(), anyString());
    try (MockedStatic<ConversationState> conversationStateMock = mockCurrentIdentityId()) {
      mockMvc.perform(delete(BILLING_PATH + "/subscription/123/cancel").with(testSimpleUser())).andExpect(status().isForbidden());
      //
      doNothing().when(billingService).cancelSubscription(anyLong(), anyString());
      mockMvc.perform(delete(BILLING_PATH + "/subscription/123/cancel").with(testSimpleUser())).andExpect(status().isNoContent());
    }

  }

  @Test
  public void testNotifyOnSubscriptionStatus() throws Exception {
    doThrow(IllegalAccessException.class)
            .doNothing()
            .when(billingService)
            .notifyOnSubscriptionStatus(anyLong(), anyString(), any(Boolean.class));
    try (MockedStatic<ConversationState> conversationStateMock = mockCurrentIdentityId()) {
      mockMvc.perform(post(BILLING_PATH + "/notify/subscription-status/123").with(testSimpleUser()))
             .andExpect(status().isForbidden());
      //
      mockMvc.perform(post(BILLING_PATH + "/notify/subscription-status/123").with(testSimpleUser()))
             .andExpect(status().isNoContent());
    }
  }

  @Test
  public void testNotifyHubAdminsOnUsersLimitExceeded() throws Exception {
    doNothing().when(billingService).notifyOnSubscriptionStatus(anyLong(), anyString(), any(Boolean.class));

    try (MockedStatic<ConversationState> conversationStateMock = mockCurrentIdentityId()) {
      mockMvc.perform(post(BILLING_PATH + "/notify/subscription-status/123?maxOfUsersExceeded=true").with(testSimpleUser()))
              .andExpect(status().isNoContent());
    }
  }

  private RequestPostProcessor testSimpleUser() {
    return user(SIMPLE_USER).password(TEST_PASSWORD).authorities(new SimpleGrantedAuthority("users"));
  }

  @SneakyThrows
  public static MockedStatic<ConversationState> mockConversationState() {
    Identity identity = mock(Identity.class);
    ConversationState conversationState = mock(ConversationState.class);
    MockedStatic<ConversationState> conversationStateStatic = mockStatic(ConversationState.class);
    conversationStateStatic.when(ConversationState::getCurrent).thenReturn(conversationState);
    when(conversationState.getIdentity()).thenReturn(identity);
    return conversationStateStatic;
  }

  @SneakyThrows
  public MockedStatic<ConversationState> mockCurrentIdentityId() {
    MockedStatic<ConversationState> conversationStateStatic = mockConversationState();
    when(ConversationState.getCurrent().getIdentity().getUserId()).thenReturn(SIMPLE_USER);
    org.exoplatform.social.core.identity.model.Identity identity =
                                                                 mock(org.exoplatform.social.core.identity.model.Identity.class);
    lenient().when(identity.getId()).thenReturn("1");
    when(identityManager.getOrCreateUserIdentity(SIMPLE_USER)).thenReturn(identity);
    return conversationStateStatic;
  }
}
