/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.service;

import com.stripe.model.Invoice;
import com.stripe.model.Plan;
import com.stripe.model.StripeObject;
import com.stripe.model.Subscription;
import com.stripe.net.Webhook;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.model.HubBillingPlan;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.utils.Utils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.stripe.model.Event;

@Service
public class WebhookHandlingService {

  private static final Log    LOG                                 = ExoLogger.getLogger(WebhookHandlingService.class);

  private static final String CUSTOMER_SUBSCRIPTION_UPDATED_EVENT = "customer.subscription.updated";

  private static final String CUSTOMER_INVOICE_PAYMENT_EVENT      = "invoice.paid";

  private static final String PLAN_UPDATED_EVENT                  = "plan.updated";

  @Autowired
  private HubBilling            hubBilling;

  @Autowired
  private HubSettingService     hubSettingService;

  @Autowired
  private BillingService        billingService;

  @Autowired
  private HubBillingPlanService hubBillingPlanService;

  public void handleWebHookEvent(String payload, String signature) throws Exception {
    Event event = null;
    String webHookSecret = hubBilling.getWebHookSecretKey();
    event = Webhook.constructEvent(payload, signature, webHookSecret);
    processEvent(event);
  }

  private void processEvent(Event event) throws Exception {
    switch (event.getType()) {
    case CUSTOMER_SUBSCRIPTION_UPDATED_EVENT -> processSubscriptionUpdatedEvent(event);
    case CUSTOMER_INVOICE_PAYMENT_EVENT -> processInvoicePaymentEvent(event);
    case PLAN_UPDATED_EVENT -> processPlanUpdatedEvent(event);
    default -> LOG.warn("Unknown event type: " + event.getType());
    }
  }

  private void processSubscriptionUpdatedEvent(Event event) throws ObjectNotFoundException {
    StripeObject stripeObject = event.getDataObjectDeserializer().getObject().orElse(null);
    if (stripeObject instanceof Subscription subscription) {
      HubBillingSettings hubBillingSettings = hubSettingService.getSettingsBySubscriptionId(subscription.getId());
      String currentPlanId = hubBillingSettings.getHubBillingPlan().getPlanId();
      String updatedPlanId = Utils.getPlanFromSubscription(subscription).getId();
      boolean planChanged = !StringUtils.equals(currentPlanId, updatedPlanId);
      hubSettingService.updateHuBillingSetting(hubBillingSettings, subscription, event.getData().getPreviousAttributes());
      if (planChanged) {
        notifyAdminsOnPlanChange(hubBillingSettings.getSpaceId());
      }
    }
  }

  private void processInvoicePaymentEvent(Event event) throws Exception {
    StripeObject stripeObject = event.getDataObjectDeserializer().getObject().orElse(null);
    if (stripeObject instanceof Invoice invoice) {
      boolean isTrialInvoice = invoice.getAmountPaid() == 0 ;
      String subscriptionId = invoice.getParent().getSubscriptionDetails().getSubscription();
      HubBillingSettings hubBillingSettings = hubSettingService.getSettingsBySubscriptionId(subscriptionId);
      if (hubBillingSettings != null && Utils.SubscriptionStatus.TRIALING.name().equalsIgnoreCase(hubBillingSettings.getSubscriptionStatus()) && !isTrialInvoice) {
        Subscription subscription = billingService.getSubscription(subscriptionId);
        hubBillingSettings.setSubscriptionStatus(Utils.SubscriptionStatus.ACTIVE.name());
        hubBillingSettings.setSubscriptionCurrentPeriodEnd(Utils.getCurrentPeriodEndDate(subscription));
        hubSettingService.updateSettings(hubBillingSettings);
      }
    }
  }
  
  private void processPlanUpdatedEvent(Event event) {
    StripeObject stripeObject = event.getDataObjectDeserializer().getObject().orElse(null);
    if (stripeObject instanceof Plan plan) {
      HubBillingPlan hubBillingPlan = hubBillingPlanService.getHubBillingPlanByPlanId(plan.getId());
      if (hubBillingPlan != null) {
        hubBillingPlan.setMeterId(plan.getMeter());
        hubBillingPlan.setProductId(plan.getProduct());
        String maxOfUsersMetadata = MapUtils.isNotEmpty(plan.getMetadata()) ? plan.getMetadata().get("maxOfUsers") : null;
        long maxOfUsers = maxOfUsersMetadata == null ? 0 : Long.parseLong(maxOfUsersMetadata);
        hubBillingPlan.setMaxUsers(maxOfUsers);
        hubBillingPlan.setBillingScheme(plan.getBillingScheme());
        hubBillingPlan.setTiersMode(plan.getTiersMode());
        hubBillingPlanService.saveHubBillingPlan(hubBillingPlan);
      }
    }
  }

  private void notifyAdminsOnPlanChange(long spaceId) {
    billingService.notifyAdminsOnPlanChange(spaceId);
  }
}
