/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.listener;

import com.stripe.exception.StripeException;
import com.stripe.model.Price;
import com.stripe.model.Subscription;
import io.meeds.billing.listener.BillingSpaceListener;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubSettingService;
import io.meeds.billing.utils.Utils;
import org.exoplatform.commons.ObjectAlreadyExistsException;
import org.exoplatform.services.listener.ListenerService;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.manager.IdentityManager;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceLifeCycleEvent;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.MockedStatic;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.bean.override.mockito.MockitoBean;

import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.mockito.ArgumentMatchers.any;

import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;


@SpringBootTest(classes = { BillingSpaceListener.class })
@ExtendWith(MockitoExtension.class)
public class BillingSpaceListenerTest {

  @MockitoBean
  private HubBilling        hubBilling;

  @MockitoBean
  private HubSettingService hubSettingService;

  @MockitoBean
  private IdentityManager   identityManager;

  @MockitoBean
  private SpaceService      spaceService;

  @MockitoBean
  private ListenerService   listenerService;

  @MockitoBean
  private BillingService    billingService;

  @Autowired
  private BillingSpaceListener billingSpaceListener;

  @BeforeEach
  void setUp() {
    billingSpaceListener.init();
  }

  @Test
  public void testSaveSettings() throws StripeException, ObjectAlreadyExistsException {
    Space space = mock(Space.class);
    Identity identity = mock(Identity.class);
    String customerId = "customer_1";
    String priceId = "price_1";
    Subscription subscription = mock(Subscription.class);
    Price price = mock(Price.class);

    when(price.getId()).thenReturn(priceId);

    when(space.getTemplateId()).thenReturn(1L);
    when(billingService.isEnabled()).thenReturn(true);
    when(hubBilling.getSpaceTemplateIds()).thenReturn(Collections.singletonList(1L));
    when(identityManager.getOrCreateUserIdentity("john")).thenReturn(identity);
    when(billingService.getPriceById(priceId)).thenReturn(price);
    when(billingService.createCustomer(space, identity)).thenReturn(customerId);
    when(billingService.subscribe(space, customerId, priceId)).thenReturn(subscription);

    try (MockedStatic<Utils> utilsMock = mockStatic(Utils.class)) {
      utilsMock.when(() -> Utils.getCurrentPeriodEndDate(subscription)).thenReturn(new Date());
      utilsMock.when(() -> Utils.getTemplatePriceId(anyLong())).thenReturn(priceId);
      SpaceLifeCycleEvent event = new SpaceLifeCycleEvent(space, "john", SpaceLifeCycleEvent.Type.SPACE_CREATED);
      // When
      billingSpaceListener.spaceCreated(event);
      // Then
      verify(hubSettingService, times(1)).createSettings(eq(space), any(Subscription.class), eq(identity));
      verify(billingService, times(1)).createCustomer(space, identity);
      verify(billingService, times(1)).subscribe(space, customerId, priceId);
    }
  }

  @Test
  public void testSpaceCreatedNotBillable() throws ObjectAlreadyExistsException {
    // Given
    Space space = new Space();
    space.setGroupId("/spaces/test");
    space.setTemplateId(2L);

    SpaceLifeCycleEvent event = new SpaceLifeCycleEvent(space, "john", SpaceLifeCycleEvent.Type.SPACE_CREATED);

    when(hubBilling.isEnabled()).thenReturn(true);
    List<Long> templateIds = Collections.singletonList(1L);
    when(hubBilling.getSpaceTemplateIds()).thenReturn(templateIds);

    // When
    billingSpaceListener.spaceCreated(event);

    // Then - nothing should be saved
    verify(hubSettingService, never()).createSettings(any(Space.class), any(Subscription.class), any(Identity.class));
  }

}
