/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billig.scheduling.task;

import io.meeds.billing.scheduling.task.HubBillingReportJob;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubSettingService;
import io.meeds.billing.utils.Utils;
import org.junit.jupiter.api.BeforeEach;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.boot.test.context.SpringBootTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.bean.override.mockito.MockitoBean;

import java.util.Collections;
import java.util.List;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@SpringBootTest(classes = HubBillingReportJob.class)
@ExtendWith(MockitoExtension.class)
public class HubBillingReportJobTest {

  @MockitoBean
  private BillingService      billingService;

  @MockitoBean
  private HubSettingService   hubSettingService;

  @Autowired
  private HubBillingReportJob hubBillingReportJob;

  @BeforeEach
  public void setup() {
    when(billingService.isEnabled()).thenReturn(true);
  }

  @Test
  public void shouldDoNothingWhenNoHubBillingSettings() throws Exception {
    when(hubSettingService.getMeteredHubBillingSettingList()).thenReturn(Collections.emptyList());
    // When
    hubBillingReportJob.reportMembersCount();
    // Then
    verify(billingService, never()).reportMembersCount(any());
  }

  @Test
  public void shouldReportMembersCountForEachHubBillingSetting() throws Exception {
    HubBillingSettings hubBillingSettings = mock(HubBillingSettings.class);
    when(hubSettingService.getMeteredHubBillingSettingList()).thenReturn(List.of(hubBillingSettings));
    // When
    hubBillingReportJob.reportMembersCount();
    // Then
    verify(billingService, times(1)).reportMembersCount(hubBillingSettings);
  }

  @Test
  public void shouldCatchExceptionWhenBillingServiceFails() throws Exception {
    HubBillingSettings hubBillingSettings = mock(HubBillingSettings.class);
    HubBillingSettings secondHubBillingSettings = mock(HubBillingSettings.class);
    when(hubBillingSettings.getSpaceId()).thenReturn(123L);
    when(hubBillingSettings.getSpaceId()).thenReturn(1234L);
    when(hubSettingService.getMeteredHubBillingSettingList()).thenReturn(List.of(hubBillingSettings, secondHubBillingSettings));
    doThrow(new Exception("Simulated failure")).when(billingService).reportMembersCount(hubBillingSettings);
    doNothing().when(billingService).reportMembersCount(secondHubBillingSettings);
    hubBillingReportJob.reportMembersCount();
    // Should continue gracefully without propagating the exception
    verify(billingService, times(1)).reportMembersCount(hubBillingSettings);
    verify(billingService, times(1)).reportMembersCount(secondHubBillingSettings);
  }

  @Test
  public void shouldDoNothingWhenNoTiredHubBillingSettings() throws Exception {
    when(hubSettingService.getTieredGraduatedBillingSettingList()).thenReturn(Collections.emptyList());
    // When
    hubBillingReportJob.computeTieredHubBillingQuantities();
    // Then
    verify(billingService, never()).updateTieredBillingQuantities(any());
  }

  @Test
  public void shouldDoNothingWhenBillingServiceDisabled() throws Exception {
    when(billingService.isEnabled()).thenReturn(false);
    // When
    hubBillingReportJob.computeTieredHubBillingQuantities();
    // Then
    verify(billingService, never()).updateTieredBillingQuantities(any());
  }

  @Test
  public void tieredHubBillingReportTest() throws Exception {
    HubBillingSettings hubBillingSettings = mock(HubBillingSettings.class);
    when(hubBillingSettings.getSubscriptionStatus()).thenReturn(Utils.SubscriptionStatus.ACTIVE.name());
    when(hubSettingService.getTieredGraduatedBillingSettingList()).thenReturn(List.of(hubBillingSettings));
    doNothing().when(billingService).updateTieredBillingQuantities(hubBillingSettings);
    // When
    hubBillingReportJob.computeTieredHubBillingQuantities();
    // Then
    verify(billingService, times(1)).updateTieredBillingQuantities(hubBillingSettings);
  }
}
