/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.scheduling.task;

import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.service.HubSettingService;
import io.meeds.billing.utils.Utils;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * Scheduled job responsible for synchronizing hub (space) usage data with
 * Stripe. This component contains two scheduled tasks:
 * Metered billing report:
 * Reports the current number of members in each hub that has metered billing enabled.
 * For each hub, a metered billing event is sent to Stripe to track usage.
 * The job runs at a configurable interval (daily by default) to midnight every day ({@code 0 0 0 * * *}).
 * Tiered billing computation:
 * Computes and updates the member quantities for hubs with a tiered graduated billing plan.
 * This ensures the correct tiered pricing is reported to Stripe
 * The job runs at a configurable interval (daily by default) to midnight every day ({@code 0 0 0 * * *}).
 */
@Component
public class HubBillingReportJob {

  private static final Log  LOG = ExoLogger.getLogger(HubBillingReportJob.class);

  @Autowired
  private BillingService    billingService;

  @Autowired
  private HubSettingService hubSettingService;

  @Scheduled(cron = "${meeds.billing.hubBillingReport.job.cron:0 0 0 * * *}")
  public void reportMembersCount() {
    if (!billingService.isEnabled()) {
      return;
    }
    List<HubBillingSettings> hubBillingSettingsList = hubSettingService.getMeteredHubBillingSettingList();
    if (hubBillingSettingsList.isEmpty()) {
      return;
    }
    hubBillingSettingsList.forEach(hubBillingSettings -> {
      try {
        billingService.reportMembersCount(hubBillingSettings);
      } catch (Exception exception) {
        LOG.error("Error when reporting members count for space with id {}, {}",
                  hubBillingSettings.getSpaceId(),
                  exception.getMessage());
      }
    });
  }

  @Scheduled(cron = "${meeds.billing.tieredHubBillingReport.job.cron:0 0 0 * * *}")
  public void computeTieredHubBillingQuantities() {
    if (!billingService.isEnabled()) {
      return;
    }
    List<HubBillingSettings> hubBillingSettingsList = hubSettingService.getTieredGraduatedBillingSettingList()
                                                                       .stream()
                                                                       .filter(this::isEligibleForQuantityUpdate)
                                                                       .toList();
    if (hubBillingSettingsList.isEmpty()) {
      return;
    }
    hubBillingSettingsList.forEach(hubBillingSettings -> {
      try {
        billingService.updateTieredBillingQuantities(hubBillingSettings);
      } catch (Exception exception) {
        LOG.error("Error when updating subscription quantity for space with id {}, {}",
                  hubBillingSettings.getSpaceId(),
                  exception.getMessage());
      }
    });
  }

  private boolean isEligibleForQuantityUpdate(HubBillingSettings settings) {
    return !StringUtils.equalsAnyIgnoreCase(settings.getSubscriptionStatus(),
                                            Utils.SubscriptionStatus.INCOMPLETE.name(),
                                            Utils.SubscriptionStatus.CANCELED.name());
  }

}
