/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.services.security;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import javax.security.auth.Subject;

/**
 * User Session encapsulates user's principals such as name, groups along with
 * JAAS subject (useful in J2EE environment) as well as other optional
 * attributes
 */
public class Identity
{

   /**
    * User's identifier.
    */
   private String userId;

   /**
    * Memberships.
    */
   private final Set<MembershipEntry> memberships;

   /**
    * javax.security.auth.Subject can be used for logout process. <code>
    * LoginContext ctx = new LoginContext("exo-domain", subject);
    * ctx.logout();
    * </code>
    */
   private Subject subject;

   /**
    * User's roles.
    */
   private final Set<String> roles;

   /**
    * @param userId the user's identifier.
    */
   public Identity(String userId)
   {
      this(userId, new HashSet<>(), new HashSet<>());
   }

   /**
    * @param userId the user's identifier.
    * @param memberships the memberships.
    */
   public Identity(String userId, Collection<MembershipEntry> memberships)
   {
      this(userId, memberships, new HashSet<>());
   }

   /**
    * @param userId the user's identifier.
    * @param memberships the memberships.
    * @param roles the user's roles.
    */
   public Identity(String userId, Collection<MembershipEntry> memberships, Collection<String> roles)
   {
      this.userId = userId;
      this.memberships = new MembershipHashSet(memberships);
      this.roles = new HashSet<>(roles);
   }

   /**
    * @return user identifier.
    */
   public String getUserId()
   {
      return userId;
   }

   /**
    * @param group the group.
    * @param membershipType the MembershipType.
    * @return true if user has given membershipType for given group, false
    *         otherwise
    */
   public boolean isMemberOf(String group, String membershipType)
   {
      return containsMembership(new MembershipEntry(group, membershipType));
   }

   /**
    * @param me the MembershipEntry.
    * @return true if user has given MembershipEntry, false otherwise.
    */
   public boolean isMemberOf(MembershipEntry me)
   {
      return containsMembership(me);
   }

   /**
    * Check is user member of group.
    *
    * @param group the group.
    * @return true if user has any membershipType for given group, false
    *         otherwise.
    */
   public boolean isMemberOf(String group)
   {
      return containsMembership(new MembershipEntry(group));
   }

   /**
    * @return set of groups to which this user belongs to.
    */
   public Set<String> getGroups()
   {
      Set<String> groups = new HashSet<>();
      for (MembershipEntry m : memberships)
      {
         groups.add(m.getGroup());
      }
      return groups;
   }

   /**
    * @deprecated for back compatibility.
    */
   @Deprecated
   public void setMemberships(Collection<MembershipEntry> memberships)
   {
      this.memberships.clear();
      this.memberships.addAll(memberships);
   }

   /**
    * @return user's memberships.
    */
   public Collection<MembershipEntry> getMemberships()
   {
      return memberships;
   }

   /**
    * Sets the roles for J2EE environment using.
    *
    * @param roles the roles.
    */
   public void setRoles(Collection<String> roles)
   {
      this.roles.clear();
      this.roles.addAll(roles);
   }

   /**
    * @return roles users roles.
    */
   public Collection<String> getRoles()
   {
      return roles;
   }

   /**
    * @return {@link Subject}
    * @deprecated Do not need store subject any more. It was used before to
    *             perform logout, since tomcat 6.0.21 logout implemented in
    *             web-container.
    */
   public Subject getSubject()
   {
      return subject;
   }

   /**
    * @param subject {@link Subject} .
    * @deprecated See {@link #getSubject()}
    */
   public void setSubject(Subject subject)
   {
      this.subject = subject;
   }

   /**
    * Check is given {@link MembershipEntry} presents in user's memberships.
    *
    * @param checkMe the MembershipEntry.
    * @return true if presents false otherwise.
    */
   private boolean containsMembership(MembershipEntry checkMe)
   {
      return memberships.contains(checkMe);
   }

   @Override
   public int hashCode() {
     return Objects.hash(memberships, roles, subject, userId);
   }

   @Override
   public boolean equals(Object obj) {
     if (this == obj) {
       return true;
     } else if (obj == null) {
       return false;
     } else if (getClass() != obj.getClass()) {
       return false;
     } else {
       Identity other = (Identity) obj;
       return Objects.equals(memberships, other.memberships) && Objects.equals(roles, other.roles)
              && Objects.equals(subject, other.subject)
              && Objects.equals(userId, other.userId);
     }
   }

   @Override
   public String toString() {
     return "Identity [userId=" + userId + ", memberships=" + memberships + ", roles=" + roles + "]";
   }

}
