/*
 * This file is part of the Meeds project (https://meeds.io/).
 * 
 * Copyright (C) 2020 - 2022 Meeds Association contact@meeds.io
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
import {REST_ROOT_PATH} from './Constants';

export function getEmailTemplates() {
  return fetch(`${REST_ROOT_PATH}/email/templates`, {
    method: 'GET',
    credentials: 'include',
  }).then(resp => {
    if (resp && resp.ok) {
      return resp.json();
    } else {
      throw new Error('Error retrieving Email templates from server', resp);
    }
  }).then(templates => {
    const promises = [];
    templates.forEach(template => {
      if (template.notePageId) {
        promises.push(
          getEmailTemplateNotePage(template)
        );
      }
    });
    return Promise.all(promises)
      .then(() => templates);
  });
}

export function getEmailTemplateNotePage(template) {
  return Vue.prototype.$notesService.getNoteById(template.notePageId)
    .then(notePage => template.notePage = notePage);
}

export function createEmailTemplate(emailTemplate) {
  return fetch(`${REST_ROOT_PATH}/email/templates`, {
    method: 'POST',
    credentials: 'include',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(emailTemplate),
  }).then(resp => {
    if (resp && resp.ok) {
      return resp.json();
    } else {
      throw new Error('Error creating Email template on server', resp);
    }
  });
}

export function updateEmailTemplate(emailTemplate) {
  return fetch(`${REST_ROOT_PATH}/email/templates/${emailTemplate.id}`, {
    method: 'PUT',
    credentials: 'include',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(emailTemplate),
  }).then(resp => {
    if (resp && resp.ok) {
      return resp.json();
    } else {
      throw new Error('Error updating Email template on server', resp);
    }
  });
}

export function deleteEmailTemplate(id) {
  return fetch(`${REST_ROOT_PATH}/email/templates/${id}`, {
    method: 'DELETE',
    credentials: 'include',
  }).then(resp => {
    if (!resp || !resp.ok) {
      throw new Error('Error deleting Email template from server', resp);
    }
  });
}

export function sendEmail(nftId, subject, content) {
  const formData = new FormData();
  formData.append('nftId', nftId);
  formData.append('subject', subject);
  formData.append('content', content);
  return fetch(`${REST_ROOT_PATH}/email`, {
    method: 'POST',
    credentials: 'include',
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
    },
    body: new URLSearchParams(formData).toString(),
  }).then(resp => {
    if (!resp || !resp.ok) {
      throw new Error('Error sending Email', resp);
    }
  });
}
