/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.container.web;

import org.exoplatform.commons.utils.PropertyManager;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.container.RootContainer;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

import java.io.IOException;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;

/**
 * This filter will allow to prevent to any accesses to a web application corresponding to a
 * {@link PortalContainer} that has been disabled.
 * 
 * Created by The eXo Platform SAS
 * Author : Nicolas Filotto 
 *          nicolas.filotto@exoplatform.com
 * 8 juil. 2010  
 */
public class PortalContainerFilter extends AbstractFilter
{
   /**
    * The logger
    */
   private static final Log LOG = ExoLogger.getLogger("exo.kernel.container.PortalContainerFilter");

   /**
    * @see jakarta.servlet.Filter#destroy()
    */
   public void destroy()
   {
   }

   /**
    * @see jakarta.servlet.Filter#doFilter(jakarta.servlet.ServletRequest, jakarta.servlet.ServletResponse, jakarta.servlet.FilterChain)
    */
   public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException,
      ServletException
   {
      if (PortalContainer.isPortalContainerNameDisabled(servletContextName))
      {
         // The current portal container has been disabled
         onPortalContainerDisabled(request, response, chain);
         return;
      }
      else if (PropertyManager.isDevelopping())
      {
         HttpSession sess = ((HttpServletRequest)request).getSession(false);
         if (sess != null && sess.getAttribute(RootContainer.SESSION_TO_BE_INVALIDATED_ATTRIBUTE_NAME) != null)
         {
            sess.invalidate();
         }
      }
      chain.doFilter(request, response);
   }

   /**
    * Allow the sub classed to execute a task when a user try to access to a 
    * web application corresponding to a {@link PortalContainer} that has been disabled.
    * @param request the {@link ServletRequest}
    * @param response the {@link ServletResponse}
    * @param response the {@link FilterChain}
    */
   protected void onPortalContainerDisabled(ServletRequest request, ServletResponse response, FilterChain chain)
      throws IOException, ServletException
   {
      if (PropertyManager.isDevelopping())
      {
         LOG.info("The portal container corresponding to the webapp '" + servletContextName
            + "' is disabled, thus the request is cancelled: target URI was "
            + ((HttpServletRequest)request).getRequestURI());
      }
      ((HttpServletResponse)response).sendError(HttpServletResponse.SC_NOT_FOUND);
   }
}
