/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.download;

import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.PortalContainerInfo;
import org.exoplatform.services.cache.ExoCache;
import org.exoplatform.services.cache.concurrent.ConcurrentFIFOExoCache;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by The eXo Platform SARL
 * Author : Tuan Nguyen
 *          tuan08@users.sourceforge.net
 * Dec 26, 2005
 */
public class DownloadService {

    private final ExoCache<String, DownloadResource> downloadResources_;

    private Map<String, DownloadResource> defaultResources_;

    private PortalContainerInfo pinfo_;

    public static final String DOWNLOAD_HANDLER_PATH = "download";

    public DownloadService(PortalContainerInfo pinfo, InitParams params) throws Exception {
        int maxSize = Integer.parseInt(params.getValueParam("download.resource.cache.size").getValue());
        downloadResources_ = new ConcurrentFIFOExoCache<String, DownloadResource>(maxSize);

        defaultResources_ = new HashMap<String, DownloadResource>();
        pinfo_ = pinfo;
    }

    public void addDefaultDownloadResource(DownloadResource resource) {
        defaultResources_.put(resource.getDownloadType(), resource);
    }

    public String addDownloadResource(DownloadResource resource) {
        String id = Integer.toString(resource.hashCode());
        if (resource.getDownloadType() != null) {
            id = resource.getDownloadType() + ":/" + id;
        }
        downloadResources_.put(id, resource);
        return id;
    }

    public DownloadResource getDownloadResource(String id) {
        return getDownloadResource(id, true);
    }

    public DownloadResource getDownloadResource(String id, boolean remove) {
        DownloadResource resource = downloadResources_.get(id);
        if (resource != null) {
            if(remove) {
                downloadResources_.remove(id);
            }
            return resource;
        }
        String[] temp = id.split(":");
        if (temp.length > 1) {
            String downloadType = temp[0];
            resource = defaultResources_.get(downloadType);
        }
        return resource;
    }

    public String getDownloadLink(String id) {
        return "/" + pinfo_.getContainerName() + "/" + DOWNLOAD_HANDLER_PATH + "?" + "resourceId=" + id;
    }

    public ExoCache<String, DownloadResource> getCache() {
        return downloadResources_;
    }
}