/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.social.core.processor;

import java.util.HashMap;
import java.util.Map;

import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.ObjectParameter;

import junit.framework.TestCase;

/**
 * Unit Test for {@link ActivityResourceBundlePlugin}.
 *
 */
public class ActivityResourceBundlePluginTest extends TestCase {

  private static final String ACTIVITY_TYPE = "activity:fake";
  private static final String RESOURCE_BUNDLE_KEY_FILE = "FakeResourceBundle";

  private Map<String, String> activityKeyTypeMapping;

  @Override
  public void setUp() throws Exception {
    super.setUp();
    activityKeyTypeMapping = new HashMap<String, String>();
    activityKeyTypeMapping.put("hello", "FakeResourceBundle.hello");
  }

  public void testGetActivityKeyType() throws Exception {
    ObjectParameter objectParameter = getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping);
    InitParams initParams = new InitParams();
    initParams.addParam(objectParameter);
    ActivityResourceBundlePlugin activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(ACTIVITY_TYPE);
    assertEquals(ACTIVITY_TYPE, activityResourceBundlePlugin.getActivityType());
  }

  public void testGetActivityKeyTypeMapping() throws Exception {
    ObjectParameter objectParameter = getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping);
    InitParams initParams = new InitParams();
    initParams.addParam(objectParameter);
    ActivityResourceBundlePlugin activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(ACTIVITY_TYPE);
    assertEquals(activityKeyTypeMapping, activityResourceBundlePlugin.getActivityKeyTypeMapping());
  }

  public void testGetResourceBundleKeyFile() throws Exception {
    ObjectParameter objectParameter = getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping);
    InitParams initParams = new InitParams();
    initParams.addParam(objectParameter);
    ActivityResourceBundlePlugin activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(ACTIVITY_TYPE);
    assertEquals(RESOURCE_BUNDLE_KEY_FILE, activityResourceBundlePlugin.getResourceBundleKeyFile());
  }

  public void testHasMessageBundleKey() throws Exception {
    ObjectParameter objectParameter = getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping);
    InitParams initParams = new InitParams();
    initParams.addParam(objectParameter);
    ActivityResourceBundlePlugin activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(ACTIVITY_TYPE);
    assertFalse(activityResourceBundlePlugin.hasMessageBundleKey("hell"));
    assertTrue(activityResourceBundlePlugin.hasMessageBundleKey("hello"));
  }

  public void testGetMessageBundleKey() throws Exception {
    ObjectParameter objectParameter = getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping);
    InitParams initParams = new InitParams();
    initParams.addParam(objectParameter);
    ActivityResourceBundlePlugin activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(ACTIVITY_TYPE);
    assertNull(activityResourceBundlePlugin.getMessageBundleKey("hell"));
    assertEquals("FakeResourceBundle.hello", activityResourceBundlePlugin.getMessageBundleKey("hello"));
  }

  public void testIsValid() throws Exception {
    ActivityResourceBundlePlugin activityResourceBundlePlugin = new ActivityResourceBundlePlugin(null);
    assertFalse(activityResourceBundlePlugin.isValid());

    activityResourceBundlePlugin = new ActivityResourceBundlePlugin(new InitParams());
    assertFalse(activityResourceBundlePlugin.isValid());

    InitParams initParams = new InitParams();
    initParams.addParam(getObjectParameter(null, null));
    activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(null);
    assertFalse(activityResourceBundlePlugin.isValid());

    initParams = new InitParams();
    initParams.addParam(getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, null));
    activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(null);
    assertFalse(activityResourceBundlePlugin.isValid());

    initParams = new InitParams();
    initParams.addParam(getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping));
    activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(null);
    assertFalse(activityResourceBundlePlugin.isValid());

    initParams = new InitParams();
    initParams.addParam(getObjectParameter(RESOURCE_BUNDLE_KEY_FILE, activityKeyTypeMapping));
    activityResourceBundlePlugin = new ActivityResourceBundlePlugin(initParams);
    activityResourceBundlePlugin.setActivityType(ACTIVITY_TYPE);
    assertTrue(activityResourceBundlePlugin.isValid());
  }

  private ObjectParameter getObjectParameter(String resourceBundleKeyFile, Map<String, String> activityKeyTypeMapping) {
    ObjectParameter objectParameter = new ObjectParameter();
    objectParameter.setName(resourceBundleKeyFile);
    ActivityResourceBundlePlugin pluginConfig = new ActivityResourceBundlePlugin();
    pluginConfig.setActivityKeyTypeMapping(activityKeyTypeMapping);
    objectParameter.setObject(pluginConfig);
    return objectParameter;
  }
}
