/*
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2024 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */
package io.meeds.evm.gamification.plugin;

import io.meeds.gamification.model.RealizationDTO;
import io.meeds.gamification.service.EventService;
import io.meeds.evm.gamification.utils.Utils;
import io.meeds.gamification.plugin.EventPlugin;
import jakarta.annotation.PostConstruct;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.services.listener.ListenerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigInteger;
import java.util.List;
import java.util.Map;

@Component
public class EvmEventPlugin extends EventPlugin {

  public static final String EVENT_TYPE = "evm";

  @Autowired
  private EventService       eventService;

  @Autowired
  ListenerService            listenerService;

  @PostConstruct
  public void init() {
    eventService.addPlugin(this);
  }

  @Override
  public String getEventType() {
    return EVENT_TYPE;
  }

  @Override
  public List<String> getTriggers() {
    return List.of(Utils.SEND_TOKEN_EVENT, Utils.RECEIVE_TOKEN_EVENT, Utils.HOLD_TOKEN_EVENT);
  }

  @Override
  public boolean isValidEvent(Map<String, String> eventProperties, String triggerDetails) {
    String desiredContractAddress = eventProperties.get(Utils.CONTRACT_ADDRESS).toLowerCase();
    String desiredTargetAddress = eventProperties.get(Utils.TARGET_ADDRESS);
    String minAmount = eventProperties.get(Utils.MIN_AMOUNT);
    String desiredNetwork = eventProperties.get(Utils.BLOCKCHAIN_NETWORK);
    String tokenDecimals = eventProperties.get(Utils.DECIMALS);
    Map<String, String> triggerDetailsMop = Utils.stringToMap(triggerDetails);
    Long sentDate = Long.parseLong(triggerDetailsMop.get(Utils.SENT_DATE));
    if ( !desiredNetwork.equals(triggerDetailsMop.get(Utils.BLOCKCHAIN_NETWORK))
        || !desiredContractAddress.equals(triggerDetailsMop.get(Utils.CONTRACT_ADDRESS).toLowerCase())) {
      return false;
    }
    boolean isValidFilters = true;
    if (StringUtils.isNotBlank(desiredTargetAddress)) {
      isValidFilters = isValidFilters && isValidTargetAddress(desiredTargetAddress, triggerDetailsMop.get(Utils.TARGET_ADDRESS));
    }
    if (StringUtils.isNotBlank(eventProperties.get(Utils.DURATION))) {
      isValidFilters = isValidFilters && isValidAmountAndDuration(new BigInteger(triggerDetailsMop.get((Utils.TOKEN_BALANCE))),
                                                                  minAmount,
                                                                  tokenDecimals,
                                                                  sentDate,
                                                                  Long.parseLong(eventProperties.get(Utils.DURATION)),
                                                                  Long.parseLong(triggerDetailsMop.get(Utils.DURATION)));
    } else {
      if (StringUtils.isNotBlank(minAmount) && StringUtils.isNotBlank(tokenDecimals)) {
        isValidFilters = isValidFilters && isValidMinAmount(minAmount,
                                          new BigInteger(triggerDetailsMop.get(Utils.MIN_AMOUNT)),
                                          tokenDecimals);
      }
    }
    return isValidFilters;
  }

  @Override
  public String getLink(RealizationDTO realization) {
    if (StringUtils.isNotBlank(realization.getObjectId())) {
      String[] transactionDetails = realization.getObjectId().split("#");
      int networkId = Integer.parseInt(transactionDetails[0]);
      String transactionHash = transactionDetails.length > 1 ? transactionDetails[1] : "";
      if (transactionHash.isEmpty()) {
        return "";
      }
      return switch (networkId) {
        case 1 -> "https://etherscan.io/tx/" + transactionHash;
        case 137 -> "https://polygonscan.com/tx/" + transactionHash;
        case 80002 -> "https://amoy.polygonscan.com/tx/" + transactionHash;
        case 11155111 -> "https://sepolia.etherscan.io/tx/" + transactionHash;
        default -> "";
      };
    }
    return "";
  }


  private boolean isValidMinAmount(String minAmount, BigInteger amountTransferred, String tokenDecimals) {
    BigInteger base = new BigInteger("10");
    BigInteger desiredMinAmount = new BigInteger(minAmount);
    if (StringUtils.isNotBlank(tokenDecimals)) {
      Integer decimals = Integer.parseInt(tokenDecimals);
      desiredMinAmount = base.pow(decimals).multiply(new BigInteger(minAmount));
    }
    return amountTransferred.compareTo(desiredMinAmount) >= 0;
  }

  private boolean isValidTargetAddress(String desiredTargetAddress, String targetAddress) {
    return desiredTargetAddress.toLowerCase().equals(targetAddress.toLowerCase());
  }

  private boolean isValidAmountAndDuration(BigInteger tokenBalance,
                                           String minAmount,
                                           String tokenDecimals,
                                           Long sentDate,
                                           Long duration,
                                           Long desiredDuration) {
    if (duration.compareTo(desiredDuration) != 0) {
      return false;
    }
    Long holdingDuration = System.currentTimeMillis() - sentDate;
    BigInteger base = new BigInteger("10");
    BigInteger desiredMinAmount = new BigInteger(minAmount);
    if (StringUtils.isNotBlank(tokenDecimals)) {
      Integer decimals = Integer.parseInt(tokenDecimals);
      desiredMinAmount = base.pow(decimals).multiply(new BigInteger(minAmount));
    }
    return tokenBalance.compareTo(desiredMinAmount) >= 0 && holdingDuration.compareTo(duration) >= 0;
  }
}
